/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: SplitMessage.java,v 1.26 1997/05/29 07:07:30 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * SplitMessages allow for the changing of the actual packets grabbed
 * from the internet socket to a structure of various fields, and vice-
 * versa.  It is used by the MailDaemon to pack all the header information
 * with message information, and prepare it for shipping as well as taking
 * a byte array and discerning what the header information is.<P>
 * This is where the message's id is stored, along with the destination
 * mailbox name, the sending mailbox name, the MailDaemon id of the sending
 * side, whether the message is an acknowledgement of another message,
 * and, of course, the message data itself left in a byte array.
 * <pre>
 * The structure of the actual SplitMessage when packed in a byte stream
 * is:
 *     [mailDaemon ID](2 bytes) / message ID (4 bytes) / mailbox name 
 *     
 *     / sending mailbox name / t/f for isAck / message body
 * </pre>
 * <P>
 * There are two collections of fields in this object.  The first group
 * has all the message components, the second is just a byte array.  The
 * message data is in either one or the other state, and the accessors
 * check to see which state is current, and if necessary will shift the
 * data to the other state before returning any value.  This allows for
 * a completely transparent conversion from one side of the fence, as they
 * say, to the other.  <STRONG><EM>Wubba Wubba Wubba.</EM></STRONG>
 * 
 * @author Luke Weisman
 * @author Wesley Tanaka
 * @author Adam Rifkin
 * @version 1.0b2 $Revision: 1.26 $ $Date: 1997/05/29 07:07:30 $
 */

class SplitMessage 
{

  /**
   * The name of the mailbox for which the message is destined.
   */
  private String  _name;
  /**
   * The name of the mailbox which sent the message.
   */
  private String  _fname;
  /**
   * Whether the message is an acknowledgement.
   */
  private boolean _isAck;
  /**
   * The ID of the message.
   */
  private int     _ID;
  /**
   * The ID of the message target's mailDaemon.
   */
  private long    _dID;
  /**
   * The ID of the message owner's mailDaemon.
   */
  private long _mydID;

  /**
   * The length of the actual message in bytes.
   */
  private int     _msgLength;
  /**
   * The message data itself.
   */
  private DataInputStream _msg;

  /**
   * The entire message as a byte stream. 
   */
  byte [] _bstr;


  // constructors

  /**
   * This constructor will take the byte array, and
   * build a SplitMessage from it.  It will automatically convert
   * the data to the field form.
   */

  SplitMessage (byte [] b)
    throws IOException
  {
    //make stream out of byte array.
    DataInputStream dis = new DataInputStream(new ByteArrayInputStream(b));

    //read off header info:
    _dID = dis.readLong();
    _mydID = dis.readLong();
    _name = dis.readLine();
    _fname = dis.readLine();
    _isAck = dis.readBoolean();
    _ID = dis.readInt();

    //save the data stream for the message
    _msg = dis;
  }

  
  /**
   * This builds a message out of the various components so it
   * can be shipped correctly.
   * <pre>
   * The structure of the SplitMessage when packed in a byte stream is:
   *     [mailDaemon ID](2 bytes) / message ID (4 bytes) / mailbox name 
   *     / sending mailbox name / t/f for isAck / message body
   * </pre>
   *
   * @param dID     Id of MaiLDaemon on receiving end
   * @param mydID   Id of MailDaemon on sending end.
   * @param PName   The name of the mailbox the message is targeted for.
   * @param FName   The name of the sending mailbox.
   * @param isAck   true if the message is an acknowledgement, false 
   *                otherwise.
   * @param ID      The ID of the message.
   * @param msg     A byte array which is a message data block.
   */

  SplitMessage( long dID,
                long mydID,
                String PName, 
                String FName,
                boolean isAck,
                int ID,
                Message msg )
  {
    //init vars.
    _dID = dID;
    _mydID = mydID;
    _name = PName;
    _fname = FName;
    _isAck = isAck;
    _ID = ID;
    _msg = null;

    try
      {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream( bos );

        //write header info
        dos.writeLong( dID );
        dos.writeLong( mydID );
        dos.writeBytes( PName + "\n" );
        dos.writeBytes( FName + "\n" );
        dos.writeBoolean( isAck );
        dos.writeInt( ID );

        //write message
        msg.writeObject( dos );

        //done
        dos.flush();
        bos.flush();

        //convert to byte array
        _bstr = bos.toByteArray();
      }
    catch (IOException ioException)
      {
        Debug.bugExit ("SplitMessage constructor",
                       "IOException in writing");
        ioException.printStackTrace();
        throw new RuntimeException ("IOException");
      }
  }

  
  // methods 

  DataInputStream Message( )
  {
    if ( _msg == null )
      {
        try
          {
            SplitMessage tmp = new SplitMessage( _bstr );
            return tmp.Message();
          }
        catch ( IOException e )
          {
            return null;
          }
      }
    else
      return _msg;
  }


  String PName( )
  {
    return _name;
  }


  String FName( )
  {
    return _fname;
  }


  boolean isAck( )
  {
    return _isAck;
  }


  int ID( )
  {
    return _ID;
  }


  /**
   * Id of target's maildaemon.
   */

  long dID( )
  {
    return _dID;
  }


  /**
   * ID of message owner's maildaemon.
   */

  long mydID( )
  {
    return _mydID;
  }


  byte [] toByteStream( )
  {
    if ( _bstr == null )
      {
        throw new NullPointerException( "_bstr null in SplitMessage." );
      }
    else
      return _bstr;
    
  }


  /**
   * Convert an integer to 4 bytes.  Store at offset in the b array.
   * @param i The integer to convert.
   * @param b The byte array to store the thing in.
   * @param offset The offset to begin store sequence on.
   */

  static void toBytes (int i, 
                       byte [] b, 
                       int offset)
  {
    b[3+offset] = (byte)(i % 256);
    i = i >>> 8;
    b[2+offset] = (byte)(i % 256);
    i = i >>> 8;
    b[1+offset] = (byte)(i % 256);
    i = i >>> 8;
    b[0+offset] = (byte)(i);
  }


  /**
   * Convert 4 bytes to an integer.
   * @param b      The byte array to pull the integer from.
   * @param offset The offset to start on in the byte array.
   */

  static int fromBytes (byte [] b, 
                        int offset)
  {
    int i = (256 + (int)b[offset]) % 256;
    i = (i << 8) + ( ((int)b[offset + 1] + 256) % 256 );
    i = (i << 8) + ( ((int)b[offset + 2] + 256) % 256 );
    i = (i << 8) + ( ((int)b[offset + 3] + 256) % 256 );
    
    return i;
  }


  public String toString( )
  {
    return "[dID=" + Long.toString( _dID, Character.MAX_RADIX )
      + ", mydID=" + Long.toString( _mydID, Character.MAX_RADIX )
      + ", name=" + _name + ", fname=" + _fname
      + ", isAck=" + _isAck + ", ID=" + _ID + "]";
  }

} // end of SplitMessage class

