/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: SyncOutbox.java,v 1.16 1997/05/02 22:15:57 adam Exp $
 */

package info.net;

// Needs cleaning up.  -- Adam

/**
 * This is one half of a synchronized connection.  When one sends a
 * message via a SyncOutbox, it blocks until the receiver has
 * taken the message out of its queue via the receive() method of the
 * SyncInbox.  The SyncOutbox will wait forever if the
 * receiver does not send acknowledgements, so it is very important
 * that the receiver be a SyncInbox.<P>
 * It is possible to only block for some specified time.  In this case
 * the send() method returns whether the time expired or if the message
 * was acknowledged.<P>
 * testreceive6 and testsend6 illustrate the use of the Sync mailboxs.
 *
 * @author Adam Rifkin
 * @author Luke Weisman
 * @version 1.0b2 $Date: 1997/05/02 22:15:57 $
 * @see Inbox
 * @see Place
 * @see Outbox
 * @see MailDaemon
 * @see MailDaemonException
 * @see Message
 */

public class SyncOutbox extends Outbox 
{

  /**
   * This is the currently bound place which all messages go to.
   */
  Place boundTo;

  /**
   * This is the result of a send.  True means acknowledged.  It is
   * a field so different threads can gain access to it.
   */
  boolean waitResult;

  /**
   * counter to id all messages so the acks will always corrispond.
   */
  int id_cntr = 0;

  /**
   * Make a mailbox called name, connected to p, which will block
   * on a send until the timeout expires or the message is acknowledged
   * @param md The maildaemon to use.
   * @param name The name of the send mailbox.
   * @param p  The place to bind to.
   * @exception MailDaemonException if the name is already used by the maildaemon.
   */
  public SyncOutbox( MailDaemon md, String name, Place p )
    throws MailDaemonException
  {
    super( md, name );
    boundTo = p;
  }


  /**
   * Same as SyncOutbox( md, name, p ) except there is no
   * specified name for the mailbox.
   * @param md The maildaemon to use.
   * @param p  The place to bind to.
   */
  public SyncOutbox( MailDaemon md, Place p )
  {
    super( md );
    boundTo = p;
  }



  /**
   * Waits for an ack to arrive, or for its time to expire.  0 means
   * wait forever.
   * @exception InterruptedException if the waiting thread gets an
   *            interrupt call.
   * @exception TimeoutException if the waiting thread
   *            times out before acknowledgment is received.
   */
  synchronized void waitForAck( long time )
    throws InterruptedException, TimeoutException
  {
    waitResult = false;
    
    //wait for the ack
    wait( time );
    
    if ( !waitResult )
      {
        //get rid of the message since it timed out.
        mailDaemon.purgeAddress( boundTo, this );

        throw new TimeoutException( "Duration of " + time + " milliseconds "
                                    + "has passed without acknowledgement." );
      }
  }


  /**
   * Just like send( msg, time ) except it will block forever if no
   * ack is fourthcoming.
   * @param msg The message to ship off with much glory.
   * @exception InterruptedException if the waiting thread gets an
   *            interrupt call.
   */
  public void send( Message msg )
    throws InterruptedException
  {
    try {
      send( msg, 0 );
    }
    catch ( TimeoutException e )
      {
        throw new RuntimeException( "Non-timing out thread timed out: "
+ e );
      }
  }


  /**
   * Sends the passed message to the currently bound target, and
   * then waits a maximum of time seconds for an acknowledgement.
   * @param     msg The message to send.
   * @param     time The min time to wait for an acknowledgement.  After
   *            this time has passed, the message will timeout when the
   *            maildaemon next looks through its delinquent message queue.
   * @return    true if the message was sent successfully.<br> false if the
   *            timeout period passed before an acknowledgement was received.
   * @exception InterruptedException if the waiting thread gets an
   *            interrupt call.
   * @exception TimeoutException if the time expires without acknowledgment.
   */
  public void send( Message msg, long time )
    throws InterruptedException, TimeoutException
  {
    //set timeout for the message.
    if ( time == 0 )
      setMinTimeout( NO_TIMEOUT );
    else
      setMinTimeout( time );

    //send out the message.
    doSend( new SyncMessage( msg, id_cntr++ ), 
            boundTo );

    waitForAck( time );
  }


  /**
   * If the message times out, then bail.
   */
  synchronized protected boolean timeOut( Message msg, Place to )
  {
    mailDaemon.purgeAddress( to, this );
    waitResult = false;
    notify();

    //we don't want to send the message again.
    return false;
  }


  /**
   * Got a denyMessage.  If it is an ACK, wake up the waiting thread.
   * otherwise process deny message as normal via the super() method.
   */
  synchronized protected boolean insertMessage(int time, Message m, Place from )
    throws MailDaemonException
  {
    if ( m instanceof SyncMessage )
      {
        if ( id_cntr - ((SyncMessage)m).getID() == 1 )
          {
            //note acknowledgement.
            waitResult = true;

            //got ack, wake up.
            notify();
          }

        return true;
      }
    else 
      return super.insertMessage( time, m, from );
  }

} // end of SyncOutbox class
