/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 *  $Id: URLStringSecurity.java,v 1.13 1997/05/29 07:07:35 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.net.*;
import java.io.*;
import java.util.*;
import info.util.Set;

/**
 * This is a URLSecurity that is very flexible, and allows for
 * selection of trusted sites.
 *
 * <P> URLStringSecurity keeps track of four sets: rejectSet, acceptSet,
 * portAcceptSet, and protocolAcceptSet.  If any rejectSet member
 * appears within in the URL host string, then the classload is
 * rejected.  Otherwise, at least one of the acceptSet members needs
 * to appear for the URL string for classloading to occur.
 *
 * <P> In addition, if the portCheckingFlag is on, then the URL's port
 * must be in the portAcceptSet to be accepted.  Also, if the
 * protocolCheckingFlag is on, then the protocol must be in the
 * protocolAcceptSet to be accepted.
 *
 * <P> If the string to compare is to be matched at the beginning or the
 * end of the URL, the toBegin and toEnd methods should be used (see
 * docs).  Port selection can also take place with the addPort method.
 *
 * @author Adam Rifkin
 * @author Luke Weisman
 * @version 1.0b2 $Date: 1997/05/29 07:07:35 $
 * @see URLSecurity
 **/

public class URLStringSecurity implements URLSecurity
{
  /**
   * If a string is in the rejectSet set, then the URL classload is
   * rejected, provided there are no other problems with port number
   * or file type.
   **/
  Set rejectSet;

  /** If a string is in the acceptSet set, then the URL classload is
   * accepted, provided there are no other problems with port number
   * or file type.  (The default is to reject the classload, so to be
   * accepted, there must be at least one acceptSet match.)
   **/
  Set acceptSet;

  /**
   * If portCheckingFlag is true, then port checking is enabled.
   **/
  boolean portCheckingFlag;

  /**
   * If the URL's specified port is not in the portAcceptSet set, and
   * if portCheckingFlag is true, then the URL classload is rejected.
   **/
  Set portAcceptSet;

  /**
   * If protocolCheckingFlag is true, then protocol type checking is
   * enabled.
   **/
  boolean protocolCheckingFlag;

  /**
   * If the URL's specified protocol (for example, http or file) is
   * not in the protocolAcceptSet set, and if the protocolCheckingFlag
   * variable is true, then the URL classload is rejected.
   **/
  Set protocolAcceptSet;


  /**
   * Contruct a URLStringSecurity; the port checking and
   * the protocol checking can only be toggled here.
   * @param portFlag flag for turning on port checking.
   * @param protocolFlag flag for turning on protocol checking.
   * @param defaultsFlag flag for default setting.
   **/
  URLStringSecurity
  ( boolean portFlag, boolean protocolFlag, boolean defaultsFlag )
  {
    rejectSet = new Set();
    acceptSet = new Set();
    portAcceptSet = new Set();
    protocolAcceptSet = new Set();

    portCheckingFlag = portFlag;
    protocolCheckingFlag = protocolFlag;

    if ( defaultsFlag )
      {
        rejectSet.add( "adam rifkin" ); // oh yeah.
        acceptSet.add( toEnd( toBegin( "www.info.caltech.edu" ) ) );
        acceptSet.add( toEnd( ".com" ) );
        protocolAcceptSet.add( "http" );
        protocolAcceptSet.add( "shttp" );
        protocolAcceptSet.add( "file" );
        portAcceptSet.add( new Integer( -1 ) );
        portAcceptSet.add( new Integer( 80 ) );
      }

  }

  /**
   * decoder finds a string's URI.
   * If there is an unrecognized code (%<garbage>) then the
   * unparsed character will be replaced with the 'x' character.
   * @param stringToCheck the string to check for a URI.
   * @return the URI within the string.
   **/
  final protected String decoder( String stringToCheck )
  {
    StringBuffer buffer = new StringBuffer( stringToCheck.length() );
    int i;
    char c;
    String st;

    for ( i = 0; i < stringToCheck.length(); i++ )
      {
        c = (char)stringToCheck.charAt( i );
        if ( c == '+' )
          c = ' ';
        else if ( c == '%' )
          {
            try
              {
                st = stringToCheck.substring( i + 1, i + 3 );
                c = (char)Integer.parseInt( st, 16 );
              }
            catch ( NumberFormatException e )
              {
                c = 'x';
              }
            catch ( StringIndexOutOfBoundsException e )
              {
                c = 'x';
              }
            i += 2;
          }
        buffer.append( c );

      }
    return new String( buffer );
  }


  /**
   * checkURL checks the various sets to determine if a URL is
   * acceptable for classloading.
   * @param urlString the URL string to check.
   * @return true if the URL classloading is accepted, false otherwise.
   */
  public boolean checkURL( String urlString )
  {
    URL url = null;

    try
      {
        url = new URL( urlString );
      }
    catch ( MalformedURLException e )
      {
        Debug.lpln( 11, "Rejected: URL is malformed." );
        return false;
      }

    String host = url.getHost();
    int port = url.getPort();
    String protocol = url.getProtocol();

    Debug.lpln( 11, "Host = " + host );
    Debug.lpln( 11, "Port = " + port );
    Debug.lpln( 11, "Protocol = " + protocol );
    Debug.lpln( 11, "File = " + url.getFile() );
    Debug.lpln( 11, "ref  = " + url.getRef() );

    host = decoder( host );
    Debug.lpln( 11, "new host = " + host );

    boolean acceptFlag = true;

    // Check original for attempts to get around first/last tags.

    if ( host.indexOf( "%-00" ) != -1 )
      acceptFlag = false;

    host = "%-00" + host + "%-00";

    Enumeration e;

    // Check the rejectSet strings.

    e = rejectSet.set();
    while( acceptFlag && e.hasMoreElements() )
      acceptFlag = ( host.indexOf( (String)e.nextElement()) == -1 );

    // Check the acceptSet strings.

    if ( acceptFlag )
      {
        e = acceptSet.set();
        acceptFlag = false;
        while( !acceptFlag && e.hasMoreElements() )
          acceptFlag = ( host.indexOf( (String)e.nextElement()) != -1 );
      }

    // Check the portAcceptSet.

    if ( acceptFlag && portCheckingFlag )
      {
        acceptFlag = false;
        e = portAcceptSet.set();
        while( !acceptFlag && e.hasMoreElements() )
          acceptFlag = ( port == ( (Integer)e.nextElement() ).intValue() );
      }

    // Check the protocolAcceptSet.

    if ( acceptFlag && protocolCheckingFlag )
      {
        acceptFlag = false;
        e = protocolAcceptSet.set();
        while( !acceptFlag && e.hasMoreElements() )
          acceptFlag = ( protocol.equals( (String)e.nextElement() ) );
      }

    return acceptFlag;
  } //end checkURL


  /**
   * Add a string to the rejectSet.
   * @param s the string to add to the rejectSet.
   **/
  public void addReject( String s )
  {
    rejectSet.add( s.intern() );
  }

  /**
   * Add a string to the acceptSet.
   * @param s the string to add to the acceptSet.
   **/
  public void addAccept( String s )
  {
    acceptSet.add( s.intern() );
  }

  /**
   * Change a String, so it can be checked against the beginning
   * of the URL only.  The result can be passed to the addAccept or
   * addReject methods.
   * @param s the string to change.
   */
  public String toBegin( String s )
  {
    return "%-00" + s;
  }

  /**
   * Change a String, so it can be checked against the end of the
   * URL only.  The result can be passed to the addAccept or addReject
   * methods.
   * @param s the string to change.
   **/
  public String toEnd( String s )
  {
    return s + "%-00";
  }

  /**
   * Add a port to the set of portAcceptSet.
   * @param p the port to add.
   **/
  public void addPort( int p )
  {
    portAcceptSet.add( new Integer( p ) );
  }

} // end of URLStringSecurity class

