/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: Debug.java,v 1.30 1997/05/29 07:07:39 kiniry Exp $
 */

package info.util;

import java.io.FileOutputStream;
import java.io.PrintStream;
import java.io.File;
import java.io.IOException;
import java.util.Vector;

/**
 * The info.util.Debug class is the centralized manner in which we
 * control debugging output in the II.
 *
 * @author Joseph R. Kiniry
 * @author Wesley Tanaka
 * @author Luke Weisman
 * @version 1.0b2 $Revision: 1.30 $ $Date: 1997/05/29 07:07:39 $
 **/

public class Debug
{

  /*
   * on - a flag indicating whether or not debugging output is on at all.
   * outputStream - the output stream to which Debug is writing; either
   *                System.err or
   * suppress - a Vector of the types of debug message we wish _not_
   *            to see.
   * toFile - a flag indicating if we are currently logging to a file
   *          rather than to System.err
   * debugOff - a flag indicating whether or not debugging output is off
   * stackOff - a flag indicating whether or not stack output is off
   * level - a string indicating the current stack level
   */

  public static boolean on =                true;
  private static PrintStream outputStream = System.err;
  private static Vector      suppress =     new Vector ();
  private static boolean     toFile =       false;
  private static boolean     debugOff =     true;
  private static boolean     stackOff =     true;
  private static String      level =        "";


  static
  {
    suppress.addElement ("Exception");   // exception error messages
    suppress.addElement ("General");     // general information
    suppress.addElement ("GraveDigger"); // grave status looking msg
    suppress.addElement ("Listen");      // listening for messages
    suppress.addElement ("Master");      // djinn master-specific info
    suppress.addElement ("Memory");      // memory-related issues
    suppress.addElement ("Message");     // interesting messages being sent
    suppress.addElement ("Names");       // print DjinnNames
    suppress.addElement ("Persistence"); // djinn persistence info
    suppress.addElement ("Port");        // info about ports/mailboxes
    suppress.addElement ("Properties");  // djinn master properties data
    suppress.addElement ("Stack");       // function stack trace
    suppress.addElement ("Table");       // show DjinnTable in master
    suppress = new Vector ();            // turn them all back on
  }


  /**
   * Clears the debugging output suppression list.
   **/

  public static void clearSuppress ()
  {
    suppress = new Vector ();
  }

  /**
   * Adds the given String to the debugging output suppression list.
   * @param suppression_string The String.
   **/

  public static void suppress (String suppresion_string)
  {
    suppress.addElement (suppresion_string);
  }


  /**
   * Prints a debug message followed by a newline.
   * @param dlevel The debugging level of the message.
   * @param object The object which generated the message.
   * @param method The method which generated the message.
   * @param message The message.
   **/

  public static void println (String dlevel,
                              String object,
                              String method,
                              String message)
  {
    println (dlevel, object + "." + method + ": " + message);
  }


  /**
   * Prints a debug message (with no extra text component) followed
   * by a newline.
   * @param dlevel The debugging level of the message.
   * @param object The object which generated the message.
   * @param method The method which generated the message.
   **/

  public static void println (String dlevel,
                              String object,
                              String method)
  {
    println (dlevel, object + "." + method);
  }


  /**
   * Prints a debug message.
   * @param dlevel The debugging level of the message.
   * @param object The object which generated the message.
   * @param method The method which generated the message.
   * @param message The message.
   **/

  public static void print (String dlevel,
                            String object,
                            String method,
                            String message)
  {
    print (dlevel, object + "." + method + ": " + message);
  }


  /**
   * Prints a debug message (with no extra text component).
   * @param dlevel The debugging level of the message.
   * @param object The object which generated the message.
   * @param method The method which generated the message.
   **/

  public static void print (String dlevel,
                            String object,
                            String method)
  {
    print (dlevel, object + "." + method);
  }


  /**
   * Used to check whether a flag has been set to suppress a given
   * message type.  This can be used by methods outside of the Debug
   * class to short-circuit debug messaging.
   *
   * @param dlevel the message type/level to check.
   * @return a flag indicating whether or not this particular
   * message type is suppressed.
   **/

  public static boolean checkSuppress (String dlevel)
  {
    for (int i = 0; i < suppress.size (); i ++)
      if (dlevel.equals ((String) suppress.elementAt (i)))
        return true;
    return false;
  }


  /**
   * Prints a debug message followed by a newline.
   * @param dlevel The debugging level of the message.
   * @param message The message.
   **/

  public static void println (String dlevel,
                              String message)
  {
    for (int i = 0; i < suppress.size (); i ++)
      if (dlevel.equals ((String) suppress.elementAt (i)))
        return;
    outputStream.println (dlevel + ": " + message);
  }


  /**
   * Prints a debug message followed by a newline.
   * @param message The message.
   **/

  public static void println (String message)
  {
    outputStream.println (message);
  }


  /**
   * Prints a debug message.
   * @param dlevel The debugging level of the message.
   * @param message The message.
   **/

  public static void print (String dlevel,
                            String message)
  {
    for (int i = 0; i < suppress.size (); i ++)
      if (dlevel.equals ((String) suppress.elementAt (i)))
        return;
    outputStream.print (message);
  }


  /**
   * Prints a debug message.
   * @param message The message.
   **/

  public static void print (String message)
  {
    outputStream.print (message);
  }


  /**
   * Sets the debug.on flag according to the passed parameter.
   * @param flag true to set debug.on to true, false to set
   * debug.on to false.
   **/

  public static void setOn (boolean flag)
  {
    on = flag;
  }


  /**
   * Controls the redirection of debugging messages to files.
   * @param flag true to redirect debugging messages to a file,
   * false to send debugging messages to standard error.
   * @param filename The file to redirect debugging messages to,
   * or null for a default filename.
   **/

  public static void toFile (boolean flag,
                             String filename)
  {
    toFile = flag;

    if (filename == null)
      filename = new String ("IIDebug.log");

    if (toFile)
      {
        try
          {
            outputStream = new PrintStream
              (new FileOutputStream
               (new File (filename)));
          }
        catch (IOException ioException)
          {
            System.err.println
              ("IOException raised when attempting to create logfile 'IIDebug': " +
               ioException.getMessage ());
          }
      }
    else
      outputStream = System.err;
  }


  /**
   * Gets the PrintStream being used for debug messages.
   * @return the PrintStream.
   **/

  public static PrintStream getStream ()
  {
    return outputStream;
  }


  /**
   * Tests to see whether or not debugging output is turned on.
   * @return true if debugging output is turned on, false otherwise.
   **/

  static public boolean on ()
  {
    return on;
  }


  // from info.net.Debug **************************************************

  /**
   * Allows for printing debug messages in the MailDaemon package.
   * There are 13 debugging levels.  To turn a given level on, simply
   * make the checkLevel method return true when the passed value
   * (debugType) is appropriate. <P>
   *
   * Warning: The various areas are not very exact at all.  In
   * addition, some of the debugging statements themselves have been
   * removed from the code to clean it up, which means some levels might
   * not print out anything.  So, be forwarned. <P>
   *
   * In addition it is possible to turn off the entire debugging
   * environment by toggling the debugOff static variable.  This can be
   * done in runtime as well as here. <P>
   *
   * <PRE>
   * level - function codes
   * --------------------------------
   * 0 - PrintInbox messages
   * 1 - MailDaemonOut message acknowledgement.
   * 2 - InMailQueue receiving messages.
   * 3 - Return address debugging and timeouts.
   * 4 - Sending stale message debugging.
   * 5 - Dealing with race problems.
   * 6 - Dealing with parsing and SplitMessage.
   * 7 - Suspending and renewing threads in the MailDaemon
   * 8 - Walking through message building
   * 9 - ClassLoader
   * 10- DjinnMaster stuff
   * 11- URLStringSecurity
   * 12- more ClassLoading and Message building
   * 13- MailDaemon ID stuff.
   * </PRE>
   *
   * @author Adam Rifkin
   * @author Luke Weisman
   * @version 1.0b2 $Date: 1997/05/29 07:07:39 $
   **/

  /**
   * This is the method which controls which debug levels are on or off.
   * The Debug class must be recompiled to see different debug levels.
   * @return true if the specified level should be printed, false otherwise.
   **/

  public static boolean checkLevel (int debugType)
  {
    if (debugOff)
      return false;

    else
      {
        switch ( debugType ) {

          // The following debug levels should print.


        case -1:
          return true;


          // The following debug levels should not print.

        case 0: //0 - PrintInbox messages
        case 1: //1 - MailDaemonOut message acknowledgement.
        case 2: //2 - InMailQueue receiving messages.
        case 3: //3 - Return address debugging and timeouts
        case 4: //4 - Sending stale message debugging.
        case 5: //5 - Dealing with race problems.
        case 6: //6 - Dealing with parsing and SplitMessage.
        case 7: //7 - Suspending and renewing threads in the MailDaemon
        case 8: //8 - Walking through message building
        case 9: //9 - ClassLoader
        case 10: //10 - DjinnMaster stuff
        case 11: //11- URLStringSecurity
        case 12: //12 - more ClassLoading and Message building
        case 13: //13 - MD Id stuff.

          // Default is not to print.

        default:

          return false;
        }
      }
  }


  /**
   * Prints the given string, the header, and a newline.
   **/

  public static void lpln (int debugType,
                           String debugString)
  {
    if (checkLevel(debugType))
      System.err.println ("[" + debugType + "] " + debugString);
  }


  /**
   * Prints the given string and a newline.
   **/

  public static void bugExit (String debugMethod,
                              String debugString)
  {
    System.err.println ("Problem with " + debugMethod + ": " +
                        debugString + " - please contact info-bugs@cs.caltech.edu");
  }


  /**
   * Pushes an element onto the stack.
   **/

  public static void push (String message)
  {
    if (!stackOff)
      {
        System.err.println (level + "+" + message);
        level += " ";
      }
  }


  /**
   * Pops an element off the stack.
   **/

  public static void pop (String message)
  {
    if (!stackOff)
      {
        level = level.substring (1);
        System.err.println (level + "-" + message);
      }
  }


  /**
   * Set the global debugging toggle.
   *
   * @param toWhat true to turn debugging on, false to turn
   * debugging off.
   **/

  public static void SetDebug (boolean toWhat)
  {
    debugOff = toWhat;
  }

} // end of Debug class
