/////////////////////////////////////////////////////////////////
//  DataDriverDevice.cpp - CDataDriverDevice implementation
//  Portions Copyright (c) 1996-97, BlueWater Systems,Inc.
/////////////////////////////////////////////////////////////////

    // main WinDK header file
#include "WinDK.h"
    // user interface and ioctl codes
#include "DD_CtlCode.h"
    // driver defines and class definition
#include "DataDriverDevice.h"

/////////////////////////////////////////////////////////////////////
//  ~CDataDriverDevice Destructor
//  Input:  none
//  Output: none
//  Notes:  The destructor must be defined before the constructor so
//          that compiler generated destructors will be placed in
//          the correct code segment.
/////////////////////////////////////////////////////////////////////
CDataDriverDevice::~CDataDriverDevice()
{
    if (m_DataBuffer != NULL)
    {
        ExFreePool(m_DataBuffer);
        m_DataBuffer = NULL;
    }
}

#ifdef WINDK_CODE_SEG_PRAGMA
#pragma code_seg("INIT")
#endif //WINDK_CODE_SEG_PRAGMA

/////////////////////////////////////////////////////////////////////
//  CDataDriverDevice - Constructor for the CDataDriverDevice class
//  Input:  pDriverObject   - NT driver object
//          pDeviceName     - name of this device
//          DeviceType      - device's type
//          pRegistryPath   - path to the registry entries for this driver
//          DeviceNumber    - device number
//          pContext        - not used
//  Output: ppDeviceObject  - NT device object for this instance of the device
//  Notes:  throws exception on error
//////////////////////////////////////////////////////////////////////
CDataDriverDevice::CDataDriverDevice(      PDRIVER_OBJECT    pDriverObject,
                                     CONST PWCHAR            pDeviceName,
                                     CONST ULONG             DeviceType,
                                           PUNICODE_STRING   pRegistryPath,
                                     CONST ULONG             DeviceNumber,
                                           PDEVICE_OBJECT    *ppDeviceObject,
                                           PVOID             pContext) :
    CDevice(pDriverObject,
            pDeviceName,
            DeviceType,
            ppDeviceObject),
    m_fIsSampling(FALSE),
	m_fConfigured(FALSE),
    m_fIsSquareWave(FALSE),
    m_MissedSamples(0),        
    m_DataBuffer(NULL),
	m_Frequency(0)            
{
    NTSTATUS    status;         
    
    SetFlags(DO_BUFFERED_IO);
        // create a Win32 accessable name for the driver 
        // in the form of DataDriverDevice0
    status = CreateWin32Name(pDeviceName);
    if (!NT_SUCCESS(status))
    {
        ThrowStatusExceptionFromConstructor(status, 
            "DATADRIVERDEVICE:CDataDriverDevice::CDataDriverDevice CreateWin32Name failed");
    }              
       // get the resources from the registry              
    CRegistry *pRegistry = new CRegistry(pRegistryPath);
    
    if (pRegistry == NULL)
    {
        ThrowStatusExceptionFromConstructor(STATUS_INSUFFICIENT_RESOURCES, 
            "DATADRIVERDEVICE:CDataDriverDevice::CDataDriverDevice Failed to allocate memory");                    
    }                
    if (pRegistry->IsKeyPresent(pDeviceName) == FALSE)
    {
        delete pRegistry;
        ThrowStatusExceptionFromConstructor(STATUS_INSUFFICIENT_RESOURCES,
            "DATADRIVERDEVICE: CDatadriverDevice::CDatadriverDevice - Missing key DataDriverDevice");
    }
    CString dataFileName;               
    
    pRegistry->SetRelativePath(Concatenate(pDeviceName, L"\\Parameters"));
    pRegistry->GetKey(L"DataFile",dataFileName);
    delete pRegistry;                         
    // If there is a key DataFile, then we try to open it and
    // use the data it contains.            
    CFileIo *pFile = NULL;                                    
        
    __try
    {    
        if (dataFileName.IsEmpty())    
        {
            status = STATUS_UNSUCCESSFUL;
            return;               
        }
        pFile = new CFileIo;
    
        if (pFile == NULL)
        {
            return;        
        }                                                                    
        status = pFile->CreateHandle(dataFileName);                                
        if (NT_ERROR(status))
        {
            return;
        }            
        ULONG size;
        
    	pFile->QueryFileSize(&size);
        m_MaxSamples = (size / sizeof(USHORT));
        if (m_MaxSamples == 0)
        {
            status = STATUS_UNSUCCESSFUL;
            return;
        }                      
        m_DataBuffer = static_cast<PUSHORT>(ExAllocatePool(NonPagedPool,size));           
        if (m_DataBuffer == NULL)
        {
            status = STATUS_INSUFFICIENT_RESOURCES;
            return;
        }                                           
        status = pFile->ReadFile(m_DataBuffer,size);
        if (NT_ERROR(status))
        {
            ExFreePool(m_DataBuffer);
            status = STATUS_UNSUCCESSFUL;
        }
        else
        {
            KdPrint(("Samples in file: %d\n",m_MaxSamples));
        }
    }
    __finally
    {
        delete pFile;
        if (status != STATUS_SUCCESS)
        {
            m_MaxSamples = 2;
            m_DataBuffer = static_cast<PUSHORT>(ExAllocatePool(NonPagedPool,m_MaxSamples * sizeof(USHORT)));
            if (m_DataBuffer == NULL)
            {
                ThrowStatusExceptionFromConstructor(STATUS_INSUFFICIENT_RESOURCES, 
                    "DATADRIVERDEVICE:CDataDriverDevice::CDataDriverDevice Failed to allocate memory");            
            }
            else
            {
                m_fIsSquareWave = TRUE;
                m_DataBuffer[0] = 500;
                m_DataBuffer[1] = 1000;            
            }
        }    
    }                                
}

#ifdef WINDK_CODE_SEG_PRAGMA
#pragma code_seg( )
#endif //WINDK_CODE_SEG_PRAGMA

void CDataDriverDevice::TimerCallback(PIRP pIrp,PVOID SystemArgument1,PVOID SystemArgument2)
{
    if (pIrp == NULL)
    {	
        m_MissedSamples++;        
        return;
    }        
    // check for already cancelled irp
    if (CheckIrpCanceled(pIrp)) 
    {
        return;
    }
    DATA_BUFFER *db = reinterpret_cast<DATA_BUFFER *>(pIrp->AssociatedIrp.SystemBuffer);
        
    if (db == NULL)
    {
        KdPrint(("DataDriver: Data buffer is NULL\n"));
        return;
    }
    db->timeStamp = 0;			        
    if (++m_IsrCounter >= m_Frequency)
	{
        m_IsrCounter = 0;
		KeQuerySystemTime(reinterpret_cast<PLARGE_INTEGER>(&db->timeStamp));	
	}
    if (m_fIsSquareWave == FALSE || m_IsrCounter == 0)
    {
        if (++m_CurrentSample >= m_MaxSamples)
        {
            m_CurrentSample = 0;            
        }         
    }        
    for (int ch = 0;ch < BUFFER_ENTRIES;ch++)
    {                            
        if (m_ChannelMap[ch] == '1')
        {
            db->data[ch] = m_DataBuffer[m_CurrentSample];   
        }            
    }        
    pIrp->IoStatus.Status = STATUS_SUCCESS;
    pIrp->IoStatus.Information = sizeof(DATA_BUFFER);
    CompleteIrpAndStartNext(pIrp,STATUS_SUCCESS);	    
}    

/////////////////////////////////////////////////////////////////////
//  CRepTimer TimerCallback - timer callback
//  Input:  pDpc - dpc object
//          pContext - CRepTimer device
//          SystemArgument1 - unused
//          SystemArgument2 - unused
//  Output: none
//  Notes:  This function calls the member TimerCallback
/////////////////////////////////////////////////////////////////////
VOID TimerCallback(PKDPC pDpc,PVOID pContext,
                   PVOID SystemArgument1, 
                   PVOID SystemArgument2)
{
    CDataDriverDevice *pDevice = reinterpret_cast<CDataDriverDevice *>(pContext);
    
    if (pDevice->m_Timer.IsTimerCanceled())
    {
        KdPrint(("DataDriver: Timer was Canceled\n"));
        return;
    }
    pDevice->TimerCallback(pDevice->GetCurrentIrp(),SystemArgument1,SystemArgument2);           
}
