#include <vdw.h>
#pragma hdrstop
#include "DataDriverDevice.h"			   

#pragma code_seg("INIT")

// DataDriverDevice Constructor
//		The device constructor is typically responsible for allocating
//		any physical resources that are associated with the device.
//
//		The device constructor often reads the registry to setup
//		various configurable parameters.
//

DataDriverDevice :: DataDriverDevice(ULONG unit) :
		KDevice(
			KUnitizedName(L"DataDriverDevice",unit),
			DATADRIVERDEVICE_DEVICE_TYPE,
			KUnitizedName(L"DataDriverDevice",unit),
			0,
			DO_BUFFERED_IO), 
        m_fIsSampling(FALSE),
		m_fConfigured(FALSE),
        m_fIsSquareWave(FALSE),
        m_MissedSamples(0),        
        m_DataBuffer(NULL),
		m_Frequency(0)
{    	                    
    if (!NT_SUCCESS(m_ConstructorStatus) )
	{
		DbgPrint("DataDriverDevice: Failed to create device, unit number %d, status %d\n",unit,m_ConstructorStatus);
		return;
	}
    KUnitizedName *regPath = KDevice::CreateRegistryPath(L"DataDriverDevice",unit);    
      
    // Check the registry if we have a file available, if we have
    // open it, allocate memory and read the file.
    KUstring        dataFileName;
    NTSTATUS        status;
    KRegistryKey    unitKey(*regPath); 
    
    if (NT_SUCCESS(unitKey.LastError()))
    {
        ULONG length = 0;
        PWSTR temp = NULL;
            
        status = unitKey.QueryValue(L"DataFile",temp,length);
        if (NT_SUCCESS(status))
        {
            dataFileName.GrowMaxBy(length);
            dataFileName.Assign(temp);
        }
        delete temp;                                    
    }
    delete regPath;    
    __try
    {       
        if (status == STATUS_SUCCESS)
        {
            KFile file;    
        
            status = file.OpenCreate(
    	        dataFileName,
    		    NULL,
    		    FILE_GENERIC_READ | SYNCHRONIZE,
    		    OBJ_CASE_INSENSITIVE,
    		    0,
    		    FILE_SHARE_READ,
    		    FILE_OPEN,
    		    FILE_NON_DIRECTORY_FILE | FILE_SYNCHRONOUS_IO_NONALERT
                );
            if (NT_ERROR(status))
            {
                return;
            }            
            FILE_STANDARD_INFORMATION si;
        
    	    file.Query(&si);
            m_MaxSamples = (si.EndOfFile.LowPart / sizeof(USHORT));
            if (m_MaxSamples == 0)
            {
                status = STATUS_UNSUCCESSFUL;
                return;
            }                                    
            m_DataBuffer = new (NonPagedPool) USHORT[m_MaxSamples];
            if (m_DataBuffer == NULL)
            {
                status = STATUS_INSUFFICIENT_RESOURCES;
                return;
            }
            ULONG bytesRead;            
                                
            status = file.Read(reinterpret_cast<PUCHAR>(m_DataBuffer),si.EndOfFile.LowPart,&bytesRead);
            if (NT_ERROR(status) || bytesRead != si.EndOfFile.LowPart)
            {
                delete m_DataBuffer;               
                status = STATUS_UNSUCCESSFUL;
            }
            else
            {
                DbgPrint("Samples in file: %d\n",m_MaxSamples);
            }
        }
    }
    __finally
    {        
        if (status != STATUS_SUCCESS)
        {
            m_MaxSamples = 2;
            m_DataBuffer = new (NonPagedPool) USHORT[m_MaxSamples];
            if (m_DataBuffer == NULL)
            {
                m_ConstructorStatus = STATUS_INSUFFICIENT_RESOURCES;
            }
            else
            {
                m_fIsSquareWave = TRUE;
                m_DataBuffer[0] = 500;
                m_DataBuffer[1] = 1000;            
            }
        }
    }        
}

#pragma code_seg()

// DataDriverDevice Destructor
//
DataDriverDevice::~DataDriverDevice()
{
    delete m_DataBuffer;
    m_DataBuffer = NULL; 
}

//	Cancel IRP
//		This routine is called when an IRP is cancelled.
//
VOID DataDriverDevice::Cancel(KIrp irp)
{
	irp.Information() = 0;
	if (irp == CurrentIrp())
	{
	    CurrentIrp() = NULL;
		CancelSpinLock::Release(irp.CancelIrql());
		irp.Status() = STATUS_CANCELLED;
		NextIrp(irp);
	}
	else
	{
		KDeviceQueue(DeviceQueue()).RemoveSpecificEntry(irp);
		CancelSpinLock::Release(irp.CancelIrql());
		irp.Complete(STATUS_CANCELLED);
	} 
}

//	StartIO
//		This routine is called when an IRP is taken off
//		the device queue (used for serializing I/O) and
//		presented for processing.
//
VOID DataDriverDevice::StartIo(KIrp irp)
{  
}

// Major function handlers
//
NTSTATUS DataDriverDevice :: Read(KIrp irp)
{
    irp.Information() = 0;	
	if (irp.BufferedReadDest() == NULL)
	{
		return(irp.Complete(STATUS_INVALID_PARAMETER));				
	}
    if (irp.ReadSize() < sizeof(DATA_BUFFER))
    {        
		irp.Information() = 0;	
		return(irp.Complete(STATUS_BUFFER_TOO_SMALL));		
    }    
   	return(QueueIrp(irp,LinkTo(Cancel)));
}    

NTSTATUS DataDriverDevice :: DeviceControl(KIrp irp)
{
    NTSTATUS status = STATUS_SUCCESS;
    
    switch (irp.IoctlCode())    
    {        
	    case IOCTL_DATADRIVER_SETUP:
		{
			if (m_fConfigured == TRUE)
			{
			    return (irp.Complete(STATUS_DEVICE_CONFIGURATION_ERROR));
			}						                                        
			if (irp.IoctlBuffer() == NULL || irp.IoctlInputBufferSize() < sizeof(DEVICE_CONFIGURATION))
			{
    		    return (irp.Complete(STATUS_INVALID_PARAMETER));
	        }
            DEVICE_CONFIGURATION    dc = *reinterpret_cast<DEVICE_CONFIGURATION *>(irp.IoctlBuffer());
			if (dc.freq <= 0 || dc.freq > 1024)
			{
            	DbgPrint("DataDriver: Frequency is out of range %d\n",dc.freq);
                dc.freq = 1;
			}            
            m_Frequency = dc.freq;
            RtlCopyMemory(m_ChannelMap,dc.channelMap,sizeof(dc.channelMap));
            m_fConfigured = TRUE;            
            break;
		}						
	    case IOCTL_DATADRIVER_START:
	    {
			if (m_fIsSampling == TRUE)
			{
    		    return (irp.Complete(STATUS_DEVICE_BUSY));
			}
			if (m_fConfigured == FALSE)
			{
			    return (irp.Complete(STATUS_DEVICE_CONFIGURATION_ERROR));
	        }
            m_MissedSamples = 0;
            m_CurrentSample = 0;            
            m_fIsSampling = TRUE;
			m_IsrCounter = m_Frequency;
            DbgPrint("DataDriver: Start timer, frequency = %d(%dms)\n",m_Frequency,static_cast<ULONG>(1000 / m_Frequency));            
            m_Timer.SetPeriodic(-1,static_cast<ULONG>(1000 / m_Frequency),LinkTo(TimerCallback),this);            
            break;
        }			
	    case IOCTL_DATADRIVER_STOP:
        {
			if (m_fConfigured == FALSE)
			{
			    return (irp.Complete(STATUS_DEVICE_CONFIGURATION_ERROR));
            }
            m_Timer.Cancel();
            m_fIsSampling = FALSE;
			if (m_MissedSamples)
			{
			    DbgPrint("DataDriver: Missed samples = %d\n",m_MissedSamples);
			}
            break;
        }    
        default:
            status = STATUS_NOT_IMPLEMENTED;
    }    
    return(irp.Complete(status));        
}
				
NTSTATUS DataDriverDevice::CleanUp(KIrp irp)
{
	if (CurrentIrp() != NULL)
	{
		CancelSpinLock::Acquire();			
		
		KIrp irp = CurrentIrp();
		if (irp != NULL && irp.WasCanceled() == FALSE)
		{
			irp.SetCancelRoutine(NULL);
			CancelSpinLock::Release();																 
			irp.Information() = 0;
			irp.Status() = STATUS_CANCELLED;
			NextIrp(irp);			
		}
		else
		{
			CancelSpinLock::Release();
		}
	}
  	KDeviceQueue dq(DeviceQueue());
	dq.CleanUp(irp.FileObject());
	irp.Information() = 0;		
	return irp.Complete(STATUS_SUCCESS);				    
}

NTSTATUS DataDriverDevice::Create(KIrp irp)
{
    return irp.Complete(STATUS_SUCCESS);
}    

NTSTATUS DataDriverDevice::Close(KIrp irp)
{
    if (m_fIsSampling == TRUE)
    {
        m_Timer.Cancel();
        m_fIsSampling = FALSE;
    }
	m_fConfigured = FALSE;
	irp.Information() = 0; 
	return irp.Complete(STATUS_SUCCESS);
}

void DataDriverDevice :: TimerCallback(PVOID arg1,PVOID arg2)
{			        
    KIrp irp = CurrentIrp();

    if (CurrentIrp() != NULL)
    {	
	    CancelSpinLock::Acquire();
		if (irp.WasCanceled())
		{
			CancelSpinLock::Release();
			return;
		}
		else
		{
			irp.SetCancelRoutine(NULL);
			CancelSpinLock::Release();
		}
        DATA_BUFFER *db = reinterpret_cast<DATA_BUFFER *>(irp.BufferedReadDest());
        
        if (db == NULL)
        {
            DbgPrint("DataDriver: Data buffer is NULL\n");
            return;
        }
        db->timeStamp = 0;			        
    	if (++m_IsrCounter >= m_Frequency)
	    {
    		m_IsrCounter = 0;
		    KeQuerySystemTime(reinterpret_cast<PLARGE_INTEGER>(&db->timeStamp));	
	    }
        if (m_fIsSquareWave == FALSE || m_IsrCounter == 0)
        {
            if (++m_CurrentSample >= m_MaxSamples)
            {
                m_CurrentSample = 0;
            }            
        }            
        for (LONG ch = 0;ch < BUFFER_ENTRIES;ch++)
        {                            
            if (m_ChannelMap[ch] == '1')
            {
                db->data[ch] = m_DataBuffer[m_CurrentSample];   
            }            
        }                                
		irp.Status() = STATUS_SUCCESS;
        
		irp.Information() = sizeof(DATA_BUFFER);
		NextIrp(irp);												   
    }
    else
    {
        m_MissedSamples++;
    }
}