#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <err.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/pioctl.h>

extern int setup_and_wait(char **);
extern int start_tracing(int, int);
extern void i386_syscall_entry(int, int);
extern void i386_syscall_exit(int, int);

int pid = 0;
int nosigs = 0;
FILE *outfile = stderr;
char *prog;
int Procfd;
char progtype[50];	/* OS and type of executable */

static inline void
usage(void) {
  fprintf(stderr, "usage:  %s [-o <file>] [-S] { [-p <pid> ] | "
	  "[ <command> <args>] }\n", prog);
  exit(1);
}

struct ex_types {
  char *type;
  void (*enter_syscall)(int, int);
  void (*exit_syscall)(int, int);
} ex_types[] = {
  { "FreeBSD a.out", i386_syscall_entry, i386_syscall_exit },
  { 0, 0, 0 },
};

main(int ac, char **av) {
  int mask;
  int c;
  int i;
  char **command;
  struct procfs_status pfs;
  char etype[25];
  struct ex_types *funcs;
  int fd;

  prog = av[0];

  while ((c = getopt(ac, av, "p:o:S")) != EOF) {
    switch (c) {
    case 'p':	/* specified pid */
      pid = atoi(optarg);
      break;
    case 'o':	/* Specified output file */
      if ((outfile = fopen(optarg, "w")) == NULL) {
	fprintf (stderr, "%s:  cannot open %s\n", av[0], optarg);
	exit(1);
      }
      break;
    case 'S':	/* Don't trace signals */ 
      nosigs = 1;
      break;
    default:
      usage();
    }
  }

  if (optind == ac && pid == -1)
    usage();

  if (!pid) {
    command = &av[optind];
    pid = setup_and_wait(command);
  }

  Procfd = start_tracing(pid, S_EXEC | S_SCE | S_SCX | S_CORE | S_EXIT |
		     (nosigs ? 0 : S_SIG));
  pfs.why = 0;

  sprintf(etype, "/proc/%d/etype", pid);
  if ((fd = open(etype, O_RDONLY)) == -1) {
    strcpy(progtype, "FreeBSD a.out");
  } else {
    int len = read(fd, progtype, sizeof(progtype));
    progtype[len-1] = '\0';
    close(etype);
  }

  for (funcs = ex_types; funcs->type; funcs++)
    if (!strcmp(funcs->type, progtype))
      break;

  do {
    int val = 0;

    if (ioctl(Procfd, PIOCWAIT, &pfs) == -1)
      perror("PIOCWAIT top of loop");
    else {
      switch(i = pfs.why) {
      case S_SCE:
	funcs->enter_syscall(pid, pfs.val);
	break;
      case S_SCX:
	funcs->exit_syscall(pid, pfs.val);
	break;
      case S_SIG:
	fprintf(outfile, "SIGNAL %d\n", pfs.val);
	break;
      case S_EXIT:
	fprintf (outfile, "process exit, rval = %d\n", pfs.val);
	break;
      default:
	fprintf (outfile, "Process stopped because of:  %d\n", i);
	break;
      }
    }
    if (ioctl(Procfd, PIOCCONT, &val) == -1)
      perror("PIOCCONT");
  } while (pfs.why != S_EXIT);
  return 0;
}
