//******************************************************************************
// idtAudio.java
//
// This applet effectively performs streaming audio in Java.
//
//******************************************************************************
import java.applet.*;
import java.awt.*;
import java.net.*;
import java.io.*;
import sun.audio.*;	// This is an unsupported class from sun which makes the
					// streaming audio possible.

// Main Class for applet idtAudio
public class idtAudio extends Applet
{
	// These values are tuned for 28.8 modems, but they will work for other speeds
	final int ENCODEDSPS = 2000;
	final int DECODEDSPS = 8000;

	// Threads and fifos
	DoT downloadThread = null;
	DeT decodeThread = null;
	FIS decodedStream;
	FIS encodedStream;

	// Header information from the encoded audio file.  Some of the information is
	// useful for licensing.
	public long capabilities;
	public long encoded_size;
	public long decoded_size;
	public long expiration_date;
	public long registration;

	// Image information
	private Image cabinetPicture = null;
	final private Rectangle indicator1 = new Rectangle( 49, 5, 6, 20 );
	final private Rectangle indicator2 = new Rectangle( 59, 5, 6, 20 );
	final private Rectangle indicator3 = new Rectangle( 69, 5, 6, 20 );
	final private Rectangle indicator4 = new Rectangle( 48, 4, 28, 22 );

    // Members for applet parameters
	private String AudioStreamURL = null;
	private int SequenceLength = 30;
	private boolean AutoPlay = false;
	private boolean Show = true;
	private boolean Debug = false;

	// Used for internal tracking and displaying LED lights
	final public int STATEOFF = 0;
	final public int STATEOK = 1;
	final public int STATEDONE = 2;
	final public int STATEWAIT = 3;
	private int isPlaying = 0;
	private int isDownloading = 0;
	private int isDecoding = 0;

    // Parameter names.  To change a name of a parameter, you need only make
	// a single change.  Simply modify the value of the parameter string below.
	private final String PARAM_AudioStreamURL = "AudioStreamURL";
	private final String PARAM_SequenceLength = "SequenceLength";
	private final String PARAM_AutoPlay = "AutoPlay";
	private final String PARAM_Show = "Show";
	private final String PARAM_Debug = "Debug";

	// APPLET INFO SUPPORT:
	//		The getAppletInfo() method returns a string describing the applet's
	// author, copyright date, or miscellaneous information.
	public String getAppletInfo()
	{
		return "idtAudio Applet V1.0 Beta 1\r\n" +
		       "by L. Richard Moore Jr.\r\n" +
		       "Copyright (c) 1996, 1997" +
		       "All Rights Reserved.";
	}

	// PARAMETER SUPPORT
	//		The getParameterInfo() method returns an array of strings describing
	// the parameters understood by this applet.
	//
    // idtAudio Parameter Information:
    //  { "Name", "Type", "Description" },
	public String[][] getParameterInfo()
	{
		String[][] info =
		{
			{ PARAM_AudioStreamURL, "String", "URL pointer to the audio stream" },
			{ PARAM_SequenceLength, "int", "Audio buffer length in seconds" },
			{ PARAM_AutoPlay, "boolean", "Flags whether the applet should immediately begin playing audio" },
			{ PARAM_Show, "boolean", "Flags whether player should be displayed on html page or not" },
			{ PARAM_Debug, "boolean", "Flags whether player uses debug display" },
		};
		return info;		
	}

	// Signals the applet to start.  For idtAudio, we don't start playing unless the
	// "AutoPlay" parameter is true.  If not, we do nothing.
	public void start()
	{
		if( AutoPlay == true )
			begin();
	}
	
	// Stops our applet.
	public void stop()
	{
		stopPlaying();
	}

	// The init() method is called by the AWT when an applet is first loaded or
	// reloaded.  Override this method to perform whatever initialization your
	// applet needs, such as initializing data structures, loading images or
	// fonts, creating frame windows, setting the layout manager, or adding UI
	// components.
    //--------------------------------------------------------------------------
	public void init()
	{
		// PARAMETER SUPPORT
		//		The following code retrieves the value of each parameter
		// specified with the <PARAM> tag and stores it in a member
		// variable.
		String param;

		// AudioStreamURL: URL pointer to the audio stream
		param = getParameter(PARAM_AudioStreamURL);
		if (param != null)
			AudioStreamURL = param;

		// SequenceLength: Audio subsection length in seconds
		param = getParameter(PARAM_SequenceLength);
		if (param != null)
			SequenceLength = Integer.parseInt(param);
		if( SequenceLength < 10 )
			throw new RuntimeException( "Invalid SequenceLength parameter" );

		// AutoPlay: Flags whether the applet should immediately begin playing audio
		param = getParameter(PARAM_AutoPlay);
		if (param != null)
			AutoPlay = Boolean.valueOf(param).booleanValue();

		// Show: Flags whether player should be displayed on html page or not
		param = getParameter(PARAM_Show);
		if (param != null)
			Show = Boolean.valueOf(param).booleanValue();

		param = getParameter(PARAM_Debug);
		if (param != null)
			Debug = Boolean.valueOf(param).booleanValue();

		// Resise ourselves to suite our image
		resize(80, 30);

		// Load the "cabinet" image if the "Show" flag is true.
		if( Show )
			try
			{
				cabinetPicture = getImage( getDocumentBase(), "idtAudio.gif" );
				MediaTracker tracker = new MediaTracker( this );
				tracker.addImage( cabinetPicture, 0 );
				tracker.waitForAll();
				tracker = null;
			}
			catch( InterruptedException e )
			{
				throw new RuntimeException( "Could not load image" );
			}

		// Create the FifoInputStreams.  One stream contains the encoded audio data
		// and the other contains the decoded audio data.  The second parameter indicates
		// an identification number used to manipulate the LEDs in the cabinet image.
		decodedStream = new FIS( this, 0, SequenceLength * DECODEDSPS );
		encodedStream = new FIS( this, 1, SequenceLength * ENCODEDSPS );
	}

	// This member begins audio play back.  It resets the FifoInputStreams, reads in the
	// audio header, and creates the threads needed to populate the Fifos.
	public synchronized void begin()
	{
		boolean haderror = false;

		// Reset FifoInputStreams
		encodedStream.reset();
		decodedStream.reset();

		try
		{
			// Perform garbage collection now so we don't interrupt audio playback
			System.gc();

			// This class is automatically created by sun.audio.*.  It expects as its parameter
			// an InputStream which contains an 8000Hz uLaw encoded audio file.
			AudioPlayer.player.start( decodedStream );

			// Create URL object pointing to the audio data specified by the AudioStreamURL parameter
			URL url = new URL( getDocumentBase(), AudioStreamURL );

			// Read header information
			InputStream is = url.openStream();
			DataInputStream dis = new DataInputStream( is );
			capabilities = dis.readLong();
			if( capabilities != 0 )
			{
				System.out.print( "Invalid audio file" );
				haderror = true;
			}
			encoded_size = dis.readLong();
			decoded_size = dis.readLong();
			expiration_date = dis.readLong();
			registration = dis.readLong();

			/*********************************************************
			This section of code might be used to make the audio files "expire" after a period
			of time and thus enforce a license policy.

			if(( expiration_date != 0 ) && ( System.currentTimeMillis() / 60000 > expiration_date ))
			{
				System.out.print( "Shareware license expired" );
				haderror = true;
			}
			**********************************************************/

			if( ! haderror )
			{
				isPlaying = STATEOK;
				paintNow();

				// Create threads which populate Fifos
				downloadThread = new DoT( this, is, encodedStream );
				decodeThread = new DeT( this, encodedStream, decodedStream );
			}
		}
		catch( MalformedURLException e )
		{
			System.out.print( "URL Expression error" );
			haderror = true;
		}
		catch( IOException e )
		{
			System.out.print( "IO Error in URL stream" );
			haderror = true;
		}
		finally
		{
			if( haderror )
				AudioPlayer.player.stop( decodedStream );
		}
	}

	// Ends any playback in progress
	public void destroy()
	{
		stopPlaying();
	}

	// Forces an immediate repaint
	public void paintNow()
	{
		Graphics g = getGraphics();
		paint( g );
	}

	// idtAudio Paint Handler
	public void paint(Graphics g)
	{
		// Draws the basic "cabinet" image
		g.drawImage( cabinetPicture, 0, 0, null );

		if( Debug == true )
		{
			// Displays all three LED's
			g.setColor( state2Color( isDownloading ));
			g.fillRect( indicator1.x, indicator1.y, indicator1.width, indicator1.height );
			g.setColor( state2Color( isDecoding ));
			g.fillRect( indicator2.x, indicator2.y, indicator2.width, indicator2.height );
			g.setColor( state2Color( isPlaying ));
			g.fillRect( indicator3.x, indicator3.y, indicator3.width, indicator3.height );
		}
		else
		{
			// Display only playback status LED
			g.setColor( state2Color( isPlaying ));
			g.fillRect( indicator4.x, indicator4.y, indicator4.width, indicator4.height );
		}
	}
	
	// Translates an LED state into a visible color
	Color state2Color( int state )
	{
		switch( state )
		{
		case STATEOFF:
			return( Color.black );
		case STATEOK:
			return( Color.green );
		case STATEDONE:
			return( Color.blue );
		case STATEWAIT:
			return( Color.red );
		}
		return( Color.orange );
	}

	// Sets the state of the download thread
	void setDownloading( int value )
	{
		isDownloading = value;
		paintNow();
	}

	// Sets the state of the decode thread
	void setDecoding( int value )
	{
		isDecoding = value;
		paintNow();
	}

	// Sets the state of a Fifo based upon the Fifo ID
	void setFifo( int id, int value )
	{
		switch( id )
		{
		case 0: // decodedStream
			switch( value )
			{
			case STATEDONE:
				isPlaying = STATEOFF;
				isDecoding = STATEOFF;
				isDownloading = STATEOFF;
				downloadThread.stop();
				decodeThread.stop();
				break;
			default:
				isPlaying = value;
				break;
			}
			break;
		case 1: // encodedStream
			switch( value )
			{
			case STATEDONE:
				break;
			default:
				isDecoding = value;
				break;
			}
			break;
		}
		paintNow();
	}

	// When the user clicks on the cabinet image, playing starts or stops.
	public boolean mouseDown(Event evt, int x, int y)
	{
		// Turn off if playing
		if( isPlaying + isDecoding + isDownloading != 0 )
			stopPlaying();
		else
			begin();
		return true;
	}

	// Stops all audio playback.  Stops all threads.  Stops the AudioPlayer.
	public void stopPlaying()
	{
		if( isPlaying + isDecoding + isDownloading != 0 )
		{
			AudioPlayer.player.stop( decodedStream );
			if( downloadThread != null )
				downloadThread.stop();
			try
			{
				if( downloadThread.inputStream != null )
					downloadThread.inputStream.close();
			}
			catch( IOException e ) {}

			if( decodeThread != null )
				decodeThread.stop();
			repaint();
		}
		isPlaying = isDecoding = isDownloading = STATEOFF;
	}
}

//
// DoT - DownloadThread class
//
// This simple little class downloads the encoded audio data from the server and places
// the data into a specified FifoInputStream
//
class DoT extends Thread
{
	InputStream inputStream;
	FIS outputStream;

	idtAudio parent;

	// Constructor requires the applet, an InputStream to read from, and a FifoInputStream
	// to write to.
	public DoT( idtAudio p, InputStream is, FIS os )
	{
		parent = p;

		inputStream = is;
		outputStream = os;
		start();
	}

	// Starting the thread executes the member where all the work gets done.
	public void run()
	{
		int bufsize = 2000;
		byte buf[] = new byte[ bufsize ];
		int count;

		Thread.currentThread().setPriority( Thread.currentThread().getPriority() + 1 );
		parent.setDownloading( parent.STATEOK );

		try
		{
			count = inputStream.read( buf, 0, bufsize );
			while( count > 0 )
			{
				outputStream.write( buf, count );
				count = inputStream.read( buf, 0, bufsize );
				Thread.currentThread().yield();
			}
		}
		catch( Exception e )
		{
		}
		finally
		{
			outputStream.setEOF();
		}

		parent.setDownloading( parent.STATEDONE );
		Thread.currentThread().suspend();
	}
}

//
// DeT - DecodeThread class
//
// This simple little class decodes the encoded audio data from the FifoInputStream and places
// the data into another FifoInputStream
//
class DeT extends Thread
{
	InputStream inputStream;
	FIS outputStream;

	// Decode variables 
	int run, rise;
	int sample = 0;
	int x, y;
	int lasty = 0;

	idtAudio parent;

	// Constructor requires the applet, an InputStream to read from, and a FifoInputStream
	// to write to.
	public DeT( idtAudio p, InputStream is, FIS os ) throws IOException
	{
		parent = p;
		inputStream = is;
		outputStream = os;

		start();
	}

	// Starting the thread executes the member where all the work gets done.
	public void run()
	{
		int bufsize = 400;
		byte buf[] = new byte[ bufsize ];
		int count;

		Thread.currentThread().setPriority( Thread.currentThread().getPriority() + 1 );
		parent.setDecoding( parent.STATEOK );
		try
		{
			count = readDirect( buf, bufsize );
			while( count > 0 )
			{
				outputStream.write( buf, count );
				count = readDirect( buf, bufsize );
			}
		}
		catch( Exception e )
		{
		}
		finally
		{
			outputStream.setEOF();
		}

		parent.setDecoding( parent.STATEDONE );
		Thread.currentThread().suspend();
	}

	// Called by run() thread to read a buffer of bytes
    public int readDirect( byte buf[], int len ) throws IOException 
	{
		int count;
		int value;
		int input;

		for( count = 0; count < len; ++count )
		{
			if( sample >= run )
			{
				lasty = y;

				input = inputStream.read();
				if( input == -1 )
					return( count );

				input = (byte) input;
				run = ( input & 0x0f ) + 1;
				input >>= 4;
				if( input < 0 )
					y = - ( input * input );
				else
					y = input * input;

				rise = y - lasty;
				sample = 0;
			}

			value = (( sample * rise ) / run + lasty ) * 256;
			sample++;
		
			/* Get the sample into sign-magnitude. */
			if( value < 0 )
			{
				value = -value;

				/* Convert from 16 bit linear to ulaw. */
				value += ULAW_BIAS;
				int exponent = lut[value >> 7];
				int mantissa = (value >> (exponent + 3)) & 0x0F;
				value = ((exponent << 4) | mantissa) ^ 0xFF;
			} 
			else 
			{
				/* Convert from 16 bit linear to ulaw. */
				value += ULAW_BIAS;
				int exponent = lut[value >> 7];
				int mantissa = (value >> (exponent + 3)) & 0x0F;
				value = ((exponent << 4) | mantissa) ^ 0x7F;
			}

			// This should be the only direct array cell reference
			// in this program.  Arrays seem to be extremely costly
			// against performance, so we want to minimize their use.
			buf[ count ] = (byte) value;
		}
		return( count );
    }

	// Tables used for uLaw encoding
    private final static int ULAW_BIAS = 0x84;
    private final static int lut[] = 
	{
		0, 0, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
		4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
		5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
		5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
		6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
		6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
		7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
    };
}

//
// FIS - FifoInputStream 
//
// This reusable class implements a standard InputStream with Fifo functionality.
// This circular buffer can be accessed asynchronously.
class FIS extends InputStream
{
	final static int WAITLOOPTIME = 50;

	byte buffer[]; // the buffer array itself
	int size; // total size of the buffer
	int head; // head pointer
	int tail; // tail pointer
	int full; // How full the buffer is
	boolean eof = false;

	// The applet creating us
	idtAudio parent;
	int id;

	// Constructor gets the applet, an ID, and a size.
	public FIS( idtAudio p, int i, int s )
	{
		id = i;
		parent = p;
		size = s;
		buffer = new byte[ size ];
		reset();
	}

	// Reset all pointers and empty buffer
	public void reset()
	{
		synchronized( this )
		{
			head = tail = 0;
			full = 0;
			eof = false;
		}
	}

	// Flags that there is no more data to submit into the Fifo
	public void setEOF()
	{
		eof = true;
	}

	// Used by threads to insert a buffer of data into the Fifo
	public void write( byte buf[], int length )
	{
		// Make the thread wait until there is enough room in the buffer
		while( full + length > size )
			try
			{
				Thread.currentThread().sleep( WAITLOOPTIME );
			}
			catch( InterruptedException e )
			{
			}

		// Copy the data into the buffer
		synchronized( this )
		{
			if( length + head > size )
			{
				int split = size - head;

				System.arraycopy( buf, 0, buffer, head, split );
				System.arraycopy( buf, split, buffer, 0, length - split );
			}
			else
				System.arraycopy( buf, 0, buffer, head, length );
			full += length;
		}

		// Adjust pointers and sizes accordingly
		head += length;
		if( head >= size )
			head -= size;
	}

	// Used by threads to read data from the FifoInputStream.  It overrides the standard
	// read() member from the standard InputStream().
	public int read( byte buf[], int pos, int length )
	{
		// Make the calling thread wait until data is available
		if( full < length )
		{
			parent.setFifo( id, parent.STATEWAIT );
			while((full < size / 2 ) && ( eof == false ))
				try
				{
					Thread.currentThread().sleep( WAITLOOPTIME );
				}
				catch( InterruptedException e )
				{
				}
			parent.setFifo( id, parent.STATEOK );
		}

		// If there is no more data to submit, return as much as we can
		if( eof == true )
		{
			if( length > full )
				length = full;
		}

		// Copy the data out of the fifo buffer
		synchronized( this )
		{
			System.arraycopy( buffer, tail, buf, pos, length );
			full -= length;
		}

		// Adjust pointers and size accordingly
		tail += length;
		if( tail >= size )
			tail -= size;

		// Adjust LED state if necessary
		if( length <= 0 )
			parent.setFifo( id, parent.STATEDONE );
		
		return( length );
	}

	// Used by threads to read data from the FifoInputStream.  It overrides the standard
	// read() member from the standard InputStream().
	public int read()
	{
		int value = 0;

		// Make the calling thread wait until data is available
		if( full == 0 )
		{
			parent.setFifo( id, parent.STATEWAIT );
			while(( full == 0 ) && ( eof == false ))
				try
				{
					Thread.currentThread().sleep( WAITLOOPTIME );
				}
				catch( InterruptedException e )
				{
				}
			parent.setFifo( id, parent.STATEOK );
		}

		// If there is no more data to submit, return as much as we can
		if( eof == true )
			if( full == 0 )
				return( -1 );

		// Copy the data out of the fifo buffer
		synchronized( this )
		{
			value = buffer[ tail ];
			full--;
		}

		// Adjust pointers and size accordingly
		value = value & 0xff;
		tail++;
		if( tail >= size )
			tail -= size;

		// Adjust LED state if necessary
		if( value < 0 )
			parent.setFifo( id, parent.STATEDONE );

		return( value );
	}
}