/* tstrecit.c -- Demo program for ternary search trees
  Test cost of recursion vs. iteration on a real structure.
  This code is derived from "Ternary Search Trees" by Jon
Bentley and Robert Sedgewick in the April, 1998, Dr. Dobb's Journal.
  Usage
	tstrecit dictfile
This runs a set of basic experiments on the run time of
ternary search trees algorithms using the dictionary
in the file dictfile (one word per line).
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

/* TERNARY SEARCH TREE ALGS */

typedef struct tnode *Tptr;
typedef struct tnode {
    char splitchar;
    Tptr lokid, eqkid, hikid;
} Tnode;
Tptr root;

char *insertstr;

Tptr rinsert(Tptr p, char *s)
{   if (p == 0) {
        p = (Tptr) malloc(sizeof(Tnode));
        p->splitchar = *s;
        p->lokid = p->eqkid = p->hikid = 0;
    }
    if (*s < p->splitchar)
        p->lokid = rinsert(p->lokid, s);
    else if (*s == p->splitchar) {
        if (*s == 0)
            p->eqkid = (Tptr) insertstr;
		else
            p->eqkid = rinsert(p->eqkid, ++s);
    } else
        p->hikid = rinsert(p->hikid, s);
    return p;
}

int rsearch(Tptr p, char *s)
{   if (!p) return 0;
    if (*s == p->splitchar) {
        if (*s == 0)
			return 1;
        return rsearch(p->eqkid, s+1);
    } else if (*s > p->splitchar)
        return rsearch(p->hikid, s);
    else 
        return rsearch(p->lokid, s);
}

int isearch(char *s)
{   Tptr p;
    p = root;
    while (p) {
        if (*s == p->splitchar) {
            if (*s++ == 0)
                return 1;
            p = p->eqkid;
        } else if (*s > p->splitchar)
            p = p->hikid;
		else
            p = p->lokid;
	}
	return 0;
}

char *srcharr[100000];
int	srchtop, nodecnt;

void pmsearch(Tptr p, char *s)
{   if (!p) return;
    nodecnt++;
    if (*s == '.' || *s < p->splitchar)
        pmsearch(p->lokid, s);
    if (*s == '.' || *s == p->splitchar)
        if (p->splitchar && *s)
            pmsearch(p->eqkid, s+1);
    if (*s == 0 && p->splitchar == 0)
        srcharr[srchtop++] =
            (char *) p->eqkid;
    if (*s == '.' || *s > p->splitchar)
        pmsearch(p->hikid, s);
}

void pmsearch2(Tptr p, char *s)
{	while (p) {
		nodecnt++;
		if (*s == '.') {
			pmsearch2(p->lokid, s);
			pmsearch2(p->hikid, s);
			if (p->splitchar == 0) return;
			p = p->eqkid;
			++s;
		} else {
			if (*s < p->splitchar)
				p = p->lokid;
			else if (*s > p->splitchar)
				p = p->hikid;
			else /* *s == p->splitchar */ {
				if (*s == 0) {
					if (p->splitchar == 0)
						srcharr[srchtop++] = (char *) p->eqkid;
					return;
				}
				p = p->eqkid;
				++s;
			}
		}
	}
}

void nearsearch(Tptr p, char *s, int d)
{   if (!p || d < 0) return;
    nodecnt++;
    if (d > 0 || *s < p->splitchar)
        nearsearch(p->lokid, s, d);
    if (p->splitchar == 0) {
        if ((int) strlen(s) <= d)
            srcharr[srchtop++] = (char *) p->eqkid;
    } else
        nearsearch(p->eqkid, *s ? s+1:s,
            (*s == p->splitchar) ? d:d-1);
    if (d > 0 || *s > p->splitchar)
        nearsearch(p->hikid, s, d);
}

void nearsearch2(Tptr p, char *s, int d)
{	while (p && d >= 0) {
		nodecnt++;
		if (d > 0) {
			nearsearch2(p->lokid, s, d);
			nearsearch2(p->hikid, s, d);
			if (p->splitchar == 0) {
				if ((int) strlen(s) <= d)
					srcharr[srchtop++] = (char *) p->eqkid;
				break;
			}
			if (*s) ++s;
			if (*s != p->splitchar) --d;
			p = p->eqkid;
		} else { /* d == 0 */
			if (*s < p->splitchar)
				p = p->lokid;
			else if (*s > p->splitchar)
				p = p->hikid;
			else /* *s == p->splitchar */ {
				if (p->splitchar == 0) {
					if ((int) strlen(s) <= d)
						srcharr[srchtop++] = (char *) p->eqkid;
					break;
				}
				if (*s) ++s;
				p = p->eqkid;
			}
		}
	}
}

/* TIMING */

int qscomp(const void *p, const void *q )
{   return strcmp(* (char**) p, * (char**) q );
}
#define DOQSORT qsort((void *) a, (size_t) n, sizeof(char *), qscomp)

void rinsall(char *a[], int n)
{   int m;
    if (n < 1) return;
    m = n/2;
    insertstr = a[m];
	root = rinsert(root,a[m]);
    rinsall(a, m);
    rinsall(a + m+1, n-m-1);
}

void insall(char *a[], int n)
{   root = 0;
	rinsall(a, n);
}

int searchtype;

void searchall(char *a[], int n)
{   int i, j;
	char s[100]; 
    for (i = 0; i < n; i++) {
		strcpy(s, a[i]);
		/* s[0]++;  Uncomment for unsuccessful searches */
		switch(searchtype) {
		case 1: j = isearch(s); break;
		case 2: j = rsearch(root, s); break;
		case 11: srchtop = 0; pmsearch(root, s); j = srchtop; break;
		case 12: srchtop = 0; pmsearch2(root, s); j = srchtop; break;
		case 21: srchtop = 0; nearsearch(root, s, 0); j = srchtop; break;
		case 22: srchtop = 0; nearsearch2(root, s, 0); j = srchtop; break;
		case 90: j = 1; break;
		}
		if (j != 1)  /* Uncomment for unsuccessful searches */
			printf("search bug at %d, val %d\n", i, j);
	}	
}

int	n = 0, starttime,  rptctr;

#define TASK(s)	printf("%s", s);
#define CIN		starttime = clock();
#define COUT	printf("\t%d", clock()-starttime);
#define NL	printf("\n")
#define REPEAT(s)	for (rptctr=0; rptctr<5; rptctr++) { s }; NL;

#define MAXWORDS 250000
#define MAXCHARS 3000000
char space[MAXCHARS], *a[MAXWORDS];

void timeall()
{   TASK("Build TST for searches");
		CIN; insall(a, n); COUT; NL;
    TASK("isearch (warmup)");
		searchtype = 1; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("null search");
	    searchtype = 90; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("rsearch");
		searchtype = 2; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("isearch");
		searchtype = 1; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("pmsearch");
	    searchtype = 11; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("pmsearch2");
	    searchtype = 12; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("nearsearch");
	    searchtype = 21; REPEAT(CIN; searchall(a, n); COUT;)
    TASK("nearsearch2");
	    searchtype = 22; REPEAT(CIN; searchall(a, n); COUT;)
}

int main(int argc, char *argv[])
{	int	i, globalstarttime;
	char *s = space, *t, *fname;
	FILE *fp;
	if (argc == 1) { /* no args */
		fname = "/usr/jlb/data/words"; /* default dict file */
	} else /* at least one arg: file name */
		fname = argv[1];
	setbuf(stdout, 0);
	if ((fp = fopen(fname, "r")) == NULL) {
		fprintf(stderr, "  Can't open file\n");
		exit(1);
	}
	globalstarttime = clock();
	TASK("Big Malloc");
	    CIN; t = (char *) malloc(8000000*sizeof(char));
	    free(t); COUT; NL;
	TASK("Reading Input");
	    CIN;
	    a[0] = s;
	    while ((i = getc(fp)) != EOF) {
		    if (i == '\n') {
			    *s++ = 0;
			    a[++n] = s;
		    } else
			    *s++ = i;
	    }
	    COUT; NL;
	TASK("System Qsort");
	    CIN; DOQSORT; COUT; NL;
	timeall();
	i = clock() - globalstarttime;
	printf("Total clicks\t%d\nTotal secs\t%4.3f\n",
		i, (double) i / CLOCKS_PER_SEC);
	return 0;
}

