/* *** _ds_lib.c ***

Copyright (c) 1998 by UWI Unisoft Wares Inc.
   UWI.Com -- The Internet Forms Company 
   400 - 1095 McKenzie Ave.
   Victoria, British Columbia Canada V8P 2L5
   1-888-517-2675
   http://www.uwi.com

Author: John Boyer

Neither John Boyer, UWI Unisoft Wares Inc., nor Dr. Dobb's Journal will be
   liable under any circumstances for any direct or indirect damages arising
   from any use, direct or indirect, of this software. This software is
   provided with no warranty, implicit or explicit.

UWI Unisoft Wares Inc. grants you license to use this software in your programs
under the following conditions:
1) This copyright and license message must remain in this source code.
2) The following copyright message must appear in the program credits and
        documentation acknowledgements of software containing this library

   The Digital Signature Library for Microsoft(R) CryptoAPI
   Written By John Boyer Copyright (c) 1998 by UWI Unisoft Wares Inc.
*/

#include "ds_lib.h"
#include "_ds_lib.h"
#include "dserror.h"

/* This is here so you can even compile with vc4.0 */
#ifndef CRYPT_E_NOT_FOUND
#define CRYPT_E_NOT_FOUND                _HRESULT_TYPEDEF_(0x80092004L)
#endif

/**********************************************************************/
/* This function obtains the common name and email address of the
	certificate. This is the textual value that a user sees.
	For example, a user will choose (from a list of these
	subject lines) an identity with which to digitally sign
	a document. The subject line will become part of what
	gets signed, so a person who must later verify the
	signature can use the subject line generated here to
	compare with the subjects in his/her address book of
	public key certificates.

	@param pCertContext UI - the certificate from which we would
		like to extract a subject

	@param pCertSubject UO - a pointer to where to store the subject
		string. This may be set to NULL if the required subject
		attributes are not available.

	@param wholeSubject NI - if NOTOK then just the common name and
		email are given. if OK, then the whole subject is
		converted to a string and given.

	@return N OK on success or an error code on failure

	@access private
*/
/**********************************************************************/

short _DSGetSubjectFromCert(PCCERT_CONTEXT pCertContext,
			      charP *pCertSubject, short wholeSubject)
{
charP		CertSubject;
charP		Attr1 = NULL;
charP		Attr2 = NULL;
long		CertSubjectSize;
short		theError;

/* Verify the input parameters */

	if (pCertContext == NULL || pCertSubject == NULL
		|| (wholeSubject != OK && wholeSubject != NOTOK))
	{
		MEInvalidParameters(_DSGETSUBJECTFROMCERT);
		return(NOTOK);
	}

/* Set the output parameter to a known value */

	(*pCertSubject) = NULL;

/* Do we want the entire credentials */

	if (wholeSubject == OK)
	{
		CertSubjectSize = (*theCryptEnv->CertNameToStrProc)(X509_ASN_ENCODING,
			&pCertContext->pCertInfo->Subject,
			CERT_SIMPLE_NAME_STR, NULL, 0);

		if ((CertSubject = cp_malloc(CertSubjectSize + 1)) == NULL)
		{
			MENoMemory(_DSGETSUBJECTFROMCERT);
			return(NOTOK);
		}

		(void) (*theCryptEnv->CertNameToStrProc)(X509_ASN_ENCODING, &pCertContext->pCertInfo->Subject,
			CERT_SIMPLE_NAME_STR, CertSubject, CertSubjectSize);

	}
	else
	{

/* Or do we just want the common name and email address from the subject?
	Note:  RDN stands for Relative Distinguished Name */

		if ((theError = _DSFindRDNAttr(szOID_COMMON_NAME, pCertContext, &Attr1)) != OK)
		{
			MELogErrorInfoStr(theError, _DSGETSUBJECTFROMCERT, ME_INTERNALFUNCTIONFAILED, NULL);
			return(theError);
		}

		if ((theError = _DSFindRDNAttr(szOID_RSA_emailAddr, pCertContext, &Attr2)) != OK)
		{
			MELogErrorInfoStr(theError, _DSGETSUBJECTFROMCERT, ME_INTERNALFUNCTIONFAILED, NULL);
			cp_free(Attr1);
			return(theError);
		}

		if ((CertSubject = cp_malloc(cp_strlen(Attr1) + cp_strlen(Attr2) + 3)) == NULL)
		{
			MENoMemory(_DSGETSUBJECTFROMCERT);
			cp_free(Attr1);
			cp_free(Attr2);
			return(NOTOK);
		}
		cp_strcpy(CertSubject, Attr1);
		cp_strcat(CertSubject, ", ");
		cp_strcat(CertSubject, Attr2);

                cp_free(Attr1);
                cp_free(Attr2);
                Attr1 = Attr2 = NULL;
	}

/* Finished */

	(*pCertSubject) = CertSubject;

	return(OK);
}

/**********************************************************************/
/* This function takes a given certificate context and retrieves
	the specified attribute.

	@param pszAttrId UI - the attribute to retrieve

	@param pCertContext UI - the certificate context

	@param pAttr UO - a pointer to where to store the attribute.
		NOTE: A successful return but pAttr equalling NULL
		means that the attribute did not exists within the
		certifcate context.

	@return N OK on success or an error code on failure

	@access private
*/
/**********************************************************************/

short _DSFindRDNAttr(LPCSTR pszAttrId, PCCERT_CONTEXT pCertContext, charP *pAttr)
{
PCERT_NAME_INFO	NameInfo;
DWORD		dwDataLen = 0;
PCERT_RDN_ATTR	theAttribute;
charP		theValue;
long		theLen;

/* Verify the input parameters */

	if (pszAttrId == NULL || pCertContext == NULL || pAttr == NULL)
	{
		MEInvalidParameters(_DSFINDRDNATTR);
		return(NOTOK);
	}

/* Set the output parameter to a known value */

	(*pAttr) = NULL;

/* Get a name info structure from the certificate's subject */

	if ((*theCryptEnv->CryptDecodeObjectProc)(X509_ASN_ENCODING, X509_NAME,
		pCertContext->pCertInfo->Subject.pbData,
		pCertContext->pCertInfo->Subject.cbData,
		CRYPT_DECODE_NOCOPY_FLAG, NULL, &dwDataLen) != TRUE)
	{
		MELogOSErrorStr(GetLastError(), _DSFINDRDNATTR, NULL);
		return(NOTOK);
	}

	if ((NameInfo = (PCERT_NAME_INFO)cp_malloc(dwDataLen)) == NULL)
	{
		MENoMemory(_DSFINDRDNATTR);
		return(NOTOK);
	}

	if ((*theCryptEnv->CryptDecodeObjectProc)(X509_ASN_ENCODING, X509_NAME,
		pCertContext->pCertInfo->Subject.pbData,
		pCertContext->pCertInfo->Subject.cbData,
		CRYPT_DECODE_NOCOPY_FLAG, NameInfo, &dwDataLen) != TRUE)
	{
		MELogOSErrorStr(GetLastError(), _DSFINDRDNATTR, NULL);
		cp_free((charP)NameInfo);
		return(NOTOK);
	}

/* Find the desired attribute within the name info structure
	It is OK if the attribute doesn't exist. It just means that this
	certificate probably won't end up on the certificate list presented
	to the user (e.g. Issuer certificates often don't have Subjects) */

	if ((theAttribute = (*theCryptEnv->CertFindRDNAttrProc)(pszAttrId, NameInfo)) == NULL)
	{
		cp_free((charP)NameInfo);
		return(OK);
	}

/* Convert the attribute into a string */

	theLen = (*theCryptEnv->CertRDNValueToStrProc)(theAttribute->dwValueType,
		&theAttribute->Value, NULL, 0);

	if ((theValue = cp_malloc(theLen + 1)) == NULL)
	{
		cp_free((charP)NameInfo);
		return(NOTOK);
	}
	(void) (*theCryptEnv->CertRDNValueToStrProc)(theAttribute->dwValueType,
		&theAttribute->Value, theValue, theLen);

/* Finished */

	cp_free((charP)NameInfo);
	(*pAttr) = theValue;

	return(OK);
}

/**********************************************************************/
/* We want to locate the issuer certificate in one of the stores on
	the local machine.

        @param SignerCert UI - the certificate of the signer whose issuer we need
        @param hCryptProv UI - the handle to the CSP
        @param pStore UO - returns the handle to the store containing the
        			issuer cert (CA, MY or ROOT).
        @param pIssuerCert UO - returns handle to the issuer certificate.

	@access private
*/
/**********************************************************************/

short _DSGetIssuerCert(PCCERT_CONTEXT SignerCert,
		       HCRYPTPROV hCryptProv,
                       HCERTSTORE *pStore,
                       PCCERT_CONTEXT *pIssuerCert)
{
char *StoresToSearch[3] = {"CA","MY","ROOT"};
int  I, NumStoresToSearch=3;
HCERTSTORE		hStore;
PCCERT_CONTEXT		IssuerCert;
DWORD 			Dummy=0;

/* Verify the input parameters */

	if (pIssuerCert!=NULL) *pIssuerCert = NULL;
        if (pStore!=NULL) *pStore = NULL;

	if (SignerCert==NULL || hCryptProv==NULL || pIssuerCert==NULL)
	{
		MEInvalidParameters(_DSGETISSUERCERT);
		return(NOTOK);
	}

/* Search the standard stores for the issuer of the given signer certificate */

        for (I=0; I<NumStoresToSearch; I++)
        {
	     if ((hStore = (*theCryptEnv->CertOpenSystemStoreProc)(hCryptProv,
             							   StoresToSearch[I])) == NULL)
	     {
	          MELogOSErrorStr(GetLastError(), _DSGETISSUERCERT, NULL);
		  return(NOTOK);
	     }

	     IssuerCert = (*theCryptEnv->CertGetIssuerCertificateFromStoreProc)(hStore, SignerCert, NULL, &Dummy);

             if (IssuerCert != NULL)
             {
                 *pIssuerCert = IssuerCert;
                 *pStore = hStore;
                 return OK;
             }

	     (*theCryptEnv->CertCloseStoreProc)(hStore, CERT_CLOSE_STORE_CHECK_FLAG);
        }

/* No internal errors, but the output params are left NULL because we
	didn't find the issuer cert either */

	return OK;
}

/**********************************************************************/
/* This function performs several tests on the validity of the
	given certificate.  It's issuing certificate is located, and
        the issuer's signature on the certificate is tested.  Then,
        we check to see if the certificate is in the CRL of the issuing
        certificate.  Then, we check the time validity of the certificate.

        @param Cert UI - the certificate we wish to test
        @param hCryptProv UI - the handle to the CSP

	@access private
*/
/**********************************************************************/

short _DSCheckCertificate(PCCERT_CONTEXT Cert, HCRYPTPROV hCryptProv)
{
HCERTSTORE		theStore=NULL;
PCCERT_CONTEXT		IssuerCert=NULL;
DWORD			Flags;
short			result = OK;

/* Verify the input parameters */

	if (Cert==NULL || hCryptProv==NULL)
	{
		MEInvalidParameters(_DSCHECKCERTIFICATE);
		return(NOTOK);
	}

/* Start by explicitly checking the certificate's expiry date. */

	if ((*theCryptEnv->CertVerifyTimeValidityProc)(NULL, Cert->pCertInfo) != 0)
		result |= DS_CERTEXPIRED;

/* Get the issuer certificate */

        IssuerCert = NULL;
        if (_DSGetIssuerCert(Cert, hCryptProv, &theStore, &IssuerCert) != OK)
        {
	     MELogErrorInfoStr(0, _DSCHECKCERTIFICATE, ME_INTERNALFUNCTIONFAILED, NULL);
             return NOTOK;
        }

        if (IssuerCert == NULL)
            return result|DS_ISSUERCERTNOTFOUND;

/* Check the integrity of the certificate, including the issuer's signature
	the issuer's CRL, and the time validity on the cert. */

        Flags = CERT_STORE_SIGNATURE_FLAG | CERT_STORE_REVOCATION_FLAG
        				  | CERT_STORE_TIME_VALIDITY_FLAG;

        if ((*theCryptEnv->CertVerifySubjectCertificateContextProc)(Cert, IssuerCert, &Flags) != TRUE)
        {
		MELogOSErrorStr(GetLastError(), _DSCHECKCERTIFICATE, NULL);
	       	(void) (*theCryptEnv->CertFreeCertificateContextProc)(IssuerCert);
		(*theCryptEnv->CertCloseStoreProc)(theStore, CERT_CLOSE_STORE_CHECK_FLAG);
                return NOTOK;
        }

       	(void) (*theCryptEnv->CertFreeCertificateContextProc)(IssuerCert);
	(*theCryptEnv->CertCloseStoreProc)(theStore, CERT_CLOSE_STORE_CHECK_FLAG);

        if (Flags&CERT_STORE_SIGNATURE_FLAG)
		result |= DS_ISSUERSIGFAILED;
        else if (Flags&CERT_STORE_REVOCATION_FLAG && !(Flags&CERT_STORE_NO_CRL_FLAG))
                result |= DS_CERTREVOKED;
        else if (Flags&CERT_STORE_TIME_VALIDITY_FLAG)
                result |= DS_CERTEXPIRED;

/* Return the results of the tests */

	return result;
}

/**********************************************************************/
/* This function tests a given certificate context for whether or not it
	has the AT_SIGNATURE property in its key info.

	@param pCertContext UI - the cert. context to be tested

	@return N OK if the cert. has the AT_SIGNATURE property
                  NOTOK on error or if the cert. doesn't have the property

	@access private
*/
/**********************************************************************/

short _DSTestSignatureProperty(PCCERT_CONTEXT pCertContext)
{
short 			Result=OK;
/* This code is commented out because, although it is the documented way
	of identifying certificates containing signature keypairs, the major
        CAs are not setting this property.  They set AT_KEYEXCHANGE instead.

CRYPT_KEY_PROV_INFO	*pKeyInfo = NULL;
BOOL			fReturnFlag;
DWORD			dwSize;

	fReturnFlag = (*theCryptEnv->CertGetCertificateContextPropertyProc)(
                                 pCertContext, CERT_KEY_PROV_INFO_PROP_ID,
                                 NULL, &dwSize);

	if (FALSE == fReturnFlag)
        {
	    MELogErrorInfoStr(0, _DSTESTSIGNATUREPROPERTY,
                                 ME_INTERNALFUNCTIONFAILED, "Error Getting Key Info");
            return NOTOK;
        }

// Allocate memory for the Key Info structure

	pKeyInfo = (CRYPT_KEY_PROV_INFO*)malloc(dwSize);

	if (!pKeyInfo)
        {
	    MENoMemory(_DSTESTSIGNATUREPROPERTY);
            return NOTOK;
        }

// Get the key info structure

	fReturnFlag = (*theCryptEnv->CertGetCertificateContextPropertyProc)(
                                 pCertContext, CERT_KEY_PROV_INFO_PROP_ID,
                                 pKeyInfo, &dwSize);

// Does it have a signature key?

	Result = pKeyInfo->dwKeySpec == AT_SIGNATURE ? OK : NOTOK;
	free(pKeyInfo);
*/
        return Result;
}

/**********************************************************************/
/* This function enumerates all the certificates in the given
	store and select the one matching 'signer'.

	@param hCertStore UI - the certificate store to search

	@param certId UI - the person whose certificate we want to find

	@param ppCertContext UO - a pointer to where to store the
		certificate context

	@return N OK on success or an error code on failure

	@access private
*/
/**********************************************************************/

short _DSFindCertificate(HCRYPTPROV hCryptProv, HCERTSTORE hCertStore,
			 charP certId, PCCERT_CONTEXT *ppCertContext)
{
PCCERT_CONTEXT		pCertContext = NULL;
BOOL			fSignerObtained = FALSE;
charP			CertSubject = NULL;
short			theError;

/* Verify the input parameters */

	if (ppCertContext != NULL) *ppCertContext = NULL;

	if (hCertStore == NULL || certId == NULL || ppCertContext == NULL)
	{
		MEInvalidParameters(_DSFINDCERTIFICATE);
		return(NOTOK);
	}

/* Loop through all certificates in the given store */

	while (fSignerObtained == FALSE)
	{
		if ((pCertContext = (*theCryptEnv->CertEnumCertificatesInStoreProc)(hCertStore,
			pCertContext)) == NULL)
			break;
		else
		{

/* Do not use the method shown in the CryptoAPI, which uses the
	function CertGetCertificateContextProperty() to test whether
	the keyinfo is of type AT_SIGNATURE.  This does not work. */

			if ((theError = _DSGetSubjectFromCert(pCertContext,
				&CertSubject, NOTOK)) != OK)
			{
				MELogErrorInfoStr(theError, _DSFINDCERTIFICATE, ME_INTERNALFUNCTIONFAILED, NULL);
				(void) (*theCryptEnv->CertFreeCertificateContextProc)(pCertContext);
				return(theError);
			}

			if (cp_strcmp(certId, CertSubject) == OK &&
                            _DSCheckCertificate(pCertContext, hCryptProv)==OK &&
                            _DSTestSignatureProperty(pCertContext)==OK)
			    fSignerObtained = TRUE;

			cp_free(CertSubject);
			CertSubject = NULL;
		}
	}

/* Finished */

	if (fSignerObtained == FALSE)
	{
	     if (pCertContext != NULL)
		 (*theCryptEnv->CertFreeCertificateContextProc)(pCertContext);
	}
	else *ppCertContext = pCertContext;

	return OK;
}

/**********************************************************************/
/* We want to start at the signer certificate and work our way up the
	chain of issuance, storing each full certificate subject in a
	string, followed by two platform-dependent line delimiters.
	We start at the CA store, then move to the ROOT store to collect
	the full chain of issuance.

	@param pSignerCert UI - The certificate of the signer

	@param pVerifyArg UI - Provides the CSP handle on input, also
		carries the return value.

	@return N OK on success or an error code on failure

	@access private
*/
/**********************************************************************/

short _DSGetCertChain(PCCERT_CONTEXT pSignerCert, DSUserDataForVerify *pVerifyArg)
{
HCERTSTORE		hStore;
PCCERT_CONTEXT		IssuerCert;
PCCERT_CONTEXT		SubjectCert = NULL;
charP			IssuerName = NULL;
charP			CertChain;
charP			tempChain;
DWORD			Dummy = 0;
char			CertSeparator[5];
short			theError, NumIssuers;
char			*storeName = "CA";

/* Verify the input parameters */

	if (pSignerCert == NULL || pVerifyArg == NULL)
	{
		MEInvalidParameters(_DSGETCERTCHAIN);
		return(NOTOK);
	}

/* Start by setting the return result to NULL */

	pVerifyArg->CertChain = NULL;

/* Initialize the CertSeparator to two linefeeds */

	strcpy(CertSeparator, LINE_DELIMITER);
	strcat(CertSeparator, LINE_DELIMITER);

/* Add the signer certificate to the beginning of the list */

	if ((theError = _DSGetSubjectFromCert(pSignerCert, &CertChain, OK)) != OK)
	{
		MELogErrorInfoStr(theError, _DSGETCERTCHAIN, ME_INTERNALFUNCTIONFAILED, NULL);
		return(theError);
	}
	if (CertChain == NULL)
		return(OK);

/* Open the first store so we can look for the issuers */

	if ((hStore = (*theCryptEnv->CertOpenSystemStoreProc)(pVerifyArg->hCryptProv, storeName)) == NULL)
	{
		MELogOSErrorStr(GetLastError(), _DSGETCERTCHAIN, NULL);
		return(NOTOK);
	}

/* Loop through the chain of issuance */

	IssuerCert = pSignerCert;
        NumIssuers = 0;
	while (SubjectCert != IssuerCert)
	{
		SubjectCert = IssuerCert;
		IssuerCert = (*theCryptEnv->CertGetIssuerCertificateFromStoreProc)(hStore, SubjectCert,
                                                               NULL, &Dummy);

		if (IssuerCert == NULL)
		{
                     while (storeName != NULL && IssuerCert == NULL)
                     {
		         (*theCryptEnv->CertCloseStoreProc)(hStore, CERT_CLOSE_STORE_CHECK_FLAG);
                         hStore = NULL;

                         /* We either get the issuers from CA or we check
                         	MY if no issuers are in the CA store;
                                either way, we check ROOT afterward. */

                         if (strcmp(storeName, "CA")==OK)
                              storeName = NumIssuers==0 ? "MY" : "ROOT";
                         else if (strcmp(storeName, "MY")==OK)
                              storeName = "ROOT";
                         else storeName = NULL;

                         if (storeName == NULL) continue;

		         if ((hStore = (*theCryptEnv->CertOpenSystemStoreProc)(pVerifyArg->hCryptProv, storeName)) == NULL)
			 {
				MELogOSErrorStr(GetLastError(), _DSGETCERTCHAIN, NULL);
				cp_free(CertChain);
				return(NOTOK);
			 }
			 IssuerCert = (*theCryptEnv->CertGetIssuerCertificateFromStoreProc)(hStore,
				                    SubjectCert, NULL, &Dummy);
                     }
		}

		if (IssuerCert == NULL || SubjectCert == IssuerCert)
			break;

                /* If we find that the two certificates have the same serial
                	number, then we don't duplicate description in the
                        list.  Basically, this happens when a copy of a cert.
                        in one store is in the other.  CertGetIssuer...()
                        returns the copy so we can bridge the gap.  This is
                        weird in this case, but if you're checking for trust,
                        then this behavior can be helpful (though I think it's
                        just as easy to look for the subject cert. if you're
                        switching stores anyway).  Thus I would call this a
                        bug in the sense that it is an unnatural or unexpected
                        (and certainly undocumented) behavior. */

                if ((*theCryptEnv->CertCompareIntegerBlobProc)(&IssuerCert->pCertInfo->SerialNumber,
                                           &SubjectCert->pCertInfo->SerialNumber))
                    ;
                else
                {
                    NumIssuers++;

		    if ((theError = _DSGetSubjectFromCert(IssuerCert, &IssuerName, OK)) != OK)
		    {
		     	MELogErrorInfoStr(theError, _DSGETCERTCHAIN, ME_INTERNALFUNCTIONFAILED, NULL);
			cp_free(CertChain);
			(void) (*theCryptEnv->CertCloseStoreProc)(hStore, CERT_CLOSE_STORE_CHECK_FLAG);
			return(theError);
		    }

		    if ((tempChain = cp_realloc(CertChain,
                                                cp_strlen(CertChain)+cp_strlen(CertSeparator)+cp_strlen(IssuerName)+1,
			                        cp_strlen(CertChain)+1)) == NULL)
		    {
			MENoMemory(_DSGETCERTCHAIN);
			cp_free(CertChain);
			(void) (*theCryptEnv->CertCloseStoreProc)(hStore, CERT_CLOSE_STORE_CHECK_FLAG);
			return(NOTOK);
		    }
		    CertChain = tempChain;
		    cp_strcat(CertChain, CertSeparator);
		    cp_strcat(CertChain, IssuerName);
                    cp_free(IssuerName);
                    IssuerName=NULL;
                }

		if (SubjectCert != pSignerCert)
			(void) (*theCryptEnv->CertFreeCertificateContextProc)(SubjectCert);
		SubjectCert = NULL;
	}

/* Finished */

	pVerifyArg->CertChain = CertChain;

	if (SubjectCert != pSignerCert && SubjectCert != NULL)
		(void) (*theCryptEnv->CertFreeCertificateContextProc)(SubjectCert);

        if (hStore != NULL)
	    (void)(*theCryptEnv->CertCloseStoreProc)(hStore, CERT_CLOSE_STORE_CHECK_FLAG);

	return(OK);
}

/**********************************************************************/
/* This function attempts to acquire the context of a CSP.

	@param phProv LO - a pointer to where to store the
		new context

	@param pszContainer UI - the name of the key container
		to use. Pass NULL to use the default container
		for the CSP.

	@param pszProvider UI - the name of the CSP. Pass NULL
		to use the Microsoft Basic CSP.  We should never use the
                user's default CSP because this takes correctness out of
                the application's hands.  If the signer uses the enhanced
                provider by default, the validator won't be able to verify
                if the basic provider is the default.

	@param dwProvType NI - the type of CSP that we need.

	@param dwFlags NI - the flags that are needed to connect
		to this CSP.

	@return N OK on success or an error code on failure

	@access private
*/
/**********************************************************************/

short _DSCryptAcquireContext(HCRYPTPROV *phProv, LPSTR pszContainer,
	                     LPSTR pszProvider, DWORD dwFlags)
{
LPSTR csp = MS_DEF_PROV;
DWORD dwProvType = PROV_RSA_FULL;

/* Verify the input parameters */

	if (phProv == NULL)
	{
		MEInvalidParameters(_DSCRYPTACQUIRECONTEXT);
		return(NOTOK);
	}

        if (pszProvider != NULL)
        {
            csp = pszProvider;
            if (strncmp(csp, "csptype=", 8) == OK)
            {
                csp += 8;
                dwProvType = atoi(csp);
                while (isascii(*csp) && isdigit(*csp))
                     csp++;
                if (*csp == '\n' || *csp == ' ')
                     csp++;
            }
        }

/* Attempt to acquire the context */

	if ((*theCryptEnv->CryptAcquireContextProc)(phProv, pszContainer, csp,
		dwProvType, dwFlags) != TRUE)
	{
        DWORD   LastError = GetLastError();

        	/* Despite any compiler warnings, these constants are not out of range
                	and this comparison does work properly. */

		if (LastError != NTE_BAD_KEYSET)
		{
			MELogOSErrorStr(LastError, _DSCRYPTACQUIRECONTEXT, NULL);
			return(NOTOK);
		}

/* If the acquire fails because of a bad keyset, then perhaps the key set
	hasn't yet been created, so we'll try creating it here. If it truly
        was bad, then we'll fail again. */

		if ((*theCryptEnv->CryptAcquireContextProc)(phProv, pszContainer, csp,
			dwProvType, dwFlags | CRYPT_NEWKEYSET) != TRUE)
		{
			MELogOSErrorStr(GetLastError(), _DSCRYPTACQUIRECONTEXT , NULL);
			return(NOTOK);
		}
	}

/* Finished */

	return(OK);
}

/**********************************************************************/
/* A callback function to get a certificate from a store
	Used by DSVerifySignature(). The user data contains a location
        to store the actual error when getting the certificate.  Some of
        the codes can be OR'd together:

        DS_CERTNOTTRUSTED
        	if the certificate is marked as untrusted on the system
                (this is not implemented because it is application-specific)

	DS_CERTEXPIRED
		if the certificate's time validity check fails

	DS_CERTREVOKED
                if the certificate is in the issuer's CRL

	DS_ISSUERSIGFAILED
		if the issuer signature on the certificate fails

	DS_ISSUERCERTNOTFOUND
                if the issuer certificate not found on local machine

	DS_MSG_CERTNOTFOUND
		if the message certificate is not found

	DS_MSG_ALTERED
                Reserved to put the result of comparing the signature
                	hash to the message hash

	OK  if all tests passed

        The tests are run in order.  So, for example, if you get a time validity
        check failure, then the context tests succeeded.

	@param pvVerifyArg UI - the user supplied data

	@param dwCertEncodingType NI - the type of encoding used

	@param pSignerId UI - the signer information

	@param hMsgCertStore UI - the certificate store

	@return U the certificate context

	@access private
*/
/**********************************************************************/

PCCERT_CONTEXT WINAPI _DSGetVerificationCertificate(
	void *pvVerifyArg, DWORD dwCertEncodingType,
        PCERT_INFO pSignerId, HCERTSTORE hMsgCertStore)
{
DSUserDataForVerify	*pVerifyArg = (DSUserDataForVerify *) pvVerifyArg;
PCCERT_CONTEXT		msgSignerCert;
charP			CertSubject = NULL;
short			theError;
charP			CertInfo;

/* Set the result to a known value */

	if (pVerifyArg) pVerifyArg->result = OK;

/* Verify the input parameters */

	if (pVerifyArg == NULL || pSignerId == NULL || hMsgCertStore == NULL)
	{
		MEInvalidParameters(_DSGETVERIFICATIONCERTIFICATE);
		return(NULL);
	}

/* If we can't find the signer certificate in the message, then game over */

	if ((msgSignerCert = (*theCryptEnv->CertGetSubjectCertificateFromStoreProc)(hMsgCertStore,
		                       dwCertEncodingType, pSignerId)) == NULL)
	{
		if (GetLastError() == CRYPT_E_NOT_FOUND)
			pVerifyArg->result = DS_MSG_CERTNOTFOUND;
		else	MELogOSErrorStr(GetLastError(), _DSGETVERIFICATIONCERTIFICATE, NULL);
		return(NULL);
	}

/* We need to get the certificate subject */

	if ((theError = _DSGetSubjectFromCert(msgSignerCert, &CertSubject, NOTOK)) != OK)
	{
		MELogOSErrorStr(GetLastError(), _DSGETVERIFICATIONCERTIFICATE, NULL);
		(void) (*theCryptEnv->CertFreeCertificateContextProc)(msgSignerCert);
		return(NULL);
	}

	pVerifyArg->signer = CertSubject;

/* The certifying authority signed the message certificate (which is just a
	copy of the signer's personal certificate).  Using the issuer cert.
        appearing on the verifier's box, does the issuer signature on the
        signer certificate check out?  Or, is the signer cert. in the CRL of
        the issuing certificate? Finally, is the certificate expired? */

        theError = _DSCheckCertificate(msgSignerCert, pVerifyArg->hCryptProv);
        if (theError < OK)
        {
	     MELogErrorInfoStr(theError, _DSGETVERIFICATIONCERTIFICATE, ME_INTERNALFUNCTIONFAILED, NULL);
	     (void) (*theCryptEnv->CertFreeCertificateContextProc)(msgSignerCert);
	     return(NULL);
        }
        else if (theError > OK)
	     pVerifyArg->result |= theError;

/* Do we trust the message certificate? */

	if ((theError = _DSTrustSigner(pVerifyArg)) != OK)
		pVerifyArg->result |= DS_CERTNOTTRUSTED;

/* Get the certificate info */

        CertInfo = cp_malloc(200);
        if (CertInfo != NULL)
        {
        DWORD KeyLength;
        WORD Date, Time, Day, Month, Year;

            KeyLength = (*theCryptEnv->CertGetPublicKeyLengthProc)(X509_ASN_ENCODING, &msgSignerCert->pCertInfo->SubjectPublicKeyInfo);

            FileTimeToDosDateTime(&msgSignerCert->pCertInfo->NotAfter, &Date, &Time);

            Day = Date&31;
            Month = ((Date - Day)>>5)&15;
            Year = (Date>>9)+1980;

            sprintf(CertInfo, "Key length is %d%sNot after %02d/%02d/%04d",
                               KeyLength, LINE_DELIMITER, Month, Day, Year);
        }
        pVerifyArg->CertInfo = CertInfo;

/* Get the certificate chain */

	if ((theError = _DSGetCertChain(msgSignerCert, pVerifyArg)) != OK)
	{
	     MELogErrorInfoStr(theError, _DSGETVERIFICATIONCERTIFICATE, ME_INTERNALFUNCTIONFAILED, NULL);
	     (void) (*theCryptEnv->CertFreeCertificateContextProc)(msgSignerCert);
	     return(NULL);
	}

/* Even if there are problems, we still allow the verify test to run. */

	return(msgSignerCert);
}

/**********************************************************************/
/* This function should check whether the signer or the chain of issuance
	of the signer is trusted.  How trust is determined and managed is
        application-specific.

	@param pVerifyArg UI - contains useful stuff like the handle
		to the CSP

	@return N OK on trust, NOTOK on no trust

	@access private
*/
/**********************************************************************/

short _DSTrustSigner(DSUserDataForVerify *pVerifyArg)
{
	return OK;
}

