// --------------------------------------------------
// Handles for Active Expressions
//
// Mauricio De Simone
// mdesimon@interlog.com
//
// Copyright (C) 1998  Mauricio De Simone
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation version 2.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// --------------------------------------------------

//
// Base handle
//
class AeHandle
{
public:

  // Add thread to joining-list
  void addThread(Thread_t t) {
#if DEBUG_EE
    DEBUGOBJ dummy("AEHANDLE", this, "Adding thread to join list");
#endif

    _joinlist->enroll(t);
  }

  // Wait until all threads are done
  virtual ~AeHandle() {
#if DEBUG_EE
    DEBUGOBJ dummy("AEHANDLE", this, "Joining");
#endif
    if (_join) {
      ThreadEngine::instance()->join(*_joinlist);
      delete _joinlist; _joinlist = NULL;
    } else {
      ThreadEngine::instance()->joinWhenDestructing(_joinlist);
    }

    
  }

protected:
  // Creation only from derived classes
  AeHandle() :
    _join(TRUE)
  {
    _joinlist = new thread_group_t;
  }

  bool _join;
  
private:
  thread_group_t* _joinlist;

};

//
// Closed
//

class C_AeHandle :
  public AeHandle
{
public:
  C_AeHandle(const C_DfeNode& node, bool join = TRUE) :
    _node((C_DfeNode*)&node)
  {
    Synchronize _with(Instantiate_Mutex);
#if DISTMEM
    if (_node->where() == ProcBinding::whereever) {
      _node->where(ProcBinding::instance()->where());
    }
#endif

    _join = join;
    _node->instantiate(*_node, *this);
  }
  
protected:
  C_DfeNode* _node;
};

//
// Input
//

template <class T_IN>
class I_AeHandle :
  public AeHandle
{
public:
  I_AeHandle(const I_DfeNode<T_IN>& node) :
    _node((I_DfeNode<T_IN>*)&node)
  {
    Synchronize _with(Instantiate_Mutex);
#if DISTMEM
    if (_node->where() == ProcBinding::whereever) {
      _node->where(ProcBinding::instance()->where());
    }
#endif

    PqueueInstantiator<T_IN> loc(1, _node->num()
#if DISTMEM
      ,(_node->where())
#endif
    );
    _node->source(loc);
    pout.locator(loc);
    
    // instantiate node
    _node->instantiate(*_node,*this);
    pout.initialize();
  }

  // Make sure eoq token is placed on
  // queue when destroyed
  virtual ~I_AeHandle() {
    pout.eoq();
  }

  //
  // Receive input from environment
  I_AeHandle<T_IN>& operator << (const T_IN& arg)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("AEHANDLE", this, "operator <<");
#endif

    // enque 
    pout << arg;
    return *this;
  }

protected:
  I_DfeNode<T_IN>* _node;
  O_PqueueHandle<T_IN> pout;
};

//
// Output
//

template <class T_OUT>
class O_AeHandle :
  public AeHandle
{
public:
  O_AeHandle(const O_DfeNode<T_OUT>& node) :
    _node((O_DfeNode<T_OUT>*)&node)
  {
    Synchronize _with(Instantiate_Mutex);
#if DISTMEM
    if (_node->where() == ProcBinding::whereever) {
      _node->where(ProcBinding::instance()->where());
    }
#endif

    PqueueInstantiator<T_OUT> loc(_node->num(), 1
#if DISTMEM
      ,(CommunicationEngine::instance()->me())
#endif
    );
  
    _node->sink(loc);
    pin.locator(loc);

    // instantiate node
    _node->instantiate(*_node, *this);
    pin.initialize();
  }

  //
  // Allow cast to bool for eos checking
  operator bool() const {
    return pin;
  }

  // Give output to environment
  O_AeHandle<T_OUT>& operator >> (T_OUT& arg)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("AEHANDLE", this, "operator >>");
#endif

    // dequeue
    pin >> arg;
    return *this;    
  }

protected:
  O_DfeNode<T_OUT>* _node;
  I_PqueueHandle<T_OUT> pin;
};

//
// Input / Output
//

template <class T_IN, class T_OUT>
class IO_AeHandle :
  public AeHandle
{
public:
  IO_AeHandle(const IO_DfeNode<T_IN, T_OUT>& node) :
    _node((IO_DfeNode<T_IN, T_OUT>*)&node)
  {
    Synchronize _with(Instantiate_Mutex);
#if DISTMEM
    if (_node->where() == ProcBinding::whereever) {
      _node->where(ProcBinding::instance()->where());
    }
#endif
    PqueueInstantiator<T_IN> locin(1, _node->num()
#if DISTMEM
      ,(_node->where())
#endif
    );
    PqueueInstantiator<T_OUT> locout(_node->num(), 1
#if DISTMEM
      ,(CommunicationEngine::instance()->me())
#endif
    );

    _node->source(locin);
    pout.locator(locin);

    _node->sink(locout);
    pin.locator(locout);

    // instantiate Node
    _node->instantiate(*_node, *this);

    pin.initialize();
    pout.initialize();
  }

  //
  // Finish with queue
  void eoq() {
    pout.eoq();
  }

  //
  // Allow cast to bool for eos checking
  operator bool() {
    return pin;
  }

  //
  // Set eoq token in queue
  virtual ~IO_AeHandle() {
    //    pout.eoq();
  }
  
  // Receive input from environment
  IO_AeHandle<T_IN, T_OUT>& operator << (const T_IN& arg)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("AEHANDLE", this, "operator <<");
#endif

    // enqueue
    pout << arg;
    return *this;
  }
  
   // Receive input from environment
  IO_AeHandle<T_IN, T_OUT>& operator >> (T_OUT& arg)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("AEHANDLE", this, "operator >>");
#endif

    // dequeue
    pin >> arg;
    return *this;
  }

protected:
  IO_DfeNode<T_IN, T_OUT>* _node;
  O_PqueueHandle<T_IN> pout;
  I_PqueueHandle<T_OUT> pin;
};



