// ------------------------------------------------------------
// Cond
//
// Wrapper for Condition Variables
//
// Mauricio De Simone
// mdesimon@interlog.com
//
// Copyright (C) 1998  Mauricio De Simone
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation version 2.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// ------------------------------------------------------------

class Cond
{
public:

  //
  // Exceptions
  //
  
  class ExInit :
    public Exception
  {
  public:
    ExInit(char* file, int line) :
      Exception(file, line)
    {}
    void print(ostream& os) const {
      os << "Cond Init";
    }
  };

  class ExDestroy :
    public Exception
  {
  public:
    ExDestroy(char* file, int line) :
      Exception(file, line)
    {}
    void print(ostream& os) const {
      os << "Cond Destroy";
    }
  };

  class ExWait :
    public Exception
  {
  public:
    ExWait(char* file, int line) :
      Exception(file, line)
    {}
    void print(ostream& os) const {
      os << "Cond Wait";
    }
  };

  class ExSignal :
    public Exception
  {
  public:
    ExSignal(char* file, int line) :
      Exception(file, line)
    {}
    void print(ostream& os) const {
      os << "Cond Signal";
    }
  };


  //
  // Methods
  //


  //
  // Construct and Initialize
  //
  
  Cond(int value = FALSE) : // throw (ExInit) 
    _value(value)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Contruction");
#endif
    
#if PTHREADS
    if (pthread_cond_init(&_cv, NULL) != 0) {
      throw ExInit(__FILE__, __LINE__);
    }
#elif UITHREADS
    if (cond_init(&_cv, USYNC_THREAD, 0) != 0) {
      throw ExInit(__FILE__, __LINE__);
    }
#endif
  }


  //
  // Destroy
  //
  
  ~Cond() // throw (ExDestroy)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Destruction");
#endif
    
#if PTHREADS
    if (pthread_cond_destroy(&_cv) != 0) {
      throw ExDestroy(__FILE__, __LINE__);
    }
#elif UITHREADS
    if (cond_destroy(&_cv) != 0) {
      throw ExDestroy(__FILE__, __LINE__);
    }
#endif
  }


  //
  // retreive the variable value
  //

  int value() // throw()
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Value Retrieval");
#endif
    
    return _value;
  }

  
  //
  // Change state for the variable
  // make sure to signal other threads
  // that might be waiting
  //
  
  void value(int value, bool _signal = TRUE) // throw (ExSignal)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Value Change");
#endif
    
    _value = value;
    if (_signal) {
      signal();
    }
  }


  //
  // Wait for signal
  //
  
  void wait(Mutex& mutex) // throw (ExWait)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Wait");
#endif
    
#if PTHREADS
    if (pthread_cond_wait(&_cv, &mutex._t) != 0) {
      throw ExWait(__FILE__, __LINE__);
    }
#elif UITHREADS
    if (cond_wait(&_cv, &mutex._t) != 0) {
      throw ExWait(__FILE__, __LINE__);
    }
#endif
  }


  //
  // Signal thread waiting on variable
  //
  
  void signal() // throw (ExSignal)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Signal");
#endif
    
#if PTHREADS
    if (pthread_cond_signal(&_cv) != 0) {
      throw ExSignal(__FILE__, __LINE__);
    }
#elif UITHREADS
    if (cond_signal(&_cv) != 0) {
      throw ExSignal(__FILE__, __LINE__);
    }
#endif
  }

  //
  // Broadcast to threads waiting on variable
  //
  
  void broadcast() // throw (ExSignal)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("COND", this, "Broadcast");
#endif
    
#if PTHREADS
    if (pthread_cond_broadcast(&_cv) != 0) {
      throw ExSignal(__FILE__, __LINE__);
    }
#elif UITHREADS
    if (cond_broadcast(&_cv) != 0) {
      throw ExSignal(__FILE__, __LINE__);
    }
#endif
  }

protected:
#if PTHREADS
  pthread_cond_t _cv;
#elif UITHREADS
  cond_t _cv;
#endif
  
  int volatile _value;
};

