// ------------------------------------------------------------
// Socket wrapper
//
// Mauricio De Simone
// mdesimon@interlog.com
//
// Copyright (C) 1998  Mauricio De Simone
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation version 2.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// ------------------------------------------------------------

//
// Baseclass for socket wrapper
// public creator can use it as client conections to
// ports or services
// protected creators can use it in server conections
//
class Socket :
  public Channel
{
public:
  // Open service in a host
  Socket(char* hostname, char* service) :
    _fd(0)
  {
    open(hostname, service, 0);
  }

  // Open port in a host
  Socket(char* hostname, int portnumber) :
    _fd(0)
  {
    open(hostname, NULL, portnumber);
  }

  // Close the socket if opened
  virtual ~Socket()
  {
    if (_fd != 0)
      close(_fd);
  }

  // Open a TCP Conection.
  // Code taken and adapted from Stevens
  // "Unix Network Programming"
  void open(char* host, char* service, int port)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("SOCKET", this, "Open");
#endif
    
    int	resvport;
    unsigned long inaddr;
    char* host_err_str();
    struct servent* sp;
    struct hostent* hp;

    //
    // Initialize the server's Internet address structure.
    // We'll store the actual 4-byte Internet address and the
    // 2-byte port# below.
    //
    memset(&_srv_addr, 0, sizeof(_srv_addr));
    _srv_addr.sin_family = AF_INET;

    if (service != NULL) {
      if ( (sp = getservbyname(service, "tcp")) == NULL) {
        FAIL("tcp_open: unknown service: %s/tcp", service);
      }
      _servinfo = *sp;			/* structure copy */
      if (port > 0) {
        _srv_addr.sin_port = htons(port);
      } else {
        _srv_addr.sin_port = sp->s_port;
      }
    } else {
      if (port <= 0) {
        FAIL("tcp_open: must specify either service or port");
      }
      _srv_addr.sin_port = htons(port);
    }

    //
    // First try to convert the host name as a dotted-decimal number.
    // Only if that fails do we call gethostbyname().
    //

    if ( (inaddr = inet_addr(host)) != INADDR_NONE) {
      /* it's dotted-decimal */
      memcpy(&_srv_addr.sin_addr, &inaddr, sizeof(inaddr));
      _hostinfo.h_name = NULL;
    } else {
      if ( (hp = gethostbyname(host)) == NULL) {
        FAIL("tcp_open:");
      }
      _hostinfo = *hp;	/* found it by name, structure copy */
      memcpy(&_srv_addr.sin_addr, hp->h_addr,
             hp->h_length);
    }

    if (port >= 0) {
      if ( (_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        FAIL("tcp_open: can't create TCP socket");
      }
    } else if (port < 0) {
      resvport = IPPORT_RESERVED - 1;
      if ( (_fd = rresvport(&resvport)) < 0) { 
        FAIL("tcp_open: can't get a reserved TCP port");
      }
    }

    configure();
    
    //
    // Connect to the server.
    //
    if (connect(_fd, (struct sockaddr *) &_srv_addr,
                sizeof(_srv_addr)) < 0) {
      close(_fd);
      FAIL("tcp_open: can't connect to server");
    }
  }

  //
  // Configure the Socket
  //

  void configure()
  {
    int reuse = 1;
    if (setsockopt(_fd, SOL_SOCKET, SO_REUSEADDR, (char*)&reuse, sizeof(reuse)) < 0) {
      FAIL("Server", "Can't Configure socket", strerror(errno));
    }

    // Send immidiately
    int delay = 1;
    if (setsockopt(_fd, IPPROTO_TCP, TCP_NODELAY, (char*) &delay, sizeof(delay)) < 0) {
      FAIL("Server", "Can't Configure socket for No delay", strerror(errno));
    }
  }

  //
  // Construct from a socket descriptor
  // Hopefully is valid and ready
  Socket(int socket) :
    _fd(socket)
  {
    configure();
  }

  //
  // Put data to socket
  // Guess where I got this from ;-)
  virtual void send(const void * data, size_t size) 
  {
#if DEBUG_EE
    DEBUGOBJ dummy("SOCKET", this, "Send");
#endif

    assert(_fd != 0);
    size_t nleft, nwritten;
    char* ptr = (char*) data;
    nleft = size;
    while (nleft > 0) {
      nwritten = write(_fd, ptr, nleft);
      if (nwritten <= 0) {
        FAIL("Write error");
      }
      nleft -= nwritten;
      ptr   += nwritten;
    }
  }

  //
  // Get data from socket
  // Block until size bytes have been received
  virtual void receive(void* data, size_t size)
  {
#if DEBUG_EE
    DEBUGOBJ dummy("SOCKET", this, "Receive");
#endif

    assert(_fd != 0);
    size_t nleft, nread;
    char*   ptr = (char*) data;
    
    nleft = size;
    while (nleft > 0) {
      nread = read(_fd, ptr, nleft);
      if (nread < 0) {
        FAIL("Read error", strerror(errno));
      } else if (nread == 0) {
        // break; // Original Stevens implementation
        // Pool until ready
        // This is spining until size bytes are colected
        // ugly but it works :-O
      }

      nleft -= nread;
      ptr   += nread;
    }
  }

private:
  int _fd;
  sockaddr_in _srv_addr;
  servent _servinfo;
  hostent _hostinfo;
};





