// --------------------------------------------------
// Thread Manipulation Module
// MDS Threads and helpers
//
// Mauricio De Simone
// mdesimon@interlog.com
//
// Copyright (C) 1998  Mauricio De Simone
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation version 2.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// --------------------------------------------------

//
// Methods
//
Thread_t 
ThreadEngine::self() // throw ()
{
  Thread_t t;
#if PTHREADS
  t._t = pthread_self();
#elif UITHREADS
  t._t = thr_self();
#endif
  return t;
}
  
// Create thread
void
ThreadEngine::create_local(Thread_t& t, void *(*start_routine) (void *), void *arg) // throw (ExCreate)
{
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Creation");
#endif

  CreateThreadInfoLocal* info = new CreateThreadInfoLocal;
  info->method = start_routine;
  info->arg = arg;

#if PTHREADS
  if (pthread_create(&t._t, &thread_attr, ExceptionSafeThreadEntryPoint, info) != 0) {
    throw (ExCreate(__FILE__, __LINE__));
  }
#elif UITHREADS
  int ret;
  if ((ret = thr_create(NULL, AE_STACK_SIZE, ExceptionSafeThreadEntryPoint, info, THR_BOUND | THR_NEW_LWP, &t._t)) != 0) {
    MSG("RET", ret);
    throw (ExCreate(__FILE__, __LINE__));
  }
#endif
}
  
// Create thread
void
ThreadEngine::create_remote(Thread_t& t, void *(*start_routine) (void *), Sizeable* arg) // throw (ExCreate)
{
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Remote Creation");
#endif

#if DISTMEM
  if ((t._where == CommunicationEngine::instance()->me()) || (t._where == ProcBinding::whereever)) {
#endif
    create_local(t, start_routine, arg);
#if DISTMEM
  } else {
    CreateThreadInfoRemote* info = new CreateThreadInfoRemote;
    info->method = start_routine;
    info->arg = arg;

    // Create thread remotely
    ThreadRemoteCreate create(t);
    Channel* c = ProcMap::instance()->channel(t._where);
    *c << C_L << create << *info << C_U;
    create.wait();
  }
#endif
}

// Join with thread
void
ThreadEngine::join(const Thread_t& t) // throw (ExJoin)
{
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Join");
#endif
  
#if DISTMEM
  if ((t._where == CommunicationEngine::instance()->me()) || (t._where == ProcBinding::whereever)) {
#endif
#if PTHREADS
  if (pthread_join(t._t, NULL)) {
    throw (ExJoin(__FILE__, __LINE__));
  }
#elif UITHREADS
  if (thr_join(t._t, NULL, NULL)) {
    throw (ExJoin(__FILE__, __LINE__));
  }
#endif
#if DISTMEM
  } else {
    // Create thread remotely
    ThreadRemoteJoin join(t);
    Channel* c = ProcMap::instance()->channel(t._where);
    *c << join;
    join.wait();
  }
#endif

}

// Join with thread Group
void
ThreadEngine::join(thread_group_t& t) // throw (ExJoin)
{
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Group Join");
#endif

  while (!t.empty()) {
    join(t.back());
    t.pop_back();
  }
}

void 
ThreadEngine::joinWhenDestructing(thread_group_t* group)
{
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Adding to Destructor Joining List");
#endif

  joinAtDes.push_back(group);
}

void
ThreadEngine::join_detached()
{
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Joining Detached Threads");
#endif

  while (!joinAtDes.empty()) {
    thread_group_t* temp = joinAtDes.back();
    join(*temp);
    joinAtDes.pop_back();
    delete temp; temp = NULL;
  }
}
  



void
ThreadEngine::kill(Thread_t& t, int signal) // throw (ExKill)
{
#if PTHREADS
  if (pthread_kill(t._t, signal)) {
    throw (ExKill(__FILE__, __LINE__));
  }
#elif UITHREADS
  if (thr_kill(t._t, signal)) {
    throw (ExKill(__FILE__, __LINE__));
  }
#endif
}


void
ThreadEngine::setsigmask(sigset_t *set)
{
#if PTHREADS
  if (pthread_sigmask(SIG_SETMASK, set, NULL)) {
    throw (ExMask(__FILE__, __LINE__));
  }
#elif UITHREADS
  if (thr_sigsetmask(SIG_SETMASK, set, NULL)) {
    throw (ExMask(__FILE__, __LINE__));
  }
#endif
}



ThreadEngine::ThreadEngine() // throw (ExInit, ExDetach, ExStack) 
#if DISTMEM
  : _whereIam(-1)
#endif
{
  //
  // Initilalize thread package
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Initialization");
#endif
  
#if PTHREADS
  if (pthread_attr_init(&thread_attr)) {
    throw (ExInit(__FILE__, __LINE__));
  }

#if AIX  
  if (pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_UNDETACHED)) {
    throw (ExDetach(__FILE__, __LINE__));
  }
#endif

  //
  // Set stack size for threads
  if (pthread_attr_setstacksize (&thread_attr, AE_STACK_SIZE)) {
    throw (ExStack(__FILE__, __LINE__));
  }

#elif UITHREADS
  // No initialization needed
#endif
}

ThreadEngine::~ThreadEngine() // throw (ExDestroy)
{
  //
  // Terminate thread package
#if DEBUG_EE
  DEBUGOBJ dummy("THREAD", NULL, "Destruction");
#endif

  
#if PTHREADS
  if (pthread_attr_destroy(&thread_attr)) {
    throw (ExDestroy(__FILE__, __LINE__));
  }
#elif UITHREADS
  // No cleanup needed
#endif
}

