// PopupText.java

import java.applet.Applet;
import java.awt.*;

public abstract class PopupText extends Applet {
    public void init() {
        getParameters();
        setBackground(background);

        if (background.equals(foreground))
            foreground = Color.black;

        font = new Font(fontName, Font.PLAIN, fontSize);

        if (font == null)
            font = getFont();

        textBox = new TextBox(frame = getFrame(), boxWidth, boxHeight, background);
    }

    public void paint(Graphics g) {
        if (term == null)
            return;

        g.setFont(font);
        g.setColor(foreground);

        Dimension size = size();

        FontMetrics metrics = getFontMetrics(font);
        int width = metrics.stringWidth(term);
        int indent = width > size.width ? 0 : (size.width - width) / 2;

        baseline = size.height - metrics.getDescent();
        g.drawString(term, indent, baseline);

        for (int i = indent; i < indent + width; i += 4)
            g.drawLine(i, baseline + 1, i + 1, baseline + 1);
    }

    public boolean mouseDown(Event event, int x, int y) {
        if (event.clickCount == 1)
            if (textBox.isShowing())           // filter spurious clicks
                textBox.hide();
            else {
                textBox.setText(getText());
                Point location = frame.location();
                if (location.x == 0 && location.y == 0) {   // fix for Netscape Navigator bug
                    Dimension screenSize = frame.getToolkit().getScreenSize();
                    location.x = screenSize.width / 2;
                    location.y = screenSize.height / 2;
                }
                textBox.show(new Point(location.x + size().width / 2, location.y + baseline + 2));
            }
        return true;
    }

    public void stop() {
        if (textBox.isShowing())
            textBox.hide();
    }

    public String[][] getParameterInfo() {
        String[][] params = {
            {"background", "#RRGGBB", "Background color as HTML color triple"},
            {"font", "String", "Font for term"},
            {"fontsize", "int", "Font size for term"},
            {"term", "String", "Term to be explained"}
        };

        return params;
    }

    private void getParameters() {
        String value;

        if ((value = getParameter("popupwidth")) != null)
            try {
                int i = Integer.parseInt(value);
                if (i >= 0)
                    boxWidth = i;
            }
            catch (NumberFormatException e) {
        }

        if ((value = getParameter("popupheight")) != null)
            try {
                int i = Integer.parseInt(value);
                if (i >= 0)
                    boxHeight = i;
            }
            catch (NumberFormatException e) {
        }

        if ((value = getParameter("background")) != null) {
            Color color = triple2Color(value);
            if (color != null)
                background = color;
        }

        if ((value = getParameter("fontsize")) != null)
            try {
                int i = Integer.parseInt(value);
                if (i >= 0)
                    fontSize = i;
            }
            catch (NumberFormatException e) {
        }

        if ((value = getParameter("font")) != null)
            fontName = value;

        term = getParameter("term");
    }

    private Color triple2Color(String triple) {
        if (triple.startsWith("#") && triple.length() == 7)
            try {
                int i = Integer.parseInt(triple.substring(1), 16);
                if (i >= 0 && i <= 0xffffff)
                  return new Color(i);
            }
            catch (NumberFormatException e) {
            }

        return null;
    }

    private Frame getFrame() {
        Component parent = getParent();

        while (parent != null && !(parent instanceof Frame))
            parent = parent.getParent();

        return (Frame) parent;
    }

    protected abstract String[] getText();

    TextBox textBox;

    private Frame frame;
    private int baseline;

    private int boxWidth = 600;
    private int boxHeight = 400;

    private String term = "<term param undefined>";
    private String fontName = "TimesRoman";
    private int fontSize = 16;
    private Font font;

    private Color foreground = new Color(0, 128, 0);    // dark green
    private Color background = Color.lightGray;
}