// Palette.java

import java.applet.*;
import java.awt.*;

public class Palette extends Applet implements Runnable {
    public Palette() {}

    public void init() {
        size = size();
    }

    public void start() {
        if (selector == null)
            (selector = new Thread(this)).start();
    }

    public void stop() {
        if (selector != null) {
            selector.stop();
            selector = null;
        }
    }

    public void paint(Graphics g) {
        synchronized (colorLock) {
            for (int sqx = 0; sqx < NB_COLUMNS; ++sqx)
                for (int sqy = 0; sqy < NB_ROWS; ++sqy)
                    drawColorSquare(g, sqy * NB_COLUMNS + sqx);
            updateSelection();
        }
    }

    public boolean mouseDown(Event evt, int x, int y) {
        int sqx = x * NB_COLUMNS / size.width;
        int sqy = y * NB_ROWS / size.height;

        if (sqx >= 0 && sqx < NB_COLUMNS && sqy >= 0 && sqy < NB_ROWS)
            synchronized (colorLock) {
                color = colors[colorIndex = sqy * NB_COLUMNS + sqx];
                try {
                    colorLock.notifyAll();
                } catch (InterruptedException e) {
                }
            }

        return true;
    }

    public void run() {
        synchronized (colorLock) {
            for (;;) {
                updateSelection();

                try {
                    colorLock.wait();
                } catch (InterruptedException e) {
                }
            }
        }
    }

    public static Color getPaletteColor() {return color;}

    private void updateSelection() {
        Graphics g = getGraphics();
        drawColorSquare(g, selectedColor);          // remove old selection
        drawColorSquare(g, selectedColor = colorIndex);   // highlight new selection
        g.dispose();
    }

    private void drawColorSquare(Graphics g, int index) {
        // must synchronize on colorLock before calling this function
        int sqx = index % NB_COLUMNS;
        int sqy = index / NB_COLUMNS;

        int x1 = sqx * size.width / NB_COLUMNS;
        int x2 = (sqx + 1) * size.width / NB_COLUMNS;
        int y1 = sqy * size.height / NB_ROWS;
        int y2 = (sqy + 1) * size.height / NB_ROWS;

        g.setColor(colors[index]);
        g.fillRect(sqx * size.width / NB_COLUMNS, sqy * size.height / NB_ROWS, x2 - x1, y2 - y1);
        g.setColor(index == colorIndex ? Color.white : Color.black);
        g.drawRect(x1, y1, x2 - x1 - 1, y2 - y1 - 1);
    }

    private Dimension size;
    private Thread selector;
    private int selectedColor;

    private static final int NB_COLUMNS = 18;
    private static final int NB_ROWS = 12;

    private static Color[] colors = new Color[216]; // const

    private static Object colorLock = new Object();
    private static Color color;
    private static int colorIndex;  // index of color in colors array

    static {
        for (int i = 0; i < 216; ++i) {
            int red = i / 36 * 0x33;
            int green = i / 6 % 6 * 0x33;
            int blue = i % 6 * 0x33;
            colors[i] = new Color(red, green, blue);
        }
        color = colors[colorIndex = 0];
    }
}