/*----------------------------------------------------------------------
AMULAW.C

Copyright(c)1998 M.W.Davies.
Part of ISDNREC, see ISDNREC.C and ISNDREC.MAK for details.

AMULAW implements some functions for converting between ALaw and 16bit 
PCM format.  There are also some mu-Law functions here that I believe are
correct, but I don't have the facilities to test them.  A simple method
of generating the header for a 16bit PCM "WAV" file is included here. 
----------------------------------------------------------------------*/

#include <windows.h>


//Scale factors for amplification.
//PCM16   -32768 to +32767       /8  -> ALAW
//PCM8      -128 to +127         *32 -> ALAW
//ALAW     -4095 to +4095
//ULAW     -8191 to +8159

#define SCALE_P16_TO_ALAW(x)  ((x)/8)
#define SCALE_ALAW_TO_P16(x)  ((x)*8)
#define FUDGE  2      //Amplification factor (samples seem very quiet on my machine)


//WAVE file header for 16 bit PCM samples
//
typedef struct{
  char riff[4];           // "RIFF"
  DWORD filelen;			  //length of file (minus header so far)
  char wave[4];           // "WAVE"
  char  fmt[4];           // "fmt "
  DWORD hdrlen;			  // length of format header (wFormat to bitsPerSamp
  WORD  wFormatTag;		  //1
  WORD  wChannels;		  //1
  DWORD dwSamplesPerSec;  //8000
  DWORD dwAvgBytesPerSec; //8000
  WORD  blockAlign;		  //2
  WORD  bitsPerSample;	  //16
  char  data[4];          // "data"
  DWORD datalen;			  //length of sample area to follow
} WAVEFILE;


BYTE reverse( BYTE inb )
{
  BYTE outb=0, i;

  //reverse bit order within byte
  for(i=0;i<8;i++){
	 outb<<=1;
	 if(inb&1)outb = outb | 1;
	 inb>>=1;
  }

  return outb;
}


// Encode takes a signed 12-bit sound sample quantity and compresses it using 
// the A-Law companding technique.
//
BYTE encode(int amplitude)
{
  BYTE outval;
  unsigned int value;

  if( amplitude>=0 ){
	 outval = 0;
	 value = amplitude;
  }else{
	 outval = 0x80;
	 value = -amplitude;
  }

  if     ( value & 0x800 ) outval |= ((value>>7)&15) | 0x70;
  else if( value & 0x400 ) outval |= ((value>>6)&15) | 0x60;
  else if( value & 0x200 ) outval |= ((value>>5)&15) | 0x50;
  else if( value & 0x100 ) outval |= ((value>>4)&15) | 0x40;
  else if( value & 0x080 ) outval |= ((value>>3)&15) | 0x30;
  else if( value & 0x040 ) outval |= ((value>>2)&15) | 0x20;
  else if( value & 0x020 ) outval |= ((value>>1)&15) | 0x10;
  else                     outval |= (value>>1)&15;

  return outval ^ 0x55;   //For transmssion on ISDN, every other bit is inverted
}


// Happening returns a value to indicate (roughly) whether there is some sound
// incoming on the line, or whether there is silence.
char happening(BYTE sample, int alaw)
{
  BYTE seg;
  char ch;

  sample = reverse(sample);
  if(alaw)
	 sample ^= 0x55;
  else
	 sample ^= 0xFF;

  seg = (sample & 0x7F) >> 4;   
  if(seg<2) ch='.';          //silence
  else if(seg<5) ch='+';	  //some sound
  else ch='*';					  //loud noise

  return ch;
}


// Decode takes an A-Law sound byte and de-compresses it to a signed 12-bit  
// sound sample using the A-Law companding technique.
//
int decode(BYTE sample)
{
  int amplitude;
  BYTE absval,lower;

  sample ^= 0x55;		 //First invert every other bit (ISDN transmission standard)
  absval = (sample>>4) & 7;       //exponent
  lower  = (sample & 0x0F)<<1;    //lower 4 bits of value

  switch(absval){
	 case 0: amplitude =   lower|0x01;     break;
    case 1:	amplitude =   lower|0x21;     break;
    case 2:	amplitude =  (lower|0x21)<<1; break;
    case 3:	amplitude =  (lower|0x21)<<2; break;
    case 4:	amplitude =  (lower|0x21)<<3; break;
    case 5:	amplitude =  (lower|0x21)<<4; break;
    case 6:	amplitude =  (lower|0x21)<<5; break;
    case 7:	amplitude =  (lower|0x21)<<6; break;
  }

  if(sample & 0x80) amplitude = -amplitude;

  return amplitude;
}


// EncodeMu takes a signed 13-bit sound sample quantity and compresses it using 
// the mu-Law companding technique.
//
BYTE encodeMu(int amplitude)
{
  BYTE outval;
  unsigned int value;

  if( amplitude>=0 ){
	 outval = 0;
	 value = amplitude;
  }else{
	 outval = 0x80;
	 value = -amplitude;
  }

  if     ( value & 0x1000 ) outval |= ((value>>8)&15) | 0x70;
  else if( value &  0x800 ) outval |= ((value>>7)&15) | 0x60;
  else if( value &  0x400 ) outval |= ((value>>6)&15) | 0x50;
  else if( value &  0x200 ) outval |= ((value>>5)&15) | 0x40;
  else if( value &  0x100 ) outval |= ((value>>4)&15) | 0x30;
  else if( value &  0x080 ) outval |= ((value>>3)&15) | 0x20;
  else if( value &  0x040 ) outval |= ((value>>2)&15) | 0x10;
  else if( value &  0x020 ) outval |= ((value>>1)&15) | 0x10;

  return ~outval;     //For transmssion on ISDN, all bits are inverted
}


// DecodeMu takes an mu-Law sound byte and de-compresses it to a signed 13-bit  
// sound sample using the mu-Law companding technique.
//
int decodeMu(BYTE sample)
{
  int amplitude;
  BYTE absval,lower;

  sample = ~sample;	 //First invert all bits (ISDN transmission standard)
  absval = (sample>>4) & 7;       //exponent
  lower  = (sample & 0x0F)<<1;    //lower 4 bits of value

  switch(absval){
	 case 0: amplitude =   lower|0x21;     break;
    case 1:	amplitude =  (lower|0x21)<<1; break;
    case 2:	amplitude =  (lower|0x21)<<2; break;
    case 3:	amplitude =  (lower|0x21)<<3; break;
    case 4:	amplitude =  (lower|0x21)<<4; break;
    case 5:	amplitude =  (lower|0x21)<<5; break;
    case 6:	amplitude =  (lower|0x21)<<6; break;
    case 7:	amplitude =  (lower|0x21)<<7; break;
  }

  if(sample & 0x80) amplitude = -amplitude;

  return amplitude;
}

// 
//	a2mu converts one A-Law sample to mu-Law
//
BYTE a2mu(BYTE sample)
{
  BYTE seg;

  seg = sample & 0x70;
  if(seg)
	 sample -= 16;			 //move into next lower segment
  else
	 sample &= 0x80;		 //preserve only sign bit ->minimum value in mu-Law

  return sample;
}

//
// mu2a converts one mu-Law sample to A-Law
//
BYTE mu2a(BYTE sample)
{
  BYTE seg;

  seg = sample & 0x70;
  if(seg==0x70)
	 sample |= 0x7F;    //maximum value in A-Law encoding
  else
	 sample += 16;      //move up into next higher segment

  return sample;
}



//
//	PCMtoAlaw converts an array of 16bit PCM values into compressed
// 8bit A-Law samples.
//
void PCMtoAlaw(BYTE *buf, BYTE *outbuf, int length)
{
  BYTE alaw;
  int value;

  while(length>0){    //number of samples to process

	 value = SCALE_P16_TO_ALAW( (buf[1])<<8 + buf[0] );
	
    alaw = encode( value );         //PCM-16 samples are centred around 0.
	 *outbuf = reverse( alaw );      //bit order is back-to-front with CAPI

	 buf += 2;
	 ++outbuf;
    --length;
  }
}



//
// AlawtoPCM converts and array of 8bit A-Law samples into the equivalent
// 16bit PCM values.
//
void AlawtoPCM(BYTE *buf, BYTE *outbuf, int length)
{
  BYTE lo,hi,alaw;
  int val;

  while(length>0){	//number of samples to process

	 alaw = reverse( *buf );
	 val = decode( alaw );

	 val = FUDGE * SCALE_ALAW_TO_P16(val);
	 lo = val & 255;
	 hi = val >> 8;
	 *outbuf++ = lo;
	 *outbuf++ = hi;

    ++buf;
    --length;
  }
}



//
//	 SetupForWrite writes a dummy WAV header to position the file pointer
//  so that we can start writing samples.  When we know how many samples
//  there are, we can go back and fix up the header later.
//
void SetupForWrite(HANDLE ofd, int isWav)
{
  WAVEFILE w;
  DWORD written;

  // Write dummy header, then sound samples can be written to the file.
  // When we know how many samples there are, we can update the header
  // with the correct length values.
  if(isWav){
    memset(&w,0,sizeof(w));
    WriteFile(ofd,&w,sizeof(w),&written,NULL);
  }
}

//
// RewriteWAVHeader goes back to overwrite the dummy WAV header with real
// values once we have finished writing samples to the WAV file.
//
void RewriteWAVHeader(HANDLE ofd, DWORD byteCount)
{
  WAVEFILE w;
  DWORD written;

  SetFilePointer(ofd, 0l, 0l, FILE_BEGIN);  //move to start of file

  strncpy(w.riff, "RIFF", 4);
  strncpy(w.wave, "WAVE", 4);
  strncpy(w.fmt,  "fmt ", 4);
  strncpy(w.data, "data", 4);
  w.wFormatTag = 1;           //WAVE_FORMAT_PCM;
  w.wChannels = 1;            //MONO
  w.dwSamplesPerSec = 8000;   //8000 samples per second, not a standard Microsoft
  w.dwAvgBytesPerSec = 16000; //value, but seems to work OK.
  w.hdrlen = 16;
  w.bitsPerSample = 16;		   //two bytes per sample
  w.blockAlign = 2;
  w.datalen = byteCount;	  
  w.filelen = sizeof(w) - 8  + w.datalen;

  WriteFile(ofd,&w,sizeof(w),&written,NULL);
}

//
//	WriteSamples writes a buffer full A-Law sound samples to the disk. 
// It either writes raw A-Law samples, or PCM-16 samples depending on the
// current mode of ISDNREC.
//
DWORD WriteSamples(HANDLE ofd, BYTE *buffer, BYTE *outbuf, DWORD samples, int isWav)
{
  DWORD written;
  static DWORD totalBytes=0;

  if(isWav){
    AlawtoPCM(buffer, outbuf, samples);
	 WriteFile(ofd, outbuf, samples*2, &written, NULL);
  }else
	 WriteFile(ofd,buffer,samples,&written,NULL);

  totalBytes += written;

  return totalBytes;
}





