/*----------------------------------------------------------------------
ISDNREC.C

Copyright(c)1998 M.W.Davies.
Code compiled as a WIN32 console-mode application with Borland C++ 5.0,
see ISDNREC.MAK for details.
Program tested under Windows 95 and Windows NT 4.0; should run with any
suitable CAPI2032.DLL supplied with an ISDN adapter.

ISDNREC is the main module, containing the CAPI handling, Finite State
Machine and file handling.  It also depends on timer services from 
TIMER.C, and format conversion routines from AMULAW.C.   
----------------------------------------------------------------------*/

#include <windows.h>
#include <conio.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <mem.h>

#include "timer.h"
#include "amulaw.h"


#include "capi2032.h"             /* ISDN-Common-API WIN32 Prototypes    */
#include "capi20.h"               /* ISDN-Common-API structs and defines */



//Constants and Macros

#define INDICATION       0x200
#define SUCCESS         0x0000
#define QUEUE_FULL      0x1103
#define QUEUE_EMPTY     0x1104
#define CAPI_BUF_SIZE   256
#define MIN(x,y) ((x)<(y)?(x):(y))


#define FL_DONTRECORD 1		 //Flags set by command line arguments
#define FL_OGM        2
#define FL_LAW        4    

#define B1_TRANS_64K  2     //Flags for CAPI_GET_PROFILE
#define B2_TRANS      2
#define B3_TRANS      1

#define BUFLEN     32000    //Buffer for file handling
#define SAMPSIZE      8

// 
void DisconPhys(void);
void IncomingDisc(CAPI_MSG *);
void IssueListen(void);
BYTE NextMessageNum(void);
void PlayTone(int);
void ProcessCall(CAPI_MSG *);
void ProcessFSM(int, CAPI_MSG *);
void ProcessInfo(CAPI_MSG *);
WORD RecvData(CAPI_MSG *);
void SendOGMessage(void);
void Sound(WORD , BYTE *);
void StartRecord(void);
void StopRecord(void);
CAPI_MSG * WaitForConfirmation(void);
                       

// Structures and global variables for CAPI handling.
typedef struct PROFILE_STRUCT {
  WORD          Number;	       //Number of installed ISDN controllers (=cards) */
  WORD          Channels;	    //Number of "B" channels supported */
  DWORD         Global_Options;
  DWORD         B1_Protocols;
  DWORD         B2_Protocols;
  DWORD         B3_Protocols;
  DWORD			 spare[11];     //prevents things being overwritten
}API_PROFILE;

API_PROFILE profile_info;
CAPI_MSG *get_msg, put_msg;
WORD applId;
BYTE PLCIcode;
WORD NCCIcode;
DWORD cipValue;
WORD controller;
int aLaw;
int recording;
int byteCount=0;     //counts bytes written to RECORD file


//General globals
int  state;         //FSM state
int  finished;      //Flag for main loop
int  flags;			  //Command line arguments

//File handling variables
char inbuf[BUFLEN+1];
char outbuf[BUFLEN*2+1];

HANDLE fd,ofd;      //File handles for OGM and record files.
char ogmfile[20];	  //OGM filename
BYTE beepBuff[CAPI_BUF_SIZE];  //contains 1khz beep samples


// FINITE STATE MACHINE FOR ISDN HANDLING
// 
// Inputs to FSM
//
#define I_KICKOFF   0
#define I_CONNI     1
#define I_CONNACTI  2
#define I_CONNIB3   3
#define I_ALLACT    4
#define I_DISCI     5
#define I_DISCB3I   6
#define I_TIMER     7     //not used at present
#define I_DATA_R    8
#define I_DATA_I    9
#define I_DISCRQ    10

#define INPUTS 11

//
// Symbols for states
//
//             IDLE    LISTEN ACTING   PROTB3  ACTB3I  CONN    DISCB3  DISC

#define s0	0x00	   //IDLE,     CAPI not yet activated
#define s1	0x10     //LISTEN,   Listening for an incoming call
#define s2	0x20     //ACTING,   Waiting for activation signal (channel up)
#define s3	0x30	   //PROTB3,   Wait for activate of "B" channel protocol
#define s4  0x40     //ACTB3I    Wait for B3 activation signal (can now receive data)
#define s5	0x50	   //CONN,     "B" channel protocol up, channel fully activated
#define s6	0x60     //DISCB3,   In the process of tearing down "B" protocol
#define s7	0x70	   //DISC,     Tearing down "B" channel

#define STATES 8

// Macros for extracting the new-state/action from the array, below:
#define GETSTATE(x) ((x)>>4)
#define GETACTION(x) ((x)&0xF)




//
// The FSM itself.  
//	  Inputs are on the left, states left to right.  Each array element contains
//   a word with new state and action.  See "ProcessFSM" for definitions of the
//   actions.
//
WORD fsm[INPUTS][STATES]={

//             IDLE    LISTEN ACTING   PROTB3  ACTB3I  CONN    DISCB3  DISC
//		         s0	     s1		 s2      s3		  s4 	     s5      s6		s7
//--------------------------------------------------------------------------
/*KICKOFF  */{ s1+1,   s1+0,   s2+0,   s3+0,   s4+0,   s5+0,   s6+0,   s7+0,  },
/*CONNI    */{	s0+0,   s2+9,	 s2+0,   s3+0,   s4+0,   s5+0,   s6+0,   s7+0,  },
/*CONNACTI */{	s0+0,   s1+0,	 s3+2,   s3+0,   s4+0,   s5+0,   s6+0,   s7+0,  },
/*CONNIB3  */{	s0+0,   s1+0,	 s2+0,   s4+11,  s4+0,   s5+0,   s6+0,   s7+0,  },
/*ALLACT   */{	s0+0,   s1+0,	 s2+0,   s3+0,   s5+3,   s5+0,   s6+0,   s7+0,  },
/*DISCI    */{	s0+0,   s1+0,	 s0+12,  s0+12,  s0+12,  s0+12,  s0+12,  s0+12, },
/*DISCB3I  */{	s0+0,   s1+0,	 s2+0,   s3+0,   s7+10,  s7+10,  s7+10,  s7+10, },
/*TIMER    */{	s0+0,   s1+0,	 s2+0,   s7+8,   s7+8,   s5+13,  s6+0,   s7+0,  },
/*DATA_R   */{	s0+0,   s1+0,	 s2+0,   s3+0,   s4+0,   s5+5,   s6+0,   s7+0,  },
/*DATA_I   */{	s0+0,   s1+0,	 s2+0,   s3+0,   s4+6,   s5+6,   s6+0,   s7+0,  },
/*DISCRQ   */{	s0+0,   s1+0,	 s2+0,   s3+0,   s6+7,   s6+7,   s7+8,   s7+0,  },
											 
};



//
//	ProcessMessage is called when a CAPI_GET_MESSAGE returns an indication
// i.e., a real incoming event that needs to be dealt with.  This translates
// CAPI messages into the equivalen inputs for my FSM (see above).  CAPI messages
// I am not interested in, I do not process.
// 
void ProcessMessage( CAPI_MSG *msg )
{
  WORD messageType, indication;

  messageType = (WORD) (msg->header.command & ~CONFIRM);
  indication = (WORD) (msg->header.command & INDICATION);

  if( indication ){  //process indication from CAPI

    switch( messageType ){
		case _CONNECT_I:
		  ProcessFSM( I_CONNI, msg );     	//Incoming ISDN call
        break;

	   case _CONNECT_ACTIVE_I:   			
		  ProcessFSM( I_CONNACTI, msg ); 	//B channel connected to destination
        break;

	   case _CONNECT_B3_I:   			
		  ProcessFSM( I_CONNIB3, msg ); 	//B channel connected to destination
        break;

	   case _CONNECT_B3_ACTIVE_I:
		  ProcessFSM( I_ALLACT, msg );      //B channel fully active
        break;

	   case _DISCONNECT_I:			//incoming disconnect 
		  ProcessFSM( I_DISCI, msg );
        break;

	   case _DISCONNECT_B3_I:		//incoming B chan protocol disconnect
		  ProcessFSM( I_DISCB3I, msg );
		  break;

	   case _DATA_B3_I:				//incoming DATA
		  ProcessFSM( I_DATA_I, msg );
        break;

	   case _INFO_I:					//incoming INFO frame
		  ProcessInfo(msg);
		  break;

	 default:
		printf("Indication not processed for 0x%04X\n",messageType);
	 }

  }
}


//
// ProcessFSM applies the input signal to the FSM and executes the action
// defined for this input in this state.  Most inputs require process variables
// in the current CAPI message, so the message is passed in as a parameter.
//
void ProcessFSM(int input, CAPI_MSG *msg)
{
  int newstate,oldstate;
  int action,value,capi_error;
  
  value = fsm[input][state];
  newstate = GETSTATE(value);
  action = GETACTION(value);

//  printf(":ProcessFSM I(%d), S(%d), A(%d), N(%d)\n",input,state,action,newstate);

  oldstate = state;
  state = newstate;

  switch(action){
	 case 0: 
      break;

	 case 1:    //LISTEN_REQ must be issued.
      IssueListen();
      break;

    case 2:    //CONNECT_ACTIVE_I, B channel connected.
      //First respond to incoming INDICATION...
      msg->header.length  = sizeof(msg->header);
      msg->header.command |= RESPONSE;
      capi_error  = CAPI_PUT_MESSAGE( applId, msg );
      if(capi_error)
        printf("Capi_error=%04X in ProcessFSM a2\n",(WORD)capi_error);

	   printf("Channel open\n");

      SetMyTimer(40);  //It shouldn't take anywhere near this long to get
                       //fully connected from here.
      //Next operation is to wait for the CONNECT_B3_IND so that the
      //"B" channel protocol is running
      
      break;     

    case 3:    //CONNECT_B3_ACTIVE_I, B channel protocol is up.
		NCCIcode = msg->header.ncci;         //Store Network Control Connection Identifier
		//NCPI structure follows in info.struct

      SetMyTimer(0);    //fully active, stop connection timer.
      
		//Issue response to indication....
      msg->header.length  = sizeof(msg->header);
      msg->header.command |= RESPONSE;
      capi_error  =	  CAPI_PUT_MESSAGE ( applId, msg);
      if(capi_error)
        printf("Capi_error=%04X in ProcessFSM a3\n",(WORD)capi_error);
      
      printf("Data Channel is open\n");

		ProcessFSM( I_DATA_R, NULL );

      break;

    case 4:   
      break;
    
    case 5:    //Queue data, Wait for Confirmation.
		SendOGMessage();    //Outgoing Message
      PlayTone(1);        //Beep
		
      StartRecord();      //start sampling incoming data
      break;
    
    case 6:    //Process incoming data, Send Response
      RecvData(msg);   
      break;
    
    case 7:    //Tear down B protocol with DISCON_B3_R
      put_msg.header.appl_id    = applId;    //application Id from CAPI_REGISTER
	   put_msg.header.ncci       =  NCCIcode;  /* use NCCI to tear down */
	   put_msg.header.command    = _DISCONNECT_B3_R;
      put_msg.header.number     = NextMessageNum();
      put_msg.header.controller = (BYTE)controller;
      put_msg.header.plci       = PLCIcode;
	   put_msg.header.length     = sizeof(put_msg.header)+1; 
	   put_msg.info.disconnect_b3_req.structs[0] = 0;
	   capi_error  = CAPI_PUT_MESSAGE ( applId, &put_msg);
		WaitForConfirmation();
      if(capi_error)
        printf("Capi_error=%04X in ProcessFSM a7\n",(WORD)capi_error);

      break;
    
    case 8:    //Issue DISCON_REQ
		DisconPhys();
      break;
    
    case 9:    //Process Incoming Call Indication
      PLCIcode = msg->header.plci;   //store physical link identifier
												 //this identifies the physical connection

		//We could issue ALERT_R (to ensure that caller hears phone ringing)
		//but we skip that since we're going to pick the call up straight away

		ProcessCall(msg);
		  
      break;
    
    case 10:   //Issue RSP to DISCON_B3_I and wait for DISCON_IND
		msg->header.ncci    =  NCCIcode;  
	   msg->header.command |= RESPONSE;
	   msg->header.length  = sizeof(msg->header); 
	   capi_error  = CAPI_PUT_MESSAGE ( applId, msg);
      if(capi_error)
        printf("Capi_error=%04X in ProcessFSM e\n",(WORD)capi_error);

		printf("\nHangup\n");

		StopRecord();

      break;
    
    case 11:   //CONNECT_B3_IND
		{
        _CON_B3_RESP *conr;

		  //First store NCCI for later use.
 		  NCCIcode = msg->header.ncci;         //Store Network Control Connection Identifier

        //Respond to incoming INDICATION...
		  conr = & msg->info.connect_b3_res;
		  conr->Accept = 0;
		  conr->structs[0] = 0;  //No NCPI structure.
        msg->header.command |= RESPONSE;
		  msg->header.length = sizeof( msg->header )+sizeof(_CON_B3_RESP);
        capi_error  = CAPI_PUT_MESSAGE( applId, msg );
        if(capi_error)
          printf("Capi_error=%04X in ProcessFSM a11\n",(WORD)capi_error);

		}
      break;

	 case 12:   //Issue RSP to DISCON_IND
      IncomingDisc(msg);

      msg->header.command |= RESPONSE;
		msg->header.length = sizeof( msg->header )+1;
      capi_error  = CAPI_PUT_MESSAGE( applId, msg );
      if(capi_error)
        printf("Capi_error=%04X in ProcessFSM a12\n",(WORD)capi_error);

		++finished;
      break;

	 case 13:
      StopRecord();
		PlayTone(2);   //let the caller know the call is ending
		ProcessFSM( I_DISCRQ, NULL );
		break;
     
  default:
	 printf("FSM Error with ProcessFSM(%d) in state (%X)\n",input,oldstate);

  }
}


void StartRecord(void)
{
  if(flags&FL_DONTRECORD)
    printf("Recording is disabled\n");
  else{
    SetMyTimer(20);   //Record only 29 seconds for the message
    printf("Recording Message\n");
	 ++recording;
  }
}


void StopRecord(void)
{
  if(recording){
    SetMyTimer(0);   //In case timer is still running when this gets called
    recording=0;
    printf("\nRecording Completed\n");
  }
}


void PutNumber(BYTE len, BYTE *st)
{
  int i;

  ++st;   //step over length field
  while( (*st==0) || !(*st & 0x80) ){
   //search for type of number/numbering plan byte
	 len--;	st++;
  }
  len--; ++st;  //step over type/plan byte, so we're 
                //pointing at the telephone number digits
  for(i=0; i<len; i++)
	 putchar(*st++);      //digits are encoded in plain ASCII
}


//
//	ProcessCall extracts all the information from the CON_IND and displays
//	the caller's number and muLaw/aLaw mode.  Resp to CON_IND is generated.
//
void ProcessCall(CAPI_MSG *msg)
{
  _CON_INDP *coni;
  _CON_RESP *conr;
  DWORD capi_error;
  WORD cip;
  BYTE *field, len, *bp;

  coni = &msg->info.connect_ind;
  cip = coni->CIP_Value;
  switch(cip){
	 case 1:   printf("Incoming speech call\n"); break;
	 case 4:   printf("Incoming 3.1kHz audio call\n"); break;
    case 16:  printf("Incoming Telephony call\n"); break;
  default:;
  }

  //now find the Calling Party Number - this may be of interest to someone
  field = &coni->structs[0];

  printf("\tNumber Called ");
  len = *field; 
  if(len){
	 printf("= ");
	 PutNumber(len,field);   //Called party number
	 putchar('\n');
  }else
	 printf("not available\n");
  field += (len+1);         // step over Called Party Number

  printf("\tCaller's Number ");
  len = *field;
  if(len){
	 printf("= ");
	 PutNumber(len,field);   //Calling party number
	 putchar('\n');	       // step over Called Party Number
  }else
	 printf("not available\n");
  field += (len+1);

  len = *field;
  field += (len+1);   // step over Called Party Subaddress

  len = *field; 
  field += (len+1);   // step over Calling Party Subaddress

  len = *field; 
  if(len==3){   		//Now pointing at Bearer Capabilities (BC)
	 //Speech/3.1kHz call, A  Law is X0 90 A3
	 //Speech/3.1kHz call, mu Law is X0 90 A2

	 aLaw = (field[3] == 0xA3);   
	 if(aLaw)
      printf("\tA-Law speech\n");
	 else
      printf("\t-Law speech\n");

  }

  //Issue response to indication....
  msg->header.command |= RESPONSE;

  conr = &msg->info.connect_res;   //Cast this into CONNECT_I|RESPONSE structure
  conr->Accept = 0;      //Accept incoming call
  bp = &conr->structs[0];

  *bp++ = 9;      //Fill in B protocol structure, 9 bytes to follow 

  //Three words follow, giving B1, B2 and B3 protocol
  *bp++ = 1;  *bp++ = 0;   //B1 protocol (64 kbit/s byte framing)
  *bp++ = 1;  *bp++ = 0;   //B2 protocol (Transparent)
  *bp++ = 0;  *bp++ = 0;   //B3 protocol (Transparent)

  //Now three structs with B1,B2,B3 parameters (zero in this case).
  *bp++ = 0;   *bp++ = 0;  *bp++ = 0;   //length fields all zero

  *bp++ = 0;      //Connected number
  *bp++ = 0;      //Connected subaddress
  *bp++ = 0;      //Low Layer compatibility
  *bp++ = 0;      //Additional information elements

  msg->header.length = sizeof(msg->header) + (bp - (BYTE *)conr);
  capi_error  = CAPI_PUT_MESSAGE ( applId, msg);
  if(capi_error)
    printf("Capi_error=%04X in ProcessCall\n",(WORD)capi_error);
    
  printf("Answering Call...\n");  
}



//
//	IssueListen is called in order to declare which kinds of incoming calls
// we are interested in, and also to enrol our application to receive 
// progress messages from the ISDN card. 
//
void IssueListen(void)
{
  DWORD capi_error;

  put_msg.header.appl_id    = (WORD)applId;
  put_msg.header.number     = NextMessageNum();
  put_msg.header.controller = (BYTE) controller;  
  put_msg.header.plci       = 0;
  put_msg.header.ncci       = 0;
  put_msg.header.length     = sizeof(put_msg.header) + 
										sizeof(put_msg.info.listen_req) + 5;
  put_msg.header.command    = _LISTEN_R; 

  put_msg.info.listen_req.Info_Mask = 0x7F;   //We're interested in all sorts of progress diag messages
  put_msg.info.listen_req.CIP_Mask = 0x10012; //Accept Speech, 3.1khz audio, or Telephony calls
  put_msg.info.listen_req.structs[0] = 0;   //CIP_Mask 2
  put_msg.info.listen_req.structs[1] = 0;   //CIP_Mask 2
  put_msg.info.listen_req.structs[2] = 0;   //CIP_Mask 2
  put_msg.info.listen_req.structs[3] = 0;   //CIP_Mask 2
  put_msg.info.listen_req.structs[4] = 0;   //Calling Party Number
  put_msg.info.listen_req.structs[5] = 0;   //Calling Party Subaddress

  capi_error  = CAPI_PUT_MESSAGE ( applId, &put_msg );
  if(capi_error)
    printf("Capi_error=%04X in IssueListen\n",(WORD)capi_error);
  WaitForConfirmation();
}



// Initialise variable at start of program
//
void  Initialise(void)
{
  memset((BYTE*)&put_msg, 0, sizeof(CAPI_MSG));
  applId=0;      //Identifier of this program to CAPI
  PLCIcode=0;    //Identifier for physical connection, once made
  state=0;		  //zero FSM state
  finished=0;    //controls main loop in MAIN()
  controller=1;  //Assuming first (or only) BRI is always used
  cipValue=1;    //default setting = speech
  ogmfile[0]=0;  //outgoing filename (if any)
  flags=0;       //command line options
  aLaw=1;        //Default mode is A-Law, unless we are told different
  recording=0;   //We are writing incoming A-law data packets to a file
}


//
// Process INFO Indication from the CAPI interface.  Mainly we are interested
// in Call Progress messages, so we can be sure the call is going through.
// 
void ProcessInfo(CAPI_MSG *msg)
{
  _INF_INDP *info;
  DWORD capi_error;
  
  info = &msg->info.info_ind;     // INFO_IND variant of CAPI message.

  if( info->Number & 0x8000 ){
	 //Q.931 Message Type

    switch(info->Number & 0xFF){	 // Low 8-bits is the ISDN message type, 
											 // as defined by Q.931 signaling protocol.

	 	case 0x01: printf("ALERTING\n"); break;
      case 0x02: printf("CALL PROCEEDING\n"); break;
      case 0x03: printf("PROGRESS\n"); break;
      case 0x62: printf("FACILITY\n"); break;
      case 0x6E: printf("NOTIFY"); break;
      case 0x7B: printf("INFORMATION\n"); break;
      case 0x75: printf("STATUS ENQUIRY\n"); break;
      case 0x0D: printf("SETUP ACKNOWLEDGE\n"); break;
      case 0x7D: printf("STATUS\n"); break;
      
      default:  //don't care about the rest
	   ;
    }
  }

  //now issue response to INFO_IND
  //
  msg->header.length     = sizeof(msg->header);
  msg->header.plci       =  PLCIcode;
  msg->header.command  |= RESPONSE;
  capi_error  = CAPI_PUT_MESSAGE ( applId, msg);
  if(capi_error)
    printf("Capi_error=%04X in ProcessInfo\n",(WORD)capi_error);
}



//
//	SendOGMessage issues transmits the sound samples from our Alaw/muLaw file
// to the ISDN network.
//
//
void SendOGMessage()
{
  BYTE *bufa;
  DWORD bytesLeft, bytesToSend, inLength;
  
  printf("Sending Outgoing Message\n");

  do{

	 ReadFile(fd,inbuf,BUFLEN,&inLength,NULL);
    bufa = inbuf;
    bytesLeft = inLength;

    while(bytesLeft>0 &&!finished){

		bytesToSend=MIN( CAPI_BUF_SIZE, bytesLeft );
      Sound( bytesToSend ,bufa );
      bufa += bytesToSend;
      bytesLeft -= bytesToSend;

    }

  }while(inLength &&!finished);
}


//
// RecvData is called to process each incoming data packet passed up by CAPI.
//	Its job is to process data in the packet, and then send a RSP to CAPI.
//
WORD RecvData(CAPI_MSG *msg)
{
  DWORD capi_error;
  WORD length, numResponse=0;
  BYTE *data;

  data = (BYTE *)msg->info.data_b3_ind.Data;
  length = msg->info.data_b3_ind.Data_Length;

  if(recording){
    byteCount = WriteSamples(ofd, data, outbuf, length, !(flags&FL_LAW));
	 putchar( happening( *data, aLaw ) );
  }

  //now issue response
  msg->header.length     = sizeof(msg->header)+sizeof(_DAT_B3_RESP);
  msg->header.command    |= RESPONSE;
  msg->info.data_b3_res.Number =	msg->info.data_b3_ind.Number;
  capi_error  = CAPI_PUT_MESSAGE ( applId, msg );
  if(capi_error)
    printf("Capi_error=%04X in RecvData\n",(WORD)capi_error);

  return numResponse;
}


//
//DisconPhys issues DISCONNECT_REQ which should cause the ISDN call to hangup.
//
void DisconPhys(void)
{
  DWORD capi_error;

  // Shut down physical layer  
  put_msg.header.appl_id    = applId;    //application Id from CAPI_REGISTER
  put_msg.header.length     = sizeof(put_msg.header)+sizeof(_DIS_REQP);
  put_msg.header.number     = NextMessageNum();
  put_msg.header.controller = (BYTE) controller;
  put_msg.header.plci       = PLCIcode;
  put_msg.header.command    = _DISCONNECT_R;
  put_msg.info.disconnect_req.structs[0] = 0;
  capi_error  = CAPI_PUT_MESSAGE ( applId, &put_msg);
  if(capi_error)
    printf("Capi_error=%04X in DisconPhys\n",(WORD)capi_error);
  WaitForConfirmation();

  printf("TX DISCONNECT\n");
  ++finished;
}


//
//	PutCause puts the text interpretation of the Q.931 error message.
// The meanings of the codes are pretty much the same in EuroISDN, BTNR191, 
// AT&T 5ESS, etc. Some changes will be required for NI-1, and older National
// ISDN standards.
//
void PutCause(int val)
{
  switch(val){
    case 0x80: printf(" = 0, Normal Disconnect"); break;
    case 0x81: printf(" = 1, Unassigned (Unallocated number)"); break;
    case 0x82: printf(" = 2, No route to specified transit network"); break;
    case 0x83: printf(" = 3, No route to destination"); break;
    case 0x84: printf(" = 4, Channel unacceptable (BT interim)"); break;
    case 0x86: printf(" = 6, Channel unacceptable"); break;
    case 0x87: printf(" = 7, Call awarded and being delivered in an established channel"); break;
    case 0x90: printf(" = 16, Normal call clearing"); break;
    case 0x91: printf(" = 17, User busy"); break;
    case 0x92: printf(" = 18, No user responding"); break;
    case 0x93: printf(" = 19, User alerting, no answer"); break;
    case 0x95: printf(" = 21, Call rejected"); break;
    case 0x96: printf(" = 22, Number changed"); break;
    case 0x9A: printf(" = 26, Non-selected user clearing"); break;
    case 0x9B: printf(" = 27, Destination out of order"); break;
    case 0x9C: printf(" = 28, Invalid number format"); break;
    case 0x9D: printf(" = 29, Facility rejected"); break;
    case 0x9E: printf(" = 30, Response to STATUS ENQUIRY"); break;
    case 0x9F: printf(" = 31, Normal unspecified"); break;
    case 0xA2: printf(" = 34, No circuit/channel available"); break;
    case 0xA6: printf(" = 38, Network out of order"); break;
    case 0xA9: printf(" = 41, Temporary failure"); break;
    case 0xAA: printf(" = 42, Switching equipment congestion"); break;
    case 0xAB: printf(" = 43, Access information discarded"); break;
    case 0xAC: printf(" = 44, Requested circuit/channel not available"); break;
    case 0xAF: printf(" = 47, Resource unavailable, unspecified"); break;
    case 0xB1: printf(" = 49, Quality of service unavailable"); break;
    case 0xB2: printf(" = 50, Requested facility not subscribed"); break;
    case 0xB9: printf(" = 57, Bearer capability not authorized"); break;
    case 0xBA: printf(" = 58, Bearer capability not authorized"); break;
    case 0xBF: printf(" = 63, Service or option not available, unspecified"); break;
    case 0xC1: printf(" = 65, Bearer capability not implemented"); break;
    case 0xC2: printf(" = 66, Channel type not implemented"); break;
    case 0xC5: printf(" = 69, Requested facility not implemented"); break;
    case 0xC6: printf(" = 70, Only restricted information bearer capability available"); break;
    case 0xCF: printf(" = 79, Service or option not available, unspecified"); break;
    case 0xD1: printf(" = 81, Invalid call reference value"); break;
    case 0xD2: printf(" = 82, Identified channel does not exist"); break;
    case 0xD3: printf(" = 83, A suspended call exists, but this identity does not"); break;
    case 0xD4: printf(" = 84, Call identity in use"); break;
    case 0xD5: printf(" = 85, No call suspended"); break;
    case 0xD6: printf(" = 86, Call having the requested call identity has been cleared"); break;
    case 0xD7: printf(" = 87, Incompatible destination"); break;
    case 0xD8: printf(" = 88, Incompatible destination"); break;
    case 0xDA: printf(" = 90, Destination address missing or incomplete"); break;
    case 0xDB: printf(" = 91, Invalid transit network selection"); break;
    case 0xDF: printf(" = 95, Invalid message, unspecified"); break;
    case 0xE0: printf(" = 96, Mandatory Information Element is missing"); break;
    case 0xE1: printf(" = 97, Message type non-existent or not implemented"); break;
    case 0xE2: printf(" = 98, Message not compatible with call state, or not implemented"); break;
    case 0xE3: printf(" = 99, Information Element non-existent or not implemented"); break;
    case 0xE4: printf(" = 100, Invalid Information Element contents"); break;
    case 0xE5: printf(" = 101, Message not compatible with call state"); break;
    case 0xE6: printf(" = 102, Recovery on time expiry"); break;
    case 0xEF: printf(" = 111, Protocol Error, unspecified"); break;
    case 0xFF: printf(" = 127, Interworking, unspecified"); break;
  default: ;
  }
}


//IncomingDisc gets called when a DISCONNECT_IND is seen from CAPI.  It decodes any
//diagnostic codes, and forces the program to terminate.
//
void IncomingDisc(CAPI_MSG *msg)
{
  DWORD capi_error;
  WORD *reason, q931;

  printf("ISDN Disconnect: ");
  reason = (WORD *) &msg->info.disconnect_ind;
  switch(*reason){							 //Disconnect reasons reported by the
													 //ISDN card via CAPI.
    case 0x3301: printf("Layer 1 down\n"); break;
    case 0x3302: printf("Layer 2 down\n"); break;
    case 0x3303: printf("Layer 3 down\n"); break;
    case 0x3304: printf("Other App got call\n"); break;
  default:  
    if((*reason & 0xFF00) == 0x3400){   //Diagnostic code comes from the ISDN
													 //network; meaning according to ITU-T Q.931                          
		q931 = (WORD)(*reason & 0x00FF);
		if(q931==0x90)
		  printf("Normal\n");
		else{
        printf("Q.931 error 0x%02X",q931);
		  PutCause(q931);
		  putchar('\n');
		}
	 }
  }

  msg->header.length     =  sizeof(msg->header); 
  msg->header.command   |= RESPONSE;
  capi_error  = CAPI_PUT_MESSAGE ( applId, msg);
  if(capi_error)
    printf("Capi_error=%04X in IncomingDisc\n",(WORD)capi_error);

  ++finished; 
}



//
//	NextMessage generates unique message numbers for the messages we send
// using PUT_MESSAGE
//
BYTE NextMessageNum(void)
{
  static int msg=0;
  
  msg = (msg + 1)%256;
  return (BYTE)msg;
}

 

//
//	WaitForConfirmation loops around waiting for a confirmation to come in
// on the GET_MESSAGE queue.  We call this whenever we have issued a REQ and
// we want to make sure it is accepted.  Note that if we accidentally pick up
// up an incoming indication, we call ProcessMessage to handle it, but then 
// return to waiting for our CNF.
//
CAPI_MSG * WaitForConfirmation(void)
{
  DWORD capi_error;
  WORD messageType,confirm;
  CAPI_MSG *msg=NULL;

  do{   
    capi_error = CAPI_GET_MESSAGE( applId, (PVOID *)&get_msg );
    confirm = 0;
    if(capi_error==SUCCESS){
      messageType = get_msg->header.command;
      confirm = (WORD) (messageType & CONFIRM);
	   if(confirm) 
		  msg=get_msg;	  //we got a confirmation
		else
        ProcessMessage( get_msg );	 //new indication -> process it.
		
    }else if(capi_error=QUEUE_EMPTY)
      SleepEx(10,TRUE);           //10 ms delay
   
  }while( capi_error!=SUCCESS && !confirm);

  return msg;
}


//
//	ProcessEvent is continually called from the main loop in order to read
// incoming events from CAPI to feed the ISDN FSM.  When the queue is empty
// and no timer events are pending, a short sleep is done.
//
void ProcessEvent(void)
{
  DWORD capi_error;
         
  capi_error = CAPI_GET_MESSAGE( applId, (PVOID *) &get_msg );
  if(capi_error==SUCCESS)
    ProcessMessage( get_msg );
  else if( TimerExpired() )
    ProcessFSM( I_TIMER, NULL );
  else if(capi_error==QUEUE_EMPTY)  //queue empty    
    SleepEx(10,TRUE);               //10 ms delay
}


//
// CAPIVersion checks that the CAPI 2.0 API is loaded
//
int CAPIVersion(void)
{
  DWORD capi_error;
  DWORD CAPIMajor;
  DWORD CAPIMinor;
  DWORD ManufacturerMajor;
  DWORD ManufacturerMinor;
  int result=0;

  capi_error  =  CAPI_GET_VERSION (&CAPIMajor, &CAPIMinor,
			                          &ManufacturerMajor, &ManufacturerMinor);
  if(capi_error==SUCCESS && CAPIMajor == 0x02){
	 printf("Common-ISDN-API 2.0 detected\n");
	 ++result;
  }else  
    printf("Common-ISDN-API Version 2.0 not installed\n");
    
  return result;
}


// 
//	 CAPIProfile checks that the ISDN card currently installed is able
//  to handle the kind of calls we are interested in, i.e. 64K transparent
//  byte framed data.  This call is something of a formality, since this is
//  something even the dumbest ISDN cards can do.
//
int CAPIProfile(void)
{
  DWORD capi_error;
  int result=0;
    
  capi_error =  CAPI_GET_PROFILE( &profile_info, 1);
  if(capi_error==SUCCESS){    
    if ((profile_info.Global_Options == 1 )  &&
      (profile_info.B1_Protocols   & B1_TRANS_64K )  &&
      (profile_info.B2_Protocols   & B2_TRANS     )  &&
      (profile_info.B3_Protocols   & B3_TRANS     )  ){
      result++;
    }else
	   printf("Profile does not meet ISDNREC requirements\n");
  }else
	  printf("CAPI_GET_PROFILE failed\n");
       
  return result;
}


//
// CAPIRegister registers our application with CAPI and gets a unique 
// application ID, which is used when we send/receive CAPI messages.
// This mechanism allows many concurrent applications to share the
// same CAPI interface.
//
int CAPIRegister(void)
{
  int result=0;
  DWORD capi_error,appl;
  DWORD MessageBufferSize = 1024+(1024 * 1);  //estimate the buffer size we need.
															 //1024 + (1024 * logical connections)
  DWORD MaxLogicalConnection = 1;
  DWORD MaxBDataBlocks = 1024/CAPI_BUF_SIZE;
  DWORD MaxBDataLen = CAPI_BUF_SIZE;

  capi_error =  CAPI_REGISTER ( MessageBufferSize, MaxLogicalConnection,
                                  MaxBDataBlocks, MaxBDataLen, &appl);
  if(capi_error==SUCCESS){
    applId = (WORD) appl;                //save applId for later
	 ++result;
  }

  return result;
}


//
//	ProcessArgs does the command line parsing for the program....
//
//
int ProcessArgs(int argc, char **argv, int *flags)
{
  int val=0;
  char *st;
  int ch,slen;

  *flags=0;
  argv++;
  if(argc==1) ++val;

  while(--argc > 0){

	 st=*argv++;
	 if(*st=='-'){
		++st;
		ch = toupper(*st++);  slen = strlen(st);
		switch(ch){

		  //'O' option specifies alternative OGM filename
		  case 'O': if( slen ){	   //more chars in this argument
                    strcpy(ogmfile,st);
  						  *flags |= FL_OGM; 

				      }else if(argc>0){  //read next arg for characters
						  st = *argv++; --argc;
                    strcpy(ogmfile,st);
  						  *flags |= FL_OGM; 
						}
					   break;

		  //'L' option forces output to RECORD.LAW file
		  case 'L': *flags |= FL_LAW;
					   break;

		  //'D' option inhibits recording of messages
        case 'D': *flags |= FL_DONTRECORD;
					   break;
	
		default:
        ++val;
      }
	 }

  }

  return val;
}


//
//	Sound plays a series of A-Law samples into the open "B" channel. This
// is used for playing both the OGM message and the 1kHz beeps.
//
void Sound(WORD len, BYTE *data)
{
  int capi_error;

  put_msg.header.ncci       = NCCIcode;      /* use NCCI to send data*/
  put_msg.header.length     = sizeof(put_msg.header) + sizeof(_DAT_B3_REQP);
  put_msg.header.command    = _DATA_B3_R;                     
  put_msg.header.controller = (BYTE) controller;
  put_msg.header.plci       = PLCIcode;
  put_msg.info.data_b3_req.Data = (DWORD) data;
  put_msg.info.data_b3_req.Data_Length = len;
  put_msg.info.data_b3_req.Flags = 0;	

  do{

    put_msg.header.number     = NextMessageNum();
    capi_error = CAPI_PUT_MESSAGE ( applId, &put_msg);
    if(capi_error==QUEUE_FULL)
      //Send Queue is full, wait a bit then try again
      SleepEx(10,TRUE);
    else if(capi_error)
      printf("Capi_error=%04X in Sound\n",(WORD)capi_error);
    else
      WaitForConfirmation();

  }while(capi_error==QUEUE_FULL);
}


//
//	PlayTone builds an A-Law/Mu-Law encoded array of samples to generate
// a 1kHz tone, amd plays it down the "B" channel.
//
void PlayTone(int plays)
{
  int i;
  int onekhz[]={1008, 2361, 2361, 1008, -1008, -2361, -2361, -1008};
  BYTE lawSample[SAMPSIZE];	 //'n' sample burst of silence, companded
  BYTE lawSilence[SAMPSIZE];   //'n' sample burst of 1khz, companded 
  BYTE *bp;

  for(i=0;i<SAMPSIZE;i++){
    if( aLaw ){  //Compress 1khz tone using A-LAW

		lawSample[i] = reverse( encode( onekhz[i] ) );
		lawSilence[i] = reverse( encode( 0 ) );

	 }else{       //Compress 1khz tone using mu-LAW
		lawSample[i] = reverse( encodeMu( onekhz[i] ) );
		lawSilence[i] = reverse( encodeMu( 0 ) );
	 }
  }


  bp = beepBuff;
  for(i=0; i<(CAPI_BUF_SIZE/SAMPSIZE); i++){
	 memcpy( bp, lawSample, SAMPSIZE );
	 bp += SAMPSIZE;
  }

  do{

	 for(i=0;i<8;i++)
	   Sound( CAPI_BUF_SIZE, beepBuff );
	 plays--;

	 if(plays){
      //short pause before playing the tone again
      for(i=0; i<125; i++)	 
	     Sound( sizeof(lawSilence), lawSilence );
	 }

  }while(plays>0);
}

//
//	OpenFiles opens the OGM file for playing, and creates the RECORD file
// that will receive the caller's message.
//
int OpenFiles(void)
{
  int record=0, play=0;
  char ofilename[20], filename[20];

  if(flags & FL_LAW)
    strcpy(ofilename,"RECORD.LAW");    //raw sample file, direct A-Law samples
  else
    strcpy(ofilename,"RECORD.WAV");    //16bit PCM WAV file for output

  if(flags & FL_DONTRECORD)
	 play++;
  else{
    ofd = CreateFile(ofilename,GENERIC_WRITE,0l, NULL, CREATE_ALWAYS, 
  					          FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN, NULL);
    if(ofd==INVALID_HANDLE_VALUE)
  	   printf("Failed to create file \"%s\"\n",ofilename);
    else{
      play++;
      SetupForWrite(ofd, !(flags&FL_LAW));
	 }
  }

  if(flags & FL_OGM)
    strcpy( filename, ogmfile );
  else
    strcpy( filename, "PLAY.LAW" );
    
  fd = CreateFile(filename,GENERIC_READ,0l, NULL, OPEN_EXISTING, 
                     FILE_FLAG_SEQUENTIAL_SCAN, NULL);
  if(fd==HFILE_ERROR)
    printf("Failed to open file \"%s\"\n",filename);
  else
    record++;

  return play && record;
}


void CloseFiles(void)
{
  if(!(flags&FL_LAW))
	 RewriteWAVHeader(ofd, byteCount);

  CloseHandle(fd);
  if(!(flags & FL_DONTRECORD))
    CloseHandle(ofd);
  
}



void main(int argc, char **argv)
{

  printf("----- ISDNREC version 1.0 -----\n");
  Initialise();
  StartTimerThread();

  ProcessArgs(argc,argv,&flags);

  if( CAPIVersion() ){
    if( CAPIProfile() && CAPIRegister() ){  // get the profile of card and register
														  // application with CAPI.	
      if( OpenFiles() ){
      
        ProcessFSM( I_KICKOFF, NULL );   //fire up the FSM
	     printf("Waiting for incoming call...\n");

        while( !finished ){
          ProcessEvent();
 	     }

        CloseFiles();
        
      }  
      CAPI_RELEASE (applId);	   /* release the application handle */
	 }
  }

  SetMyTimer(0);
  KillTimerThread();
  
}



