/*----------------------------------------------------------------------
TIMER.C

Copyright(c)1998 M.W.Davies.
Part of ISDNREC, see ISDNREC.C and ISNDREC.MAK for details.

This rather elaborate scheme (of creating a thread that implements the
timer with SleepEx) is used because the SetTimer() did not appear to have
any effect for a console applications; the SetTimer() call returned OK, 
but the handler never got called.  
----------------------------------------------------------------------*/

#include <windows.h>


LPTHREAD_START_ROUTINE threadFuncP;
CRITICAL_SECTION t_cs;
HANDLE timerThread;
long t_countDown, t_expired, t_activated;

//
//	SetMyTimer starts a countdown from the number of seconds specifed.
// When it reaches zero, t_expired becomes true.  SetMyTimer(0) will 
// stop the timer.
//
void SetMyTimer(long timerval)
{
  EnterCriticalSection( &t_cs );
  t_countDown=timerval;
  LeaveCriticalSection( &t_cs );
}


//
// TimerExpired is a function used by the main thread to check whether
// the timer event set off by SetMyTimer() happened.
//
long TimerExpired(void)
{
  long rc;

  EnterCriticalSection( &t_cs );
  rc=t_expired;
  t_expired=0l;
  LeaveCriticalSection( &t_cs );

  return rc;
}


//
//	TimerThread is the main loop of the thread, implementing the waiting
// and timer countdown.
//
void TimerThread(void)
{

  EnterCriticalSection( &t_cs );
  t_countDown = 0l;
  LeaveCriticalSection( &t_cs );
  t_expired = 0l;
  t_activated = 1l;

  while(t_activated){

	 SleepEx(1000,FALSE);

	 EnterCriticalSection( &t_cs );

	 if(t_countDown>0){  
		t_countDown--;
		if(t_countDown==0){
		  t_expired=1l;
		}
	 }

	 LeaveCriticalSection( &t_cs );

  }

  CloseHandle( timerThread );
  ExitThread(0);

}


//
//	StartTimerThread creates the timer thread and sets up TimerThread()
// as the main loop.
//
void StartTimerThread(void)
{
  DWORD id;

  threadFuncP = TimerThread;
  InitializeCriticalSection( &t_cs );
  timerThread = CreateThread( NULL, 1024, threadFuncP, NULL, 0, &id);
}


//
//	KillTimerThread gets TimerThread() to terminate, and picks up the
// return status.
//
void KillTimerThread(void)
{
  long result,status;

  t_activated=0l;
  do{
    result = GetExitCodeThread( timerThread, &status );
  }while( result && status == STILL_ACTIVE );  

  DeleteCriticalSection( &t_cs );
  CloseHandle( timerThread );
 
}
