
/*
 *	main.c
 *
 *	QNX 4, main.c shell version 0.005
 *
 *	(C) Copyright 1997 by Robert Krten, all rights reserved.
 *
 *	This module represents the main module for the X-10 client program. 
 *
 *	This program will send IPC messages to the X-10 server.
 *
 *	1997 01 12	R. Krten		created
*/

#ifdef __USAGE

%C [options] argument

where [options] are optional parameters chosen from:
    -n name        use this X-10 server (default /PARSE/X10)
    -v             verbose operation
Argument consists of the following:
    house# { <keynum> | command }

Where:
    house# is selected from the range a..p (or A..P)
    <keynum> is a number from 1 to 16
    command is one of the following:
        auoff      all units off
        alon       all lights on
        on         on
        off        off
        dim        dim
        bright     bright
        aloff      all lights off
        exc        extended code
        harq       hail request
        haac       hail acknowledge
        psdl       preset dim low
        psdh       preset dim high
        exd        extended data
        son        status = on
        soff       status = off
        srq        status request

This is the control program to be used with the X-10 server.
Its job is to send a message to the X-10 server, consisting
of a bunch of X-10 formatted commands to be sent.

For example, to turn on a light in a house, assume that the
X-10 controller for that light is set to house code C and unit
number 5:

	x c 5 on

This actually formats up a data stream consisting of:

    <header><housecode c><keycode 5>
    <header><housecode c><keycode 5>
    <header><housecode c><on>
    <header><housecode c><on>

Note that everything is repeated twice as per the X-10 protocol.

#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/kernel.h>
#include <sys/name.h>

#include "x10.h"

#define	NumCommands		32

static	void	optproc (int, char **);

char	*progname = "x";
char	*blankname= " ";
extern	char	*version;					// version.c

int		house;
int		key;
int		cmd;
char	*optn;
int		optv;								// verbose operation

static	char	houseTranslate [16] =
{
	0x06,	0x0e,	0x02,	0x0a,
	0x01,	0x09,	0x05,	0x0d,
	0x07,	0x0f,	0x03,	0x0b,
	0x00,	0x08,	0x04,	0x0c
};

static	char	*commands [NumCommands] =
{
	{"Key 13"},	{"auoff"},	{"Key 5"},	{"alon"},
	{"Key 3"},	{"on"},		{"Key 11"},	{"off"},
	{"Key 15"},	{"dim"},	{"Key 7"},	{"bright"},
	{"Key 1"},	{"aloff"},	{"Key 9"},	{"exc"},
	{"Key 14"},	{"harq"},	{"Key 6"},	{"haac"},
	{"Key 4"},	{"psdl"},	{"Key 12"},	{"psdh"},
	{"Key 16"},	{"exd"},	{"Key 8"},	{"son"},
	{"Key 2"},	{"soff"},	{"Key 10"},	{"srq"}
};

static	char	keyTranslate [16] =
{
	0x0c,	0x1c,	0x04,	0x14,
	0x02,	0x12,	0x0a,	0x1a,
	0x0e,	0x1e,	0x06,	0x16,
	0x00,	0x10,	0x08,	0x18
};

main (argc, argv)
int		argc;
char	**argv;
{
	optproc (argc, argv);
}

void
usageError ()
{
	fprintf (stderr, "%s:  error in use, type \"use %s\" for usage\n", progname, progname);
	exit (1);
}

void
optproc (argc, argv)
int		argc;
char	**argv;
{
	int		opt;
	char	*arg;
	int		i;

	if (!argc) {
		usageError ();
	}

	optn = NULL;
	house = -1;
	key = -1;
	cmd = -1;

	while ((opt = getopt (argc, argv, "n:v")) != -1) {
		switch (opt) {
		case	'n':
			optn = optarg;
			break;
		case	'v':
			optv++;
			if (optv > 1) {
				printf ("Verbosity is %d\n", optv);
			}
			break;
		default:
			usageError ();
			break;
		}
	}

	if (optn == NULL) {
		optn = "/PARSE/X10";
	}

	for (; optind < argc; optind++) {
		arg = argv [optind];
		if (arg [1] == 0) {		// a single character
			if (('a' <= arg [0] && arg [0] <= 'p')
			||  ('A' <= arg [0] && arg [0] <= 'P')) {
				house = houseTranslate [(arg [0] & 0x0f) - 1];
			} else if ('0' <= arg [0] && arg [0] <= '9') {
				key = keyTranslate [(atoi (arg) - 1) & 0x0f];
			}
		} else {
			if (arg [2] == 0 && isdigit (arg [1]) && isdigit (arg [0])) {
				key = keyTranslate [(atoi (arg) - 1) & 0x0f];
			}
			for (i = 0; i < NumCommands; i++) {
				if (!strcmp (commands [i], arg)) {
					break;
				}
			}
			if (i == NumCommands) {
				fprintf (stderr, "%s:  unknown keyword \"%s\" on line\n", progname, arg);
				exit (1);
			}
			cmd = i;
		}
	}
	if (optv) {
		printf ("Got house %d key %d cmd %d\n", house, key, cmd);
	}
	formatAndSendRequest (house, key, cmd);
}

formatAndSendRequest (h, k, c)
int		h;
int		k;
int		c;
{
	IPC_t	ipc;
	int		pid;

	ipc.type = X10_Type;
	ipc.subtype = X10_TX;

	strcpy (ipc.buffer, "001110");
	appendTCBits (ipc.buffer, 4, h);
	appendTCBits (ipc.buffer, 5, k);
	strcat (ipc.buffer, "000000");
	strcat (ipc.buffer, "1110");
	appendTCBits (ipc.buffer, 4, h);
	appendTCBits (ipc.buffer, 5, k);
	strcat (ipc.buffer, "000000");
	strcat (ipc.buffer, "1110");
	appendTCBits (ipc.buffer, 4, h);
	appendTCBits (ipc.buffer, 5, c);
	strcat (ipc.buffer, "000000");
	strcat (ipc.buffer, "1110");
	appendTCBits (ipc.buffer, 4, h);
	appendTCBits (ipc.buffer, 5, c);
	if (optv) {
		printf ("Sending \"%s\"\n", ipc.buffer);
	}
	
	if ((pid = qnx_name_locate (0, optn, sizeof (ipc), NULL)) == -1) {
		fprintf (stderr, "%s:  unable to locate X-10 server \"%s\"\n", progname, optn);
		perror (NULL);
		exit (1);
	}
	if (Send (pid, &ipc, NULL, sizeof (ipc), 0)) {
		fprintf (stderr, "%s:  error while sending to X-10 server\n", progname);
		perror (NULL);
		exit (1);
	}
}

static char bitmask [8] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};

appendTCBits (buf, nbits, i)
char	*buf;
int		nbits;
int		i;
{
	int		p;

	for (p = nbits - 1; p >= 0; p--) {
		strcat (buf, i & bitmask [p] ? "1" : "0");
		strcat (buf, i & bitmask [p] ? "0" : "1");
	}
}

