
/*
 *	decode.c
 *
 *	QNX 4
 *
 *	(C) Copyright 1997 by Robert Krten, all rights reserved.
 *
 *	This module contains a finite state machine to decode the
 *	X-10 protocol on a bit-by-bit basis.
 *
 *	1997 01 12	R. Krten		created
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include "x10.h"

extern	char	*progname;					// main.c
extern	char	*version;					// version.c

extern	int		optv;						// main.c, verbose operation flag

//	Major states

#define	S_Reset					0			// reset state
#define	S_MatchSignature		1			// matching signature array
#define	S_AccumulatingHouse		2			// matched 1110, looking for housecode
#define	S_AccumulatingKeycodes	3			// got a housecode, decode the function

//	Substates
#define	SS_AccumulateTrue		100			// subroutine to accumulate true/complement bits
#define	SS_AccumulateComplement	101			// other half

static	int		state = S_Reset;
static	int		prevstate = S_Reset;		// 2 level deep state stack
static	char	signature [4] = { 1, 1, 1, 0 };
static	char	houseCodes [16] = { "MECKOGAINFDLPHBJ" };

char	*keyCodes [32] =
{
	{"Key 13"},	{"All units off"},	{"Key 5"},	{"All lights on"},
	{"Key 3"},	{"On"},				{"Key 11"},	{"Off"},
	{"Key 15"},	{"Dim"},			{"Key 7"},	{"Bright"},
	{"Key 1"},	{"All lights off"},	{"Key 9"},	{"Extended Code"},
	{"Key 14"},	{"Hail Request"},	{"Key 6"},	{"Hail Acknowledge"},
	{"Key 4"},	{"Preset Dim (L)"},	{"Key 12"},	{"Preset Dim (H)"},
	{"Key 16"},	{"Extended Data"},	{"Key 8"},	{"Status = on"},
	{"Key 2"},	{"Status = off"},	{"Key 10"},	{"Status Request"}
};

static	int		matchIndex;					// where we are in a multi-bit match
static	char	*matchArray;				// what we are matching (signature, houseCodes, etc)
static	int		matchSize;					// its size
static	char	buffer [16];				// holding / accumulating buffer
static	int		accumulateSize;				// number of bits to accumulate
static	int		accumulateIndex;			// where we are in the accumulation
static	char	*accumulateBuffer;			// where to accumulate into

X10Decode_t		x10Buffer;					// place for decodes to go

void
initFSM ()
{
	state = S_Reset;
}

void
decode (n)
int		n;
{
	int		val;

	switch (state) {
	case	S_Reset:						// Reset looks for 1110 to transit to Signature
		if (n) {
			matchIndex = 1;
			matchSize = 4;
			matchArray = signature;
			state = S_MatchSignature;
		}
		break;
	case	S_MatchSignature:
		if (n == matchArray [matchIndex]) {	// walk along signature
			if (matchIndex == matchSize - 1) {			// matched last one
				prevstate = S_AccumulatingHouse;
				state = SS_AccumulateTrue;
				accumulateSize = 4;
				accumulateIndex = 0;
				accumulateBuffer = buffer;
//				printf ("<START>"); fflush (stdout);
				time (&x10Buffer.when);
				x10Buffer.house = x10Buffer.keycode = 0xff;		// invalidate
			} else {
				matchIndex++;
			}
		} else {
			state = S_Reset;
			decode (n);						// try this one again as a protocol lead-in
		}
		break;
	case	S_AccumulatingHouse:			// subroutine return here
		val = convertBitsToByte (buffer, 4);
//		printf ("<HOUSE %c>", houseCodes [val]); fflush (stdout);
		x10Buffer.house = houseCodes [val];
		prevstate = S_AccumulatingKeycodes;
		state = SS_AccumulateTrue;
		accumulateSize = 5;
		accumulateIndex = 0;
		accumulateBuffer = buffer;
		break;
	case	S_AccumulatingKeycodes:			// subroutine return here
		val = convertBitsToByte (buffer, 5);
//		printf ("<%s>\n", keyCodes [val]);
		x10Buffer.keycode = val;
		dumpx10Buffer ();
		state = S_Reset;
		break;
	case	SS_AccumulateTrue:
		accumulateBuffer [accumulateIndex] = n;
		state = SS_AccumulateComplement;
		break;
	case	SS_AccumulateComplement:
		if (accumulateBuffer [accumulateIndex++] == n) {		// wasn't complement!
			state = S_Reset;
		} else {
			if (accumulateIndex == accumulateSize) {			// done
				state = prevstate;
				decode (n);					// try it again to return from subroutine
			} else {
				state = SS_AccumulateTrue;
			}
		}
		break;
	}
}

int
convertBitsToByte (bits, nbits)
char	*bits;
int		nbits;
{
	int		val;
	int		i;

	val = 0;
	for (i = 0; i < nbits; i++) {
		val <<= 1;
		val |= bits [i] & 1;
	}
	return (val);
}

void
dumpx10Buffer ()
{
	FILE	*fp;
	struct	tm *t;

	if ((fp = fopen ("/usr/log/x10.log", "a")) == NULL) {
		return;
	}
	t = localtime (&x10Buffer.when);
	fprintf (fp, "%04d%02d%02d %02d:%02d:%02d House %c %s\n",
			t -> tm_year + 1900, t -> tm_mon + 1, t -> tm_mday,
			t -> tm_hour, t -> tm_min, t -> tm_sec,
			x10Buffer.house,
			keyCodes [x10Buffer.keycode]);
	fclose (fp);
}

