// XWaveInput.h
#ifndef XWAVEINPUT_H
#define XWAVEINPUT_H

// by Randall Cook
// Copyright (C) 1998 Randall Cook. All Rights Reserved.

//
// cross platform data types, constants, structures, and classes:
//

enum XSoundRate {	// samples per second * 65536 (fixed point)
	kXRate11025		= 0x2B110000,
	kXRate11127		= 0x2B7745D1,
	kXRate22050		= 0x56220000,
	kXRate22255		= 0x56EE8BA3,
	kXRate44100		= 0xAC440000
};

enum XSoundSize {	// number of bits
	kXSize8bit		= 8,	// offset 128 format
	kXSize16bit		= 16,	// 16 bit two's complement
    kXSize32bit		= 32	// 32 bit two's complement
};

enum XSoundChan {	// number of channels
	kXChanMono		= 1,
	kXChanStereo	= 2
};

typedef struct {
    unsigned long rate;		// rate * 65536; this allows for 
    						// non-integral sample rates
	XSoundSize size;		// number of bits (8 or 16)
	XSoundChan chan;		// channels (1 or 2)
	
	void Set(XSoundRate r, XSoundSize s, XSoundChan c)
		{ rate = r; size = s; chan = c; }
} XSoundFormat;

inline bool operator==(const XSoundFormat& a, const XSoundFormat& b) {
	return a.rate == b.rate && a.size == b.size && a.chan == b.chan;
}

inline bool operator!=(const XSoundFormat& a, const XSoundFormat& b) {
	return a.rate != b.rate || a.size != b.size || a.chan != b.chan;
}

// bit flag constants for use in XWaveInputCaps below
enum XWaveInputCapabilities {
	kXMonoVarGain	= 1 << 0,	// mono variable gain
	kXStereoVarGain	= 1 << 1,	// stereo variable gain
	kXAGC			= 1 << 2,	// automatic gain control
    // the rest are in order from "least desirable" to "most desirable"
    // this ordering is critical to the operation of GetWinSIHardware()
	kX811Mono		= 1 << 3,	// 8 bit, 11 KHz, mono
	kX811Stereo		= 1 << 4,	// 8 bit, 11 KHz, stereo
	kX1611Mono		= 1 << 5,	// 16 bit, 11 KHz, mono
	kX1611Stereo	= 1 << 6,	// 16 bit, 11 KHz, stereo
	kX822Mono		= 1 << 7,	// 8 bit, 22 KHz, mono
	kX822Stereo		= 1 << 8,	// 8 bit, 22 KHz, stereo
	kX844Mono		= 1 << 9,	// 8 bit, 44 KHz, mono
	kX844Stereo		= 1 << 10,	// 8 bit, 44 KHz, stereo
	kX1622Mono		= 1 << 11,	// 16 bit, 22 KHz, mono
	kX1622Stereo	= 1 << 12,	// 16 bit, 22 KHz, stereo
	kX1644Mono		= 1 << 13,	// 16 bit, 44 KHz, mono
	kX1644Stereo	= 1 << 14,	// 16 bit, 44 KHz, stereo
	// define more constants so we can use the & operator to check for 
	// capabilities more easily
	kXGain		= kXMonoVarGain | kXStereoVarGain | kXAGC,
	kX8			= kX811Mono | kX811Stereo | kX822Mono | 
				  kX822Stereo | kX844Mono | kX844Stereo,
	kX16		= kX1611Mono | kX1611Stereo | kX1622Mono | 
				  kX1622Stereo | kX1644Mono | kX1644Stereo,
	kX11		= kX811Mono | kX811Stereo | kX1611Mono | kX1611Stereo,
	kX22		= kX822Mono | kX822Stereo | kX1622Mono | kX1622Stereo,
	kX44		= kX844Mono | kX844Stereo | kX1644Mono | kX1644Stereo,
	kXMono		= kX811Mono | kX1611Mono | kX822Mono | kX1622Mono | 
				  kX844Mono | kX1644Mono,
	kXStereo	= kX811Stereo | kX1611Stereo | kX822Stereo | 
				  kX1622Stereo | kX844Stereo | kX1644Stereo
};

enum XWaveInputErrorCodes {
	kXNoError				= 0,
	kXBadParamError			= -1,
	kXNoDeviceError			= -2,
    kXNoWindowError			= -3,
    kXSoundCardBusyError	= -4,
    kXAlreadyOnError		= -5,
    kXAlreadyOfferror		= -6
};

typedef int XWaveInputError;

enum GainMode {
	kGainUnavailable,
	kGainLeaveAlone,
	kGainVariableMono,
	kGainVariableStereo,
	kGainAGC
};

enum {
	kMinGain = 0x00008000,
	kUnityGain = 0x00010000,
	kMaxGain = 0x00017FFF
};

// a structure describing how a device should be configured for 
// real-time audio recording
typedef struct XWaveInputSpec {
	XSoundFormat format;	// sample rate, size, and # of channels
	GainMode mode;			// how to set the gain
	unsigned long gain;		// the value we set it to: if gain type
							// is kGainVariableXXXX, this is a number 
							// ranging from kMinGain to kMaxGain; if 
							// gain type is kGainAGC, this is a boolean
	char name[256];			// the name of the device; if name[0] == 0,
							// use the system default device

	void Clear();
} XWaveInputSpec;

// a structure that describes the capabilities of a sound input device
typedef struct {
	long caps;			// a bitmap describing the device capabilities
						// see XWaveInputCapabilities for the bits
	char name[256];		// the name of the device

	void Clear() { caps = 0; name[0] = 0; }
} XWaveInputCaps;

class XWaveInputDataProcessor {
	public:
	virtual void Process(unsigned char *data, long len,
						 XWaveInputSpec& wis, long procData) = 0;
};

class XWaveInputDevice {
	private:
	XWaveInputDataProcessor* processor;	// object that processes
										// the recorded samples
										
	long procData;						// its data parameter
	
	bool recording;						// true when recording

	protected:
	XWaveInputSpec curSpec;				// configuration for the 
										// current recording
	
	// an interrupt routine should call HandleNewSamples to deliver 
	// the new samples to the object that can process them
	void HandleNewSamples(unsigned char* deviceBuffer, long bufLen);
	
	// these functions talk to actual devices and must be overridden
	// for each platform you want to support
	virtual XWaveInputError StartRecording() = 0;
	virtual XWaveInputError StopRecording() = 0;
	
	public:
	XWaveInputDevice();
	~XWaveInputDevice();

	bool IsRecording() { return recording; }
	
	virtual bool HardwareAvailable(XWaveInputSpec& spec) = 0;
    // Determines what hardware is available. You fill out spec with 
    // what you are looking for. spec is changed to show the closest 
    // configuration available. Returns true if device could be 
    // queried, false otherwise.

	XWaveInputError Start(XWaveInputSpec& spec, XWaveInputDataProcessor* dp = 0, long x = 0);
    // Begins wave input recording. Pass it a spec that has been 
    // "checked" by HardwareAvailable(). dp is a pointer to an object 
    // that supports the XWaveInputDataProcessor interface. dp's 
    // Process() function may be called at interrupt time, depending 
    // on implementation. x is dp's parameter. Returns kXNoError if 
    // no error occurred when starting up recording.

	XWaveInputError Stop();
    // Ceases wave input recording. Returns kXNoError if no error.
};

//
// general purpose functions:
// (these must be implemented for MacOS and Windows)
//

int XGetWaveInputDeviceCount();
// Returns the number of wave input devices on the system.

XWaveInputError XGetWaveInputDeviceCapabilities(XWaveInputCaps& c,
												int devIndex = -1);
// If devIndex is -1, set c to the capabilities of the default device 
// and return kXNoError. If devIndex is >=0 and < number of devices, 
// set c to the caps of device devIndex and return kXNoError. If 
// devIndex is >= number of devices, return kXNoDeviceEror.

XWaveInputError XGetWaveInputDeviceCapabilities(XWaveInputCaps& c,
												const char* name);
// If name is 0, set c to the capabilities of the default device and 
// return kXNoError. If name is valid, set c to the caps of the 
// requested device and return kXNoError. If name is invalid, return 
// kXNoDeviceError.

int XGetWaveInputDeviceIndex(const char* name);
// Converts device name into index, or -1 (default) if no device of 
// that name exists on the system.

#endif
