//XWaveInputMac.cpp

#include "XWaveInputMac.h"
#include <StringCompare.h>

/*
extern "C" {
	pascal void MyInterruptRoutine(SPB* spb, unsigned char* buf, 
								   short peak, long bufLen);
}
*/

extern "C" pascal void MyInterruptRoutine(SPB* spb, unsigned char* buf, 
							   short /*peak*/, long bufLen)
{
	XWaveInputDeviceMac* xwi = (XWaveInputDeviceMac*)spb->userLong;
	if (xwi)
		xwi->HandleNewSamples(buf, bufLen);
}

static SIInterruptUPP gInterruptor = NewSIInterruptProc(MyInterruptRoutine);

static unsigned long fixGain(unsigned long gain)
{
	if (gain < 0x00008000)
		return 0x00008000;
	else if (gain > 0x00018000)
		return 0x00018000;
	else
		return gain;
}

static void CtoP(const char *cs, Str255 ps)
{
	int i;
	char *psp;

	if (ps && cs) {
		for (i = 0, psp = (char*)ps + 1; *cs && i < 255; i++)
			*psp++ = *cs++;
		ps[0] = i;
	}
}

static void PtoC(const Str255 ps, char *cs)
{
	int i;
	char *psp;

	if (ps && cs) {
		for (i = 0, psp = (char*)ps + 1; i < ps[0]; i++)
			*cs++ = *psp++;
		*cs = 0;
	}
}

static void PtoCn(const Str255 ps, char *cs, int n)
{
	int i;
	char *psp;

	if (ps && cs) {
		n = ps[0] < n - 1 ? ps[0] : n - 1;
		for (i = 0, psp = (char*)ps + 1; i < n; i++)
			*cs++ = *psp++;
		*cs = 0;
	}
}

OSErr XWaveInputDeviceMac::SetMacSIHardware()
// assumes that XWaveInputDevice::device is valid and that the device is open
{
	OSErr err = noErr;
	short x;
	long xx, x2[2];
	int reason = 0;
	
	#if defined(powerc) || defined(__powerc)
	#pragma options align=mac68k
	#endif
	struct {
		short	a;
		short	b;
	} pair;
	struct {
		short	a;
		short	b;
		short	c;
	} triple;
	#if defined(powerc) || defined(__powerc)
	#pragma options align=reset
	#endif
	
	// set number of channels to record on
	if (err == noErr) {
		x = curSpec.format.chan;
		err = SPBSetDeviceInfo(device, 'chan', (Ptr)&x);
	}
	
	// set to no compression
	if (err == noErr) {
		xx = 'NONE';
		err = SPBSetDeviceInfo(device, 'comp', (Ptr)&xx);
	}
	
	// set sample size
	if (err == noErr) {
		x = curSpec.format.size;
		err = SPBSetDeviceInfo(device, 'ssiz', (Ptr)&x);
	}
	
	// set to not two's complement format
	if (err == noErr) {
		x = (curSpec.format.size == 8) ? 0 : 1;
		err = SPBSetDeviceInfo(device, 'twos', (Ptr)&x);
	}
		
	// set to no play-through volume
	if (err == noErr) {
		x = 0;
		err = SPBSetDeviceInfo(device, 'plth', (Ptr)&x);
		if (err == siUnknownInfoType)
			err = noErr;
	}
	
	// set gain value (will need to do this again because on some macs setting gain has no
	// effect unless the device is actually recording, not just "open"
	if (err == noErr) {
		if (curSpec.mode == kGainVariableMono) {
			xx = fixGain(curSpec.gain);	
			err = SPBSetDeviceInfo(device, 'gain', (Ptr)&xx);
		} else if (curSpec.mode == kGainVariableStereo) {
			xx = fixGain(curSpec.gain);	
			x2[0] = xx;
			x2[1] = xx;
			err = SPBSetDeviceInfo(device, 'sgai', (Ptr)x2);
		} else if (curSpec.mode == kGainAGC) {	
			x = curSpec.gain;
			err = SPBSetDeviceInfo(device, 'agc ', (Ptr)&x);
		}
		if (err == siUnknownInfoType)
			err = noErr;
	}
	
	// set sample rate
	if (err == noErr) {
		xx = curSpec.format.rate;
		err = SPBSetDeviceInfo(device, 'srat', (Ptr)&xx);
	}
	
	// turn off vox recording
	if (err == noErr) {
		pair.a = 0;
		pair.b = 0;
		err = SPBSetDeviceInfo(device, 'voxr', (Ptr)&pair);
		if (err == siUnknownInfoType)
			err = noErr;
	}
	
	// turn off vox stop
	if (err == noErr) {
		triple.a = 0;
		triple.b = 0;
		triple.c = 0;
		err = SPBSetDeviceInfo(device, 'voxs', (Ptr)&triple);
		if (err == siUnknownInfoType)
			err = noErr;
	}
	
	return err;
}

void XWaveInputDeviceMac::SaveMacSIHardware()
// assumes that XWaveInputDevice::device is valid and that the device is open
{
	OSErr err;
	err = SPBGetDeviceInfo(device, 'chan', (Ptr)&snapshot.chanVal);
	snapshot.saveChan = (err == noErr);
	err = SPBGetDeviceInfo(device, 'comp', (Ptr)&snapshot.compVal);
	snapshot.saveComp = (err == noErr);
	err = SPBGetDeviceInfo(device, 'ssiz', (Ptr)&snapshot.ssizVal);
	snapshot.saveSsiz = (err == noErr);
	err = SPBGetDeviceInfo(device, 'twos', (Ptr)&snapshot.twosVal);
	snapshot.saveTwos = (err == noErr);
	err = SPBGetDeviceInfo(device, 'plth', (Ptr)&snapshot.plthVal);
	snapshot.savePlth = (err == noErr);
	err = SPBGetDeviceInfo(device, 'gain', (Ptr)&snapshot.gainVal);
	snapshot.saveGain = (err == noErr);
	err = SPBGetDeviceInfo(device, 'sgai', (Ptr)snapshot.sgaiVal);
	snapshot.saveSgai = (err == noErr);
	err = SPBGetDeviceInfo(device, 'agc ', (Ptr)&snapshot.agcVal);
	snapshot.saveAGC = (err == noErr);
	err = SPBGetDeviceInfo(device, 'srat', (Ptr)&snapshot.sratVal);
	snapshot.saveSrat = (err == noErr);
	err = SPBGetDeviceInfo(device, 'voxr', (Ptr)snapshot.voxrVal);
	snapshot.saveVoxr = (err == noErr);
	err = SPBGetDeviceInfo(device, 'voxs', (Ptr)snapshot.voxsVal);
	snapshot.saveVoxs = (err == noErr);
}

void XWaveInputDeviceMac::RestoreMacSIHardware()
// assumes that XWaveInputDevice::device is valid and that the device is open
{
	if (snapshot.saveChan)
		SPBSetDeviceInfo(device, 'chan', (Ptr)&snapshot.chanVal);
	if (snapshot.saveComp)
		SPBSetDeviceInfo(device, 'comp', (Ptr)&snapshot.compVal);
	if (snapshot.saveSsiz)
		SPBSetDeviceInfo(device, 'ssiz', (Ptr)&snapshot.ssizVal);
	if (snapshot.saveTwos)
		SPBSetDeviceInfo(device, 'twos', (Ptr)&snapshot.twosVal);
	if (snapshot.savePlth)
		SPBSetDeviceInfo(device, 'plth', (Ptr)&snapshot.plthVal);
	if (snapshot.saveGain)
		SPBSetDeviceInfo(device, 'gain', (Ptr)&snapshot.gainVal);
	if (snapshot.saveSgai)
		SPBSetDeviceInfo(device, 'sgai', (Ptr)snapshot.sgaiVal);
	if (snapshot.saveAGC)
		SPBSetDeviceInfo(device, 'agc ', (Ptr)&snapshot.agcVal);
	if (snapshot.saveSrat)
		SPBSetDeviceInfo(device, 'srat', (Ptr)&snapshot.sratVal);
	if (snapshot.saveVoxr)
		SPBSetDeviceInfo(device, 'voxr', (Ptr)snapshot.voxrVal);
	if (snapshot.saveVoxs)
		SPBSetDeviceInfo(device, 'voxs', (Ptr)snapshot.voxsVal);
}

void XWaveInputDeviceMac::PrepareMacRecordRequest()
{
	// set up a sound input parameter block
	recReq.inRefNum = device;
	recReq.count = 0;					// set these fields to zero because we
	recReq.milliseconds = 0;			// just want to analyze the data, not
	recReq.bufferLength = 0;			// to record it
	recReq.bufferPtr = 0;
	recReq.completionRoutine = 0;		/* nothing to be called when finished */
	recReq.interruptRoutine = gInterruptor;
	recReq.userLong = (long)StripAddress(this);
	recReq.unused1 = 0;
}

XWaveInputDeviceMac::XWaveInputDeviceMac()
{
	device = 0;
}

XWaveInputDeviceMac::~XWaveInputDeviceMac()
{
}

XWaveInputError XWaveInputDeviceMac::StartRecording()
{
	OSErr err = noErr;
	
	// open the default sound input device
	if (err == noErr) {
		Str255 s;
		CtoP(curSpec.name, s);
		err = SPBOpenDevice(s, siWritePermission, &device);
	}
	
	// save current hardware state and then set it the way we want it
	if (err == noErr) {
		SaveMacSIHardware();
		err = SetMacSIHardware();
	}
	
	// start recording
	if (err == noErr) {
		PrepareMacRecordRequest();
		err =  SPBRecord((SPB*)StripAddress(&recReq), true);
	}
	
	if (err == noErr) {
		// set gain here, since it sometimes only works when device is actively recording
		long xx, x2[2];
		short x;
		if (curSpec.mode == kGainVariableMono) {
			xx = fixGain(curSpec.gain);
			err = SPBSetDeviceInfo(device, 'gain', (Ptr)&xx);
		} else if (curSpec.mode == kGainVariableStereo) {
			xx = fixGain(curSpec.gain);
			x2[0] = xx;
			x2[1] = xx;
			err = SPBSetDeviceInfo(device, 'sgai', (Ptr)x2);
		} else if (curSpec.mode == kGainAGC) {	
			x = curSpec.gain;
			err = SPBSetDeviceInfo(device, 'agc ', (Ptr)&x);
		}
		
		// check gain setting
		if (curSpec.mode == kGainVariableMono) {
			unsigned long postGain;
			err = SPBGetDeviceInfo(device, 'gain', (Ptr)&postGain);
			if (err == noErr) {
				curSpec.gain = postGain;
			}
		}
	}
	
	// if we got an error, clean up
	if (err != noErr) {
		// close device if opened
		SPBCloseDevice(device);
		device = 0;
	}
	
	return err;
}

XWaveInputError XWaveInputDeviceMac::StopRecording()
{
	if (device) {
		SPBStopRecording(device);
		RestoreMacSIHardware();
		SPBCloseDevice(device);
		device = 0;
	}
	
	return kXNoError;
}

bool XWaveInputDeviceMac::HardwareAvailable(XWaveInputSpec& spec)
{
	OSErr err;
	short x;
	long xx, d, x2[2];
	int i;
	Str255 s;

	#if defined(powerc) || defined(__powerc)
	#pragma options align=mac68k
	#endif
	struct {
		short	n;
		Handle	h;
	} list;
	#if defined(powerc) || defined(__powerc)
	#pragma options align=reset
	#endif

	// open the requested device
	if (spec.name[0]) {
		CtoP(spec.name, s);
		err = SPBOpenDevice(s, siReadPermission, &d);
	} else
		err = SPBOpenDevice(0, siReadPermission, &d);

	// if we couldn't open it, set all wis fields to 0 and exit
	if (err != noErr)
		return false;
	
	// get closest sample rates
	err = SPBGetDeviceInfo(d, 'srav', (Ptr)&list);	// get list of available rates
	if (err == noErr) {
		long diff = 0x7FFFFFFF;		// initialize to a large value
		int choice = -1;			// -1 means no choice made
		for (i = 0; i < list.n; i++) {
			long temp = ((unsigned long*)(*list.h))[i] - spec.format.rate;
			if (temp < 0)
				temp = -temp;		// get absolute value of temp
			if (temp < diff) {		// if closer than previous rate
				diff = temp;		// remember how close
				choice = i;			// and which one it is
			}
		}
		// set to closest to wis->rate samples per second
		if (choice != -1)
			spec.format.rate = (XSoundRate)((unsigned long*)(*list.h))[choice];
		DisposeHandle(list.h);
		if (choice == -1)
			goto fail;
	} else
		goto fail;
	
	// get closest sample size
	err = SPBGetDeviceInfo(d, 'ssav', (Ptr)&list);	// get list of available sample sizes
	if (err == noErr) {
		int diff = 0x7FFFFFFF;		// initialize to a large value
		int choice = -1;			// -1 means no choice made
		for (i = 0; i < list.n; i++) {
			int temp = ((short*)(*list.h))[i] - spec.format.size;
			if (temp < 0)
				temp = -temp;		// get absolute value of temp
			if (temp < diff) {		// if closer than previous rate
				diff = temp;		// remember how close
				choice = i;			// and which one it is
			}
		}
		// set to closest to wis->sampSize
		if (choice != -1)
			spec.format.size = (XSoundSize)((short*)(*list.h))[choice];
		DisposeHandle(list.h);
		if (choice == -1)
			goto fail;
	} else
		goto fail;
	
	// see if the requested number of channels are available
	err = SPBGetDeviceInfo(d, 'chav', (Ptr)&x);
	if (err == noErr) {
		if (spec.format.chan > x)
			spec.format.chan = (XSoundChan)x;
	} else
		goto fail;
	
	// deal with the gain settings
	switch (spec.mode) {
		case kGainVariableMono:
			err = SPBGetDeviceInfo(d, 'gain', (Ptr)&xx);
			if (err == noErr)
				spec.gain = xx;
			else
				spec.mode = kGainUnavailable;
			break;
		case kGainVariableStereo:
			err = SPBGetDeviceInfo(d, 'sgai', (Ptr)x2);
			if (err == noErr)
				spec.gain = (x2[0] + x2[1]) / 2;
			else
				spec.mode = kGainUnavailable;
			break;
		case kGainAGC:
			err = SPBGetDeviceInfo(d, 'agc ', (Ptr)&x);
			if (err == noErr)
				spec.gain = x;
			else
				spec.mode = kGainUnavailable;
			break;
	}
	
	// get the device name
	err = SPBGetDeviceInfo(d, 'name', (Ptr)s);
	if (err == noErr)
		PtoCn(s, spec.name, sizeof(spec.name));
	else
		spec.name[0] = 0;
	
	return true;
	
	fail:
	SPBCloseDevice(d);
	return false;
}

static OSErr GetMacSICaps(XWaveInputCaps& c, const char* name)
// assumes c is valid and name is valid or 0
{
	long d, xx, x2[2];
	short x, channels;
	Str255 s;
	OSErr err;
	long result = 0;
	int has11KHz = 0, has22KHz = 0, has44KHz = 0;
	int has8bit = 0, has16bit = 0;
	#if defined(powerc) || defined(__powerc)
	#pragma options align=mac68k
	#endif
	struct {
		short	n;
		Handle	h;
	} list;
	#if defined(powerc) || defined(__powerc)
	#pragma options align=reset
	#endif
	
	// open the requested device
	if (name) {
		CtoP(name, s);
		err = SPBOpenDevice(s, siReadPermission, &d);
	} else
		err = SPBOpenDevice(0, siReadPermission, &d);
		
	if (err == noErr) {
		// get mono variable gain capability
		err = SPBGetDeviceInfo(d, 'gain', (Ptr)&xx);
		if (err == noErr)
			result |= kXMonoVarGain;
			
		// get stereo variable gain capability
		err = SPBGetDeviceInfo(d, 'sgai', (Ptr)x2);
		if (err == noErr)
			result |= kXStereoVarGain;
			
		// get AGC capability
		err = SPBGetDeviceInfo(d, 'agc ', (Ptr)&x);
		if (err == noErr)
			result |= kXAGC;
			
		// get rate capabilities
		err = SPBGetDeviceInfo(d, 'srav', (Ptr)&list);
		if (err == noErr) {
			for (int i = 0; i < list.n; i++) {
				long rate = ((unsigned long*)(*list.h))[i] >> 16;
				if (rate >= 11000 && rate <= 11200)
					has11KHz = true;
				if (rate >= 22000 && rate <= 22400)
					has22KHz = true;
				if (rate >= 44000 && rate <= 44800)
					has44KHz = true;
			}
			DisposeHandle(list.h);
		}
	
		// get sample size capability
		err = SPBGetDeviceInfo(d, 'ssav', (Ptr)&list);	// get list of available sample sizes
		if (err == noErr) {
			for (int i = 0; i < list.n; i++) {
				int size = ((short*)(*list.h))[i];
				if (size == 8)
					has8bit = true;
				if (size == 16)
					has16bit = true;
			}
			DisposeHandle(list.h);
		}

		// get channel capability
		err = SPBGetDeviceInfo(d, 'chav', (Ptr)&channels);
		if (err != noErr)
			channels = 0;
		
		// get the device name
		err = SPBGetDeviceInfo(d, 'name', (Ptr)s);
		if (err == noErr)
			PtoCn(s, c.name, sizeof(c.name));
		else
			c.name[0] = 0;
		
		// fill out all the bits
		if (has8bit && has11KHz && (channels >= 1))
			result |= kX811Mono;
		if (has8bit && has11KHz && (channels >= 2))
			result |= kX811Stereo;
		if (has8bit && has22KHz && (channels >= 1))
			result |= kX822Mono;
		if (has8bit && has22KHz && (channels >= 2))
			result |= kX822Stereo;
		if (has8bit && has44KHz && (channels >= 1))
			result |= kX844Mono;
		if (has8bit && has44KHz && (channels >= 2))
			result |= kX844Stereo;
		if (has16bit && has11KHz && (channels >= 1))
			result |= kX1611Mono;
		if (has16bit && has11KHz && (channels >= 2))
			result |= kX1611Stereo;
		if (has16bit && has22KHz && (channels >= 1))
			result |= kX1622Mono;
		if (has16bit && has22KHz && (channels >= 2))
			result |= kX1622Stereo;
		if (has16bit && has44KHz && (channels >= 1))
			result |= kX1644Mono;
		if (has16bit && has44KHz && (channels >= 2))
			result |= kX1644Stereo;
		
		c.caps = result;
	}
	return err;
}

int XGetWaveInputDeviceCount()
{
	int n = 1;
	Str255 s;
	Handle h;
	while (SPBGetIndexedDevice(n, s, &h) == noErr) {
		n += 1;
		DisposeHandle(h);
	}
	return n - 1;
}

XWaveInputError XGetWaveInputDeviceCapabilities(XWaveInputCaps& c,
												int devIndex)
// If devIndex is -1, set c to the capabilities of the default device 
// and return kXNoError. If devIndex is >= 0 and < number of devices, 
// set c to the capabilities of device devIndex and return kXNoError. 
// If devIndex is >= number of devices, return kXNoDeviceEror.
{
	c.Clear();

	// turn device index into a name
	int err = noErr;
	if (devIndex == -1) {	// use default device
		err = GetMacSICaps(c, 0);
	} else {
		Str255 s;
		Handle h;
		err = SPBGetIndexedDevice(devIndex + 1, s, &h);
		if (err == noErr) {
			DisposeHandle(h);
			char cs[256];
			PtoC(s, cs);
			err = GetMacSICaps(c, cs);
		}
	}
	return err == noErr ? kXNoError : kXNoDeviceError;
}

XWaveInputError XGetWaveInputDeviceCapabilities(XWaveInputCaps& c,
												const char* name)
// If name is 0, set c to the capabilities of the default device and 
// return kXNoError. If name is valid, set c to the caps of the 
// requested device and return kXNoError. If name is invalid, return 
// kXNoDeviceError.
{
	return GetMacSICaps(c, name);
}

int XGetWaveInputDeviceIndex(const char* name)
// Converts device name into index, or -1 (default) if no device of 
// that name exists on the system.
{
	if (name == 0 || name[0] == 0)
    	return -1;

    int i = 1;
    Str255 sName;
    CtoP(name, sName);
    OSErr err = noErr;
	while (err == noErr) {
    	Str255 s;
		Handle h;
		err = SPBGetIndexedDevice(i, s, &h);
		if (err == noErr) {
			DisposeHandle(h);
            if (EqualString(s, sName, false, false))
            	break;
            else
            	i += 1;
		}
    }
    return (err == noErr) ? i : -1;
}
