// XWaveInputWin.cpp

#include "XWaveInputWin.h"

static HINSTANCE gInstance;

static LRESULT CALLBACK XWaveInputWindProc(HWND w, UINT msg, WPARAM wParam, LPARAM lParam)
{
	if (msg == MM_WIM_DATA) {
    	XWaveInputDeviceWindows* xwi = (XWaveInputDeviceWindows*)GetWindowLong(w, 0);
        
        if (xwi)
        	xwi->ProcessRecycleBufferMessage(lParam);
        
        return 0;
    } else
		return DefWindowProc(w, msg, wParam, lParam);
}

void InitXWaveInputWindows(HINSTANCE h)
{
	gInstance = h;
	
    // create and register XWaveInput window class
	WNDCLASS wc;
    wc.style = 0;
    wc.lpfnWndProc = (WNDPROC)XWaveInputWindProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 4;
    wc.hInstance = gInstance;
    wc.hIcon = NULL;
    wc.hCursor = NULL;
    wc.hbrBackground = NULL;
    wc.lpszMenuName = 0;
	wc.lpszClassName = "XWaveInput";
	RegisterClass(&wc);
}

void XWaveInputDeviceWindows::ProcessRecycleBufferMessage(LPARAM lParam)
{
	// Extract the InputBuffer pointer from the WAVEHDR structure.
	LPWAVEHDR hdr = (LPWAVEHDR)lParam;
	InputBuffer* ib = (InputBuffer*)hdr->dwUser;
	
	if (ib)
    	RecycleBuffer(ib);
}

bool XWaveInputDeviceWindows::ErrorSuggestsBusyDevice(MMRESULT err)
// these errors (3, 4, 8, and 32) often come when the device is busy
{
	return err == MMSYSERR_NOTENABLED || err == MMSYSERR_ALLOCATED ||
		   err == MMSYSERR_NOTSUPPORTED || err == WAVERR_BADFORMAT;
}

MMRESULT XWaveInputDeviceWindows::TryToOpenWaveInDevice(int devIndex, 
	WAVEFORMATEX& wf, DWORD callbackObject, DWORD callbackObjectType)
// Prepare to try to open the device several times if necessary. 
// Sets device, and returns the error code from the open call.
{
	MMRESULT err = 0;

	// Prepare to try to open the device several times if necessary.
	// If it works the first time, we'll break out. We try for 3.5 
	// seconds since that is a reasonable amount of time to try to 
	// open the device. 0.25 seconds is a reasonable time to wait 
	// between tries, since the user will have to wait at most this 
	// amount of time for things to start. The 14 below is 3.5/0.25.
	
	for (int count = 0; count < 14; count++) {
		err = waveInOpen(&device, devIndex, &wf, callbackObject, 
						 (DWORD)this, callbackObjectType);
		
		if (ErrorSuggestsBusyDevice(err))
			Sleep(250);
		else
			break;
	}

	return err;
}

void XWaveInputDeviceWindows::PrepareBuffers()
{
	for (int i = 0; i < kBufferCount; i++)
    	bufferList[i]->Prepare(device);
}

void XWaveInputDeviceWindows::UnprepareBuffers()
{
	for (int i = 0; i < kBufferCount; i++)
    	bufferList[i]->Unprepare();
}

void XWaveInputDeviceWindows::ProcessBufferData(InputBuffer* ib)
{
    if (ib->GetDataLen()) {
        // Process the samples here. There are ib->GetDataLen() bytes
        // of data at ib->GetData().
        HandleNewSamples(ib->GetData(), ib->GetDataLen());
    }
}

void XWaveInputDeviceWindows::RecycleBuffer(InputBuffer* ib)
{
	// Only recycle buffers that have been filled and are not being 
	// flushed out.
	if (ib->ContainsValidData() && device != 0) {
        // Since we are using a window as the notification object, we 
        // must process the samples now.
		ProcessBufferData(ib);
			
		// Ihis does the actual recycling.
        ib->Prepare(device);
    }
}

XWaveInputDeviceWindows::XWaveInputDeviceWindows()
{
    char cs[256];
    wsprintf(cs, "XWaveInputWin%lX", (unsigned long)this);
    privateWindow = CreateWindow("XWaveInput", cs, WS_OVERLAPPEDWINDOW, 100, 100, 320, 200, NULL, NULL, gInstance, 0);
	if (privateWindow)
    	SetWindowLong(privateWindow, 0, (long)this);

	device = 0;
	
	for (int i = 0; i < kBufferCount; i++)
		bufferList[i] = new InputBuffer(4096);
}

XWaveInputDeviceWindows::~XWaveInputDeviceWindows()
{
	if (IsRecording())
		Stop();
	
	for (int i = 0; i < kBufferCount; i++)
		delete bufferList[i];
}

void XWaveInputDeviceWindows::ConvertXWaveInSpecToWindows(
	XWaveInputSpec& wis, WAVEFORMATEX& wfx)
{
	wfx.wFormatTag = WAVE_FORMAT_PCM;
	wfx.nChannels = wis.format.chan;
	wfx.nSamplesPerSec = (unsigned long)wis.format.rate / 65536L;
	if (wis.format.size == kXSize16bit) {
		wfx.nBlockAlign = 2;
		wfx.wBitsPerSample = 16;
	} else {
		wfx.nBlockAlign = 1;
		wfx.wBitsPerSample = 8;
	}
	wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.nBlockAlign * wfx.nChannels;
	wfx.cbSize = 0;
}

XWaveInputError XWaveInputDeviceWindows::StartRecording()
{
	// Describe the desired recording format.
	WAVEFORMATEX wf;
	ConvertXWaveInSpecToWindows(curSpec, wf);
	
	// Open the default device.
	MMRESULT err = TryToOpenWaveInDevice(WAVE_MAPPER, wf,
			  (DWORD)privateWindow, CALLBACK_WINDOW);
	
	if (err == MMSYSERR_NOERROR) {
    	PrepareBuffers();				// Prepare the supply of input buffers.

		err = waveInStart(device);		// Actually begin recording.

		if (err != MMSYSERR_NOERROR) {	// Clean up on errors.
        	UnprepareBuffers();
			waveInClose(device);
			device = 0;
        }
	}

	return err;
}

XWaveInputError XWaveInputDeviceWindows::StopRecording()
{
	// Since the goal of this function is to stop recording and
	// close the device, it is reasonable to ignore errors.

	if (IsRecording()) {
		waveInReset(device);
		waveInStop(device);
	    UnprepareBuffers();
		waveInClose(device);
	    device = 0;
    }
}


bool XWaveInputDeviceWindows::HardwareAvailable(XWaveInputSpec& spec)
{
	XWaveInputCaps wic;

    // get the current capabilities of the requested hardware
    int err = XGetWaveInputDeviceCapabilities(wic, spec.name);
    if (err != 0)
    	return false;

    // turn the requested hardware into a caps bitmap
    long desired = -1;							// start with all bits 1
    if (spec.format.chan == kXChanStereo)		// if stereo requested
    	desired &= kXStereo;					// keep only stereo bits
    else if (spec.format.chan == kXChanMono)	// if mono requested
    	desired &= kXMono;						// keep only mono bits
    if (spec.format.size == kXSize16bit)		// if 16 bit requested
    	desired &= kX16;						// keep only 16 bit bits
    else if (spec.format.size == kXSize8bit)	// if 8 bit requested
    	desired &= kX8;                     	// keep only 8 bit bits
    if (spec.format.rate == kXRate44100)		// if 44KHz requested
    	desired &= kX44;						// keep only 44KHz bits
    else if (spec.format.rate == kXRate22050)	// if 22KHz requested
    	desired &= kX22;						// keep only 22KHz bits
    else if (spec.format.rate == kXRate11025)	// if 11KHz requested
    	desired &= kX11;						// keep only 11KHz bits

    // if hardware not available
    if ((desired & wic.caps) == 0) { // hardware is not available
        // find the "best" hardware that is available
        // start with the bit value for the "best" (16/44/stereo)
        // and shift it right until it matches the current caps
        long best = kX1644Stereo;
        while ((wic.caps & best) == 0 && best >= kX811Mono)
        	best /= 2;

		if ((best & ~kXGain) == 0)	// if device supports no rates/bits/channels
			return false;
			
        // deal with channels
        if ((best & kXStereo) != 0 && spec.format.chan == kXChanStereo)
        	spec.format.chan = kXChanStereo;
        else
        	spec.format.chan = kXChanMono;

        // deal with sample size
        if ((best & kX16) != 0 && spec.format.size == kXSize16bit)
        	spec.format.size = kXSize16bit;
        else
        	spec.format.size = kXSize8bit;

        // deal with rate
        if ((best & kX44) != 0 && spec.format.rate == kXRate44100)
        	spec.format.rate = kXRate44100;
        else if ((best & kX22) != 0 && spec.format.rate == kXRate22050)
        	spec.format.rate = kXRate22050;
        else
        	spec.format.rate = kXRate11025;
    }
    
    // gain is not supported under Windows yet
	spec.mode = kGainUnavailable;
    spec.gain = 0;
    
    return true;
}

int XGetWaveInputDeviceCount()
// Returns the number of wave input devices on the system.
{
	return waveInGetNumDevs();
}

XWaveInputError XGetWaveInputDeviceCapabilities(XWaveInputCaps& c,
												int devIndex)
// If devIndex is -1, set c to the capabilities of the default device 
// and return kXNoError. If devIndex is >=0 and < number of devices, 
// set c to the caps of device devIndex and return kXNoError. If 
// devIndex is >= number of devices, return kXNoDeviceEror.
{
	c.Clear();

	WAVEINCAPS wic;
	int err = waveInGetDevCaps(devIndex, &wic, sizeof(wic));
	if (err == 0) {
		lstrcpy(c.name, wic.szPname);
		c.caps = 0;
		if (wic.dwFormats & WAVE_FORMAT_1M08)
			c.caps |= kX811Mono;
		if (wic.dwFormats & WAVE_FORMAT_1S08)
			c.caps |= kX811Stereo;
		if (wic.dwFormats & WAVE_FORMAT_2M08)
			c.caps |= kX822Mono;
		if (wic.dwFormats & WAVE_FORMAT_2S08)
			c.caps |= kX822Stereo;
		if (wic.dwFormats & WAVE_FORMAT_4M08)
			c.caps |= kX844Mono;
		if (wic.dwFormats & WAVE_FORMAT_4S08)
			c.caps |= kX844Stereo;
		if (wic.dwFormats & WAVE_FORMAT_1M16)
			c.caps |= kX1611Mono;
		if (wic.dwFormats & WAVE_FORMAT_1S16)
			c.caps |= kX1611Stereo;
		if (wic.dwFormats & WAVE_FORMAT_2M16)
			c.caps |= kX1622Mono;
		if (wic.dwFormats & WAVE_FORMAT_2S16)
			c.caps |= kX1622Stereo;
		if (wic.dwFormats & WAVE_FORMAT_4M16)
			c.caps |= kX1644Mono;
		if (wic.dwFormats & WAVE_FORMAT_4S16)
			c.caps |= kX1644Stereo;
	}
	return err;
}

XWaveInputError XGetWaveInputDeviceCapabilities(XWaveInputCaps& c,
												const char* name)
// If name is 0, set c to the capabilities of the default device and 
// return kXNoError. If name is valid, set c to the caps of the 
// requested device and return kXNoError. If name is invalid, return 
// kXNoDeviceError.
{
	c.Clear();

	if (name == 0 || name[0] == 0) {
		return XGetWaveInputDeviceCapabilities(c, -1);
	} else {
		int index = XGetWaveInputDeviceIndex(name);
		return (index != -1) ? XGetWaveInputDeviceCapabilities(c, index)
							 : kXNoDeviceError;
	}
}

int XGetWaveInputDeviceIndex(const char* name)
// Converts device name into index, or -1 (default) if no device of 
// that name exists on the system.
{
	if (name == 0 || name[0] == 0)
    	return -1;

    int i;
    int n = waveInGetNumDevs();
    int err = kXNoDeviceError;
	WAVEINCAPS wic;
	
	for (i = 0; i < n; i++) {
		err = waveInGetDevCaps(i, &wic, sizeof(wic));
		if (err == 0 && lstrcmpi(name, wic.szPname) == 0)
			break;
	}
	return (i < n && err == 0) ? i : -1;
}
