// Listing 3b: InputBuffer.cpp (monospaced, tabs every 4 spaces)
// A class for abstracting and managing input buffers on Windows.
// by Randall Cook
// Copyright (C) 1998 Randall Cook. All Rights Reserved.

#include "InputBuffer.h"

InputBuffer::InputBuffer(int bufSize)
{
	blockHandle = GlobalAlloc(GMEM_MOVEABLE | GMEM_SHARE, bufSize);
	state = (blockHandle != 0) ? kAllocated : kNotAvailable;
	blockPtr = 0;
	header.lpData = 0;
	header.dwBufferLength = bufSize;
	header.dwBytesRecorded = 0;
	header.dwUser = (DWORD)this;
	header.dwFlags = 0;
	header.dwLoops = 0;
	header.lpNext = 0;
	header.reserved = 0;
	device = 0;
}

InputBuffer::~InputBuffer()
{
	if (state >= kPrepared)
    	Unprepare();
    Unlock();
	if (blockHandle)
    	GlobalFree(blockHandle);
}

void InputBuffer::Prepare(HWAVEIN dev)
{
    Lock();	// lock first
    if (state >= kLocked) {
    	header.lpData = (char*)blockPtr;
        MMRESULT err;
        err = waveInPrepareHeader(dev, &header, sizeof(WAVEHDR));

        // add it to the buffer list
        if (err == 0) {
            waveInAddBuffer(dev, &header, sizeof(WAVEHDR));
            state = kPrepared;
            device = dev;
        }
    }
}

void InputBuffer::Unprepare()
{
    if (state == kPrepared) {
        waveInUnprepareHeader(device, &header, sizeof(WAVEHDR));
        state = kLocked;
        Unlock();
    }
}

void InputBuffer::Lock()
{
	if (blockHandle != 0 && state == kAllocated) {
    	blockPtr = (unsigned char*)GlobalLock(blockHandle);
        if (blockPtr)
        	state = kLocked;
    }
}

void InputBuffer::Unlock()
{
	if (blockHandle != 0 && state == kLocked) {
    	GlobalUnlock(blockHandle);
        blockPtr = 0;
        state = kAllocated;
    }
}

bool InputBuffer::ContainsValidData()
{
	const DWORD kValidMask = WHDR_DONE | WHDR_PREPARED;
    return (header.dwFlags & kValidMask) == kValidMask;
}
