// Listing 1 (monospaced, tabs every 4 spaces)
// A simple MacOS program that can do real-time sound processing.
// by Randall Cook
// Copyright (C) 1998 Randall Cook. All Rights Reserved.
// ------1---------2---------3---------4---------5---------6---------7
#include <SoundInput.h>

extern "C" {
	pascal void MyInterruptRoutine(SPB* spb, unsigned char* buf, 
								   short peak, long bufLen);
}

pascal void MyInterruptRoutine(SPB* spb, unsigned char* buf, 
							   short peak, long bufLen)
{
	// Process bufLen bytes of data at buf. You should cast buf to be
	// a pointer to 16 bit samples (if necessary) and adjust bufLen 
	// accordingly. You can put a pointer to globals or some other 
	// useful structure in spb->userLong when you begin the recording. 
	// The operating system computes the largest amplitude of the 
	// samples just recorded and passes that value in peak.
}

SIInterruptUPP gInterruptor = NewSIInterruptProc(MyInterruptRoutine);

OSErr ConfigureDevice(long device)
{
	// In production code, we would query the device with 
	// SPBGetDeviceInfo to make sure it can do what we want it
	// to. Here, we will only set the sample rate, sample size, 
	// and number of channels to 44100 Hz, 16 bit, mono. A real 
	// program would also save and restore the current state of 
	// the device as well as configure most of the two dozen 
	// parameters a device can possibly support.
	
	short shortVal;	// To ensure that SPBSetDeviceInfo accesses
	long longVal;	// properly sized data, we use these utility 
					// variables.
	
	// Set sample rate to 44100 Hz.
	longVal = 44100L << 16;	// siSampleRate needs a pointer to a long
							// which contains the rate * 65536
	err = SPBSetDeviceInfo(device, siSampleRate, (Ptr)&longVal);
	
	// Set sample size to 16 bit.
	if (err == noErr) {
		shortVal = 16;	// siSampleSize needs a pointer to a short
		err = SPBSetDeviceInfo(device, siSampleSize, (Ptr)&shortVal);
	}
	
	// Set number of channels to 1 (mono).
	if (err == noErr) {
		shortVal = 1;	// siNumberChannels needs a pointer to a short
		err = SPBSetDeviceInfo(device, siNumberChannels, (Ptr)&shortVal);
	}
	
	return err;
}

void SetRecordingParameters(SPB& recordParams)
{
	// Indicate which open device we want to record on.
	recordParams.inRefNum = device;
	
	// Setting count, milliseconds, bufferLength, and bufferPtr
	// to zero indicates that the recording is to be continuous.
	recordParams.count = 0;
	recordParams.milliseconds = 0; 
	recordParams.bufferLength = 0;
	recordParams.bufferPtr = 0;
	
	// No function needs to be called when finished.
	recordParams.completionRoutine = 0;
	
	// Set the interrupt routine.
	recordParams.interruptRoutine = gInterruptor;
	
	// This can be any 32 bit value we want. It is usually
	// something useful for the interrupt routine.
	recordParams.userLong = 0;
	
	recordParams.unused1 = 0;	// Unused: must be zero.
}

OSErr MacOSRecord()
{
	bool deviceOpen = false;
	bool recording = false;
	long device = 0;
	SPB recordParams;
	
	// Open the default device.
	OSErr err = SPBOpenDevice(0, siWritePermission, &device);
	deviceOpen = (err == noErr);
	
	// Configure the device.
	if (err == noErr) {
		err = ConfigureDevice(device);
	}
	
	// Begin recording.
	if (deviceOpen) {
		// Fill out recordParams for continuous real-time recording.
		SetRecordingParameters(recordParams);
		
		// Now start the recording.
		err = SPBRecord(&recordParams, true);
		recording = (err == noErr);
	}
	
	// Do anything we want while the program is recording.
	if (recording) {
		// We probably should return to the main event loop now,
		// but we won't in order to keep this example simple.
	}
	
	// Stop recording.
	if (recording) {
		err = SPBStopRecording(device);
		recording = false;
	}
	
	// Close the device.
	if (deviceOpen) {
		err = SPBCloseDevice(device);
		deviceOpen = false;
	}
	
	return err;
}
