// Listing 6 (monospaced, tabs every 4 spaces)
// A faster clipping fix algorithm.
// by Randall Cook
// Copyright (C) 1998 Randall Cook. All Rights Reserved.

enum ClipState {
	kClipUnprimed,
	kClipNormal,
	kClipInvert
};

// Fixes nSamples of 16 bit samples at buf in place. state is the result
// of the last call, or kClipUnprimed to initialize. lastSample is 
// the result of the last call, or 0 to initialize. For best results,
// the abs function below should be inlined. Assumes 32 bit integers.
void FastFixClip16(short* buf, long nSamples, ClipState& state, 
				   long& lastSample)
{
	int last;	// Use a local instead of lastSample so the 
				// compiler can keep it in a register.
	int cur;
	const long clipThreshold = 35000;	// This can vary from system
										// to system.

	// initialize state and last
	if (state == kClipUnprimed) {
		last = *buf;
		state = kClipNormal;
	} else
		last = lastSample;
	
	nSamples += 1;			// Add 1 to compensate for the --nSamples
							// in the loops below, which we do since
							// --x is genreally faster than x--.

	while (nSamples > 1) {	// We need (nSamples > 1) rather than 
							// (nSamples > 0) since we added 1 above.
		
		// Invert loop:
		if (state == kClipInvert) {
			while (--nSamples) {
				cur = *buf;
				if (cur == -32768)
					cur = -32767;
				if ((cur ^ last) < 0) {
					// we crossed zero. do we need to switch modes?
					if (abs(cur - last) > clipThreshold) {
						// threshold exceeded: switch modes
						buf += 1;
						state = kClipNormal;
						last = cur;
						break;
					} else
						*buf++ = -cur;
				} else
					*buf++ = -cur;
				last = cur;
			}
		}

		// No invert loop:
		if (state == kClipNormal) {
			while (--nSamples) {
				cur = *buf++;
				if ((cur ^ last) < 0) {
					// we crossed zero. do we need to switch modes?
					if (abs(cur - last) > clipThreshold) {
						// threshold exceeded: switch modes
						if (cur == -32768)
							cur = -32767;
						buf[-1] = -cur;
						state = kClipInvert;
						last = cur;
						break;
					}
				}
				last = cur;
			}
		}
	}

	lastSample = last;
}
