import java.io.*;
import java.lang.reflect.*;
import java.util.*;

public class CodePacker
{
	protected String m_destination;
	protected String m_newName;
	protected String[] m_pFiles;
	public Method m_crypto = null;
	
	public static void main (String[] args)
	{
		if( args.length < 3 )
		{
			System.out.println ( "CodePacker has no arguments" );
			return;
		}
		
		CodePacker cp = new CodePacker( args );
		cp.build();
	}
	
	public CodePacker( String[] fileNames )
	{
		m_destination = fileNames[0];
		m_newName = fileNames[1];
		m_pFiles = new String[ fileNames.length - 2 ];
		System.arraycopy( fileNames, 2, m_pFiles, 0 ,fileNames.length - 2 );
	}
	
	public void build ( )
	{
		
		byte[] encryptedClasses = readClassFiles( );
		writeClassFile( m_destination, m_newName, encryptedClasses );
	}
	
	protected byte[] readClassFiles( )
	{
		CodePackerClassLoader cl = new CodePackerClassLoader();
		
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream out = new DataOutputStream( bout );
		for ( int i = 0; i < m_pFiles.length; i++ )
		{
			try
			{
				Method c = null;
				
				byte[] classBytes = cl.readClassFile ( m_pFiles[i], true );
				
				c = cl.getEncrypt( stripExtension ( m_pFiles[i] ) );
				
				if ( m_crypto != null )
				{
					System.out.println ( "Encrypting something " + classBytes.length );
					classBytes = (byte[])m_crypto.invoke( null, new Object[] { classBytes } );
					System.out.println ( "Encrypted something " + classBytes.length );
				}
				
				if ( c != null )
				{
					System.out.println ( "Switching cryptography method" );
					m_crypto = c; // switch to new crypto class
					out.writeByte( 1 ); // crypto class
				} else if ( i == ( m_pFiles.length - 1 ) ) 
				{
					// is this the last file in the list
					// main class is always last class in list
					out.writeByte( 2 ); 
				} else
				{
					out.writeByte( 0 ); // do nothing special with the class
				}
							
				out.writeInt( classBytes.length);
				out.write( classBytes );
				
			} catch ( Exception e )
			{
				System.out.println ( e.getMessage() + " occured in build " );
			}
		}
		
		return bout.toByteArray();
		
	}
	
	protected String stripExtension( String className )
	{
		if ( className.endsWith( ".class" ) == true )
		{
			return className.substring(0, className.length() - ".class".length() );
		}
		
		return className;
	}
		
	
	public void writeClassFile ( String name, String newName, byte[] bytes )
	{
		try
        {
			RandomAccessFile inFile = new RandomAccessFile( name,"r" );
            RandomAccessFile outFile = new RandomAccessFile( newName, "rw" );
			
			// write magic number, major and minor version info
			int magic = inFile.readInt();
			int minor = inFile.readUnsignedShort();
			int major = inFile.readUnsignedShort();
			
            outFile.writeInt( magic );			
			outFile.writeShort( minor );	
			outFile.writeShort( major ); 
            
			// write the attribute table
            int constCount = inFile.readUnsignedShort();
			outFile.writeShort( constCount + 2 );
			
			int codePackerAttributeIndex = constCount;
			long[] offsetTable = new long[ codePackerAttributeIndex ] ;
			short j = 0;
            for ( j=0; j < (constCount-1); j++ )
            {
                int type = inFile.readUnsignedByte();
				outFile.writeByte( type );
                    
                switch ( type )
                {        
                case 1:// utf-8
                    int len = inFile.readUnsignedShort();
                    byte[] utf8bytes = new byte[len];
                    inFile.read ( utf8bytes );
					
					//System.out.println ( (j+1) + " UTF8 " + new String(utf8bytes) );
					
					outFile.writeShort( len );
					outFile.write( utf8bytes );
                    break;
					
                case 3: // int
                case 4: // float; 
                    outFile.writeInt(inFile.readInt());
                    break;
					
                case 5: // long
                case 6: // double
                    outFile.writeLong(inFile.readLong());
                    j++;
                    break;
					
                case 7: // class
                case 8: // string
					if ( type == 7 )
					{
						int tmp = inFile.readShort();
						//System.out.println ( (j+1) + " Class utf8 at index " + tmp );
						offsetTable[ j+1 ] = outFile.getFilePointer();
						outFile.writeShort ( tmp );
					} else
					{
						outFile.writeShort(inFile.readShort());
					}
                    break;
					
                case 9: // field
                case 10: // method
                case 11: // interface
                case 12: // name and type
                    outFile.writeInt(inFile.readInt());
                    break;
                        
                default:
                    System.out.print ( "Unknown type" );
                    break;
                }
            }
			// write a new UTF8 object for the new class name
			int classNameIndex = j+1;
			outFile.writeByte( (byte) 1 );
			String tmpName = stripExtension ( newName );
			outFile.writeShort( tmpName.getBytes().length );
			outFile.write( tmpName.getBytes() );
			
			// write the type, length and new attribute info
			int customAttributeIndex = j+2;
			outFile.writeByte( (byte) 1 );
			outFile.writeShort( "CodePackerCustomAttribute".getBytes().length );
			outFile.write( "CodePackerCustomAttribute".getBytes() );
			
			/**
			 * Attribute table is now written
			 */

			// write access flags
			outFile.writeShort( inFile.readUnsignedShort() ); // access Flags
			
			// replace this index with classConstIndex to represent out new name
			int thisIndex = inFile.readUnsignedShort(); // this index
			outFile.writeShort( thisIndex );		
			long currentPosition = outFile.getFilePointer();
			
			outFile.seek( offsetTable[ thisIndex ] );
			outFile.writeShort ( classNameIndex );
			outFile.seek ( currentPosition );
			
			// write the super class 
			outFile.writeShort( inFile.readUnsignedShort() );
                
			// write the interface table
            int interface_count = inFile.readUnsignedShort();
			byte[] interfaceBuffer = new byte[ interface_count * 2 ];
			inFile.read( interfaceBuffer );
			
			outFile.writeShort( interface_count );
			outFile.write ( interfaceBuffer );
			    
            // write the field table
			writeFieldMethodTable( inFile, outFile );
			
			// write the method table 
			writeFieldMethodTable( inFile, outFile );
			
			
			// write attribute table
			long attributesOffset = outFile.getFilePointer();
			System.out.println ( "Attribute Offset at " + attributesOffset );
            int attributes_count = inFile.readUnsignedShort();
			outFile.writeShort ( attributes_count ); 
            
			// write exisiting attributes
            for ( int k = 0; k < attributes_count; k++ )
            {
                writeAttributes( inFile, outFile );
            }
            
            // write class data attributes
            int attributesAdded = writeAttributeRecords( outFile, customAttributeIndex, bytes );
            
			// write attribute to indicate start of attribute table
			outFile.writeShort ( customAttributeIndex );
			outFile.writeInt ( 9 );
			outFile.writeByte ( (byte) 1 );
			outFile.writeLong ( attributesOffset );
			attributesAdded++;
			
			
            // set the attribute count with the new attributes           
            outFile.seek ( attributesOffset );
			outFile.writeShort( attributes_count + attributesAdded );
			System.out.println ( "attributes: " + ( attributes_count + attributesAdded ) ); 
        } catch ( Exception e )
        {
            e.printStackTrace();
        }		
	}
	
	public int writeAttributeRecords( RandomAccessFile outFile,
									  int index,
									  byte[] classBytes ) throws IOException
    {
        int attrCount = 0;
        int blockSize = 1024 * 48;
		
		int blockCount = ( classBytes.length / blockSize )
						 + ( classBytes.length % blockSize != 0 ? 1 : 0 );
		int offset = 0;
		System.out.println ( "writing " + blockCount + " blocks of classes" );
		System.out.println ( "classBytes = " + classBytes.length  );
		for( int i = 0; i < blockCount; i++ )
		{
			outFile.writeShort ( index );
			int chunkSize = 0;
			
			
			if ( classBytes.length - offset >= blockSize )
			{
				chunkSize = blockSize;
			} else
			{
				chunkSize = (int)(classBytes.length - offset);
			}
			System.out.println ( "\tChunk size= " + chunkSize );
			byte[] chunk = new byte[chunkSize];
			System.arraycopy( classBytes, offset, chunk,0, chunkSize );
			offset += chunkSize;
			
			outFile.writeInt( chunk.length + 1 );
			outFile.writeByte ( (byte) 0 );
			outFile.write( chunk );
			attrCount++;
		}
		
        return attrCount;
    }
    
	
	public void writeFieldMethodTable ( RandomAccessFile inFile, RandomAccessFile outFile ) throws IOException
	{
		int table_count = inFile.readUnsignedShort();
		outFile.writeShort( table_count );
		for ( int j = 0; j < table_count; j++ )
		{
			byte[] entryBuffer = new byte[ 6 ];
			inFile.read( entryBuffer );
			outFile.write( entryBuffer );
			
			int attribute_count = inFile.readUnsignedShort();
			outFile.writeShort( attribute_count );
			
			for ( int k = 0; k < attribute_count; k++ )
			{
				writeAttributes( inFile, outFile );
			}
		}
	}
	
	public void writeAttributes ( RandomAccessFile inFile, RandomAccessFile outFile ) throws IOException
    {
        outFile.writeShort( inFile.readShort() );
        int attribute_length = inFile.readInt();
		byte[] attributeBuffer = new byte[ attribute_length ];
		inFile.read( attributeBuffer );
		
		// this may not be an int
		outFile.writeInt ( attribute_length );
		outFile.write( attributeBuffer );
    }
	
		
	public class CodePackerClassLoader extends ClassLoader
	{
		private Hashtable ht = new Hashtable();
				
		public byte[] readClassFile ( String name, boolean define ) throws IOException
		{
			FileInputStream is = new FileInputStream( name );
			byte[] results = new byte[ is.available() ];
			is.read( results );
			
			if ( define == true )
			{
				this.defineClass( null, results, 0, results.length );
			}
			
			return results;
		}
		
		public Method getEncrypt( String name )
		{
			Method res = null;
			
			try
			{
				Class c = loadClass( name );
				res = c.getDeclaredMethod( "encrypt", new Class[] { byte[].class } );
			} catch (Throwable t)
			{
				return null;
			}
			
			return res;
		}
	
		public Class loadClass ( String name, boolean verify ) throws ClassNotFoundException
		{
			Class c = null;
			c = (Class)ht.get( name );
			if( c == null )
			{
				c = this.findSystemClass( name );
				if ( c != null )
				{
					ht.put( name, c );
				}
			}
			
			if ( ( c != null ) && ( verify ) )
			{
				this.resolveClass( c );
			}
			return c;
		}
	}
}
