/*

Copyright 1998, Tim Kientzle.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of Tim Kientzle may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR AUTHORS ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR AUTHORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

*/

import java.io.*;
import java.awt.*;
import java.applet.Applet;

/*
 * A pretty simple applet design.  Since there's only a couple of
 * UI elements, I subclass those elements and attach most of the
 * program functionality to those elements.  This is a simple approach
 * that works well for simple programs.
 *
 * Specifically, I've created a custom:
 *   * Applet, that builds the basic UI and holds references to everything else
 *   * A custom text field, that notifies the word list whenever the
 *     text changes
 *   * A word list box, that looks up words in a database, updates its own
 *     display and notifies the file list box.
 *   * A file list box, that lists the files matching the search.
 *   * A 'word' object that stores a string together with an array
 *     of integers (indicating files containing that word)
 *
 * This is all pretty standard stuff, except for the file list box.
 * With standard components, everything is fast enough to
 * update dynamically as you type---including database searches---except
 * for the file list box.
 *
 * A standard list box simply can't handle several
 * hundred (or thousand) additions at a pop.  (Netscape 4/Mac even
 * loses the rest of the applet GUI when you try this!)  So,
 * the file list box is really a custom UI widget that pulls a
 * few tricks.  Fortunately, that mess
 * is all safely encapsulated within the file list box class.
 */


/***************************************************************************/

/**
 * The applet is pretty simple.  The two methods are init(), which
 * lays out all of the components, and openDatabase(), which tries
 * valiantly to open a database file.
 */
public class Search extends Applet {

  SearchTextField searchTextField;
  SearchFileList fileList;
  SearchWordList wordList;
  DBBTree indexDB; // Index database

  public void start() {
    System.out.println("Application Starting.");
    if(indexDB != null) {
      try {
	indexDB.open();
      } catch (Exception e) {
	System.err.println("**** Couldn't re-open index ****\n");
      }
    }
    super.start();
    requestFocus(); // Try to get focus
    searchTextField.requestFocus(); // Try to move focus to text field
    validate(); // Make sure everything is in its place
    repaint();  // Paint everything
  }

  public void stop() {
    System.out.println("Application Stopping.");
    if(indexDB != null)
      indexDB.close();
    super.stop();
  }

  public void init() {
    String labelText;

    System.out.println("Initing Search Applet");

    indexDB = openDatabase("index.db");

    setBackground(Color.white);
    setLayout(new BorderLayout(10,10));
    
    Panel searchPanel = new Panel();
    searchPanel.setLayout(new BorderLayout(0,0));
    add("North", searchPanel);
    labelText = null;
    try {
      labelText = getParameter("searchTitle");
    } catch (Exception e) {
    }
    if(labelText != null && labelText.length() > 0) {
      Label searchLabel = new Label(labelText);
      searchPanel.add("West", searchLabel);
    }
    searchTextField = new SearchTextField(this);
    searchPanel.add("Center", searchTextField);
    
    Panel filePanel = new Panel();
    filePanel.setLayout(new BorderLayout(0,0));
    add("Center", filePanel);
    labelText = null;
    try {
      labelText = getParameter("filesTitle");
    } catch (Exception e) {
    }
    if(labelText != null && labelText.length() > 0) {
      Label fileLabel = new Label(labelText,Label.CENTER);
      filePanel.add("North", fileLabel);
    }
    fileList = new SearchFileList(this);
    filePanel.add("Center", fileList);
    
    Panel wordPanel = new Panel();
    wordPanel.setLayout(new BorderLayout(0,0));
    add("West", wordPanel);
    labelText = null;
    try {
      labelText = getParameter("wordsTitle");
    } catch (Exception e) {
    }
    if(labelText != null && labelText.length() > 0) {
      Label wordLabel = new Label(labelText,Label.CENTER);
      wordPanel.add("North", wordLabel);
    }
    wordList = new SearchWordList(this);
    wordList.setBackground(Color.white);
    wordPanel.add("Center", wordList);

    requestFocus(); // Try to get focus
    searchTextField.requestFocus(); // Try to move focus to text field
  }

  public String getAppletInfo() {
    return "DDJ Search Applet, Copyright July 1998 by Tim Kientzle";
  }

  // TODO: Under some versions of Netscape Navigator, if you use the
  // code here to go to an HTML page, then use "Find in Page..." on
  // that page, then use the back button, the applet will be
  // non-functional.
  void showPage(java.net.URL url) {
    String where = getParameter("where");
    getAppletContext().showDocument(url,where);
  }

  // There doesn't seem to be any one way to open a file that
  // works on all platforms, so this function tries several...
  DBBTree openDatabase(String filename) {
    StringBuffer work;
    DBBTree db;

    // First, try seeing if there's such a thing as a
    // 'current directory'
    try {
      db = new DBBTree(new File(filename));
      System.out.println("    Opened " + filename);
      return db;
    } catch (IOException e) {
      System.out.println("    Couldn't open " + filename);
      System.out.println("    IOException: " + e);
    }

    // Manipulate the document base URL to build a more complete pathname.
    try {
      filename = new java.net.URL(getDocumentBase(),filename).getFile();
    } catch (java.net.MalformedURLException e) {
      // If you can't build a more complete pathname, you're sunk
      return null;
    }

    // Try filename part of file: URL
    try {
      db = new DBBTree(filename);
      System.out.println("    Opened " + filename);
      return db;
    } catch (IOException e) {
      System.out.println("    Couldn't open " + filename);
      System.out.println("    IOException: " + e);
    }

    // Try changing the / to the native separator
    // Note: some platforms lie about the native separator... <sigh>
    work = new StringBuffer(filename);
    try {
      char separator = File.separatorChar; 
      System.out.println("    file.separator = " + separator);
      for(int i=0;i<work.length();i++) // Change / to native separator
	if(work.charAt(i) == '/') work.setCharAt(i,separator);
      db = new DBBTree(work.toString());
      System.out.println("    Opened " + work);
      return db;
    } catch (IOException e) {
      System.out.println("    Couldn't open " + work);
      System.out.println("    IOException: " + e);
    }

    // Strip first character and see what happens...
    filename = filename.substring(1);
    try {
      db = new DBBTree(filename);
      System.out.println("    Opened " + filename);
      return db;
    } catch (IOException e) {
      System.out.println("    Couldn't open " + filename);
      System.out.println("    IOException: " + e);
    }
    
    // Everything failed... <sigh>
    return null;
  }
}





