/*

Copyright 1998, Tim Kientzle.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of Tim Kientzle may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR AUTHORS ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR AUTHORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

*/

import java.io.*;
import java.awt.*;

/***************************************************************************/

/**
 * The file list box is given a list of integers by the word list box.
 * It looks up the title and file name of each of those files and
 * displays it in a list box.  For efficiency, I've implemented my
 * own list box from scratch.  The primary feature of my list box
 * is that it is 'lazy.'  In particular, it only looks up files in
 * the database when they're needed for display.  For example,
 * even if a search results in 1300 files, this class will only
 * immediately look up the 10-15 names needed for the current
 * display.  If the user scrolls to another part of the display,
 * I'll look those names up then.
 *
 * This file list box is composed of a Panel, a Canvas, and two
 * Scrollbars.  I subclass the Panel to contain everything.  I
 * also subclass the Canvas, which is where the database lookups
 * and paint logic is kept.
 */
public class SearchFileList extends Panel {
  Search applet;  // The parent applet
  SearchFileCanvas canvas;
  Scrollbar vertScrollbar;
  Scrollbar horzScrollbar;

  public void updateFileList( int [] fileIds ) {
    canvas.updateFileList(fileIds);
  }

  public SearchFileList(Search a) {
    applet = a;
    canvas = new SearchFileCanvas(a);
    vertScrollbar = new Scrollbar(Scrollbar.VERTICAL);
    horzScrollbar = new Scrollbar(Scrollbar.HORIZONTAL);

    GridBagLayout  gridbag  =  new  GridBagLayout();
    GridBagConstraints  c  =  new  GridBagConstraints();
    setLayout(gridbag);
    c.weightx  = 1.0;
    c.weighty  = 1.0;
    c.fill  =  GridBagConstraints.BOTH;
    gridbag.setConstraints(canvas,c);
    add(canvas);

    c.weightx = 0;
    c.weighty = 0;
    c.fill = GridBagConstraints.VERTICAL;
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.insets = new Insets(0,2,0,2);
    gridbag.setConstraints(vertScrollbar,c);
    add(vertScrollbar);
    c.gridwidth = GridBagConstraints.RELATIVE;
    c.gridheight = GridBagConstraints.REMAINDER;
    c.fill = GridBagConstraints.HORIZONTAL;
    c.insets = new Insets(2,0,2,0);
    gridbag.setConstraints(horzScrollbar,c);
    add(horzScrollbar);

    validate(); // Layout this component
    repaint();

    canvas.setVerticalScrollbar(vertScrollbar);
    canvas.setHorizontalScrollbar(horzScrollbar);
  }

  /**
   * Scroll events simply result in repainting the canvas.
   */
  public boolean handleEvent(Event event) {
    if (  (event.target == vertScrollbar) ||(event.target == horzScrollbar)
	&&(  (event.id == Event.SCROLL_ABSOLUTE)
	   || (event.id == Event.SCROLL_LINE_DOWN)
	   || (event.id == Event.SCROLL_LINE_UP)
	   || (event.id == Event.SCROLL_PAGE_DOWN)
	   || (event.id == Event.SCROLL_PAGE_UP)
	  )
       )
    {
      canvas.repaint();
      return true;
    } else {
      return super.handleEvent(event);
    }
  }    
}

/**
 * The paint() implementation is a bit involved, since a repaint
 * needs to look up filenames in the database, paint the correct
 * strings based on the scrollbar positions, and update the
 * scrollbars.
 */
class SearchFileCanvas extends Canvas {
  Search applet;  // The parent applet
  java.util.Vector fileCaptionsList; // An in-memory cache of file titles
  java.util.Vector fileNamesList; // An in-memory cache of filenames
  int selected; // Currently selected item

  private void setSelection(int x, int y) {
    y -= fontHeight - fontAscent;
    int newSelection = (y + vertScrollbar.getValue()) / fontHeight;
    int oldSelection = selected;
    // To minimize flicker, only redraw parts that have changed.
    if (newSelection != selected) {
      // Repaint old selection
      repaint(0,(selected+1)*fontHeight-fontAscent-vertScrollbar.getValue()-1,
	      1000,fontHeight);
      if(myFileIds != null && newSelection < myFileIds.length) {
	selected = newSelection; // Change selection
	// Repaint new selection
	repaint(0,
		(selected+1)*fontHeight-fontAscent-vertScrollbar.getValue()-1,
		1000,fontHeight);
      } else {
	selected = -1;
      }
    }
  }

  public boolean mouseMove(Event event, int x, int y) {
    if(event.target != this) {
      return super.mouseUp(event,x,y);
    }
    setSelection(x,y);

    if(myFileIds == null || selected < 0) {
      applet.showStatus("");
      return true;
    }
    try {
      String filename = (String)fileNamesList.elementAt(myFileIds[selected]);
      java.net.URL url = new java.net.URL(applet.getDocumentBase(),filename);
      applet.showStatus(url.toString());
    } catch (Exception e) {
      // Failure to display a URL is not fatal...
      System.err.println("*** Exception: "  + e );
    }
    return true;
  }

  public boolean mouseDown(Event event, int x, int y) {
    if(event.target != this) {
      return super.mouseUp(event,x,y);
    }
    mouseMove(event,x,y); // Reset selection, etc.

    // On a click, show selected article
    if(myFileIds == null 
       || selected > myFileIds.length
       || selected < 0)
      return true;
    try {
      String filename = (String)fileNamesList.elementAt(myFileIds[selected]);
      java.net.URL url = new java.net.URL(applet.getDocumentBase(),filename);
      System.out.println("Showing: " + url);
      applet.showPage(url);
    } catch (Exception e) {
      System.err.println("*** Couldn't display selection: Error: "  + e );
    }

    return true;
  }

  SearchFileCanvas(Search a) {
    applet = a;
    fileCaptionsList = new java.util.Vector(500);
    fileNamesList = new java.util.Vector(500);
  }

  Scrollbar vertScrollbar;
  Scrollbar horzScrollbar;

  // Of course, bounds() is deprecated, but getBounds() isn't in 1.0
  // <sigh>
  private Rectangle getWindowSize() { return bounds(); }

  void setVerticalScrollbar(Scrollbar s) {
    vertScrollbar = s;
    int height = getWindowSize().height;
    vertScrollbar.setValues(0,height,0,height);
  }
  void setHorizontalScrollbar(Scrollbar s) {
    horzScrollbar = s;
    currentMaxWidth = getWindowSize().width-10;
    horzScrollbar.setValues(0,currentMaxWidth,0,currentMaxWidth);
  }

  int [] nextFileIds;
  int [] myFileIds;
  int currentMaxWidth;

  /**
   * Change the list of files that should be displayed.
   */
  public void updateFileList( int [] fileIds ) {
    nextFileIds = fileIds;
    repaint();
  }

  /**
   * Find the caption and filename corresponding to the given file number.
   * These strings are cached in <I>fileCaptionsList</I> and
   * <I>fileNamesList</I>, so the full database lookup only occurs
   * once per file.
   *
   * The file caption and filename are separated by a tab.
   */
  String getFileCaption(int fileId) {
    // If this element is already in the cache, just return
    String ret = null;
    try {
      ret = (String)fileCaptionsList.elementAt(fileId);
      if(ret != null) return ret;  // Is in cache, return now
      // Not in cache, continue
    } catch (ArrayIndexOutOfBoundsException e) {
      // The cache isn't even that big... continue.
    }

    // Try looking up this file in the file database
    // If succeeds, search result is in 'ret'
    byte [] b = {0, (byte)(fileId/256), (byte)(fileId&255)};
    try {
      byte [] fileRaw = applet.indexDB.search(b);
      if(fileRaw != null) // Convert ISO Latin-1 string to Unicode
	ret = new String(fileRaw,0);
      // if search returned null, leave ret = null
    } catch (java.io.IOException e) {
      // Search failed, leave ret = null
    }

    // Search failed: print an error and synthesize a suitable entry
    if(ret == null) {
      System.err.println("*** Search for file " + fileId + " failed ***");
      ret = "<Error: fileId " + fileId + ">";
    }

    // Find last tab, separate filename part
    int fileStart = ret.lastIndexOf('\u0009');
    String fileName;
    if(fileStart > 0) {
      fileName = ret.substring(fileStart+1);
      ret = ret.substring(0,fileStart);
    } else {
      fileName = "";
    }

    // Insert results into appropriate vectors

    // <grrr...> What's the point of an automatically-sized
    // Vector class if I can't insert wherever I want without
    // first manually resizing the darned thing??!!
    if(fileCaptionsList.size() < fileId + 1) // Make sure it's big enough..
      fileCaptionsList.setSize(fileId+1);
    if(fileNamesList.size() < fileId + 1) // Make sure it's big enough..
      fileNamesList.setSize(fileId+1);
    
    fileCaptionsList.setElementAt(ret,fileId); // Insert entry
    fileNamesList.setElementAt(fileName,fileId);
    return ret;
  }

  /**
   * The font height is needed to align the window, and to
   * translate mouse click locations to string names.
   */
  int fontHeight;
  int fontAscent;

  /**
   * Paint a warning message!
   */
  private void paintWarning(Graphics g, String warning) {
    Rectangle bounds = getWindowSize();
    
    FontMetrics fontMetrics = g.getFontMetrics();
    int fontHeight = fontMetrics.getHeight()+5;
    int currentY = fontHeight;
    g.setColor(Color.red);
    g.clipRect(4,4,bounds.width-8,bounds.height-8);
  
    java.util.StringTokenizer st 
      = new java.util.StringTokenizer(warning,"\t");
    while (st.hasMoreTokens()) {
      String word = st.nextToken();
      int thisWidth = fontMetrics.stringWidth(word);
      g.drawString(word,(bounds.width-thisWidth)/2, currentY);
      currentY += fontHeight; // Next line
    }
  }

  protected void paintBevels (Graphics g, Rectangle bounds) {
    // Draw bevels around edge of canvas
    g.setColor(new Color(192,192,192));
    g.drawLine(0,0,0,bounds.height-1); // Outer left
    g.drawLine(0,0,bounds.width-1,0); // Outer top
    g.setColor(new Color(128,128,128));
    g.drawLine(1,1,1,bounds.height-2); // Mid top
    g.drawLine(1,1,bounds.width-2,1); // Mid left
    g.setColor(new Color(64,64,64));
    g.drawLine(2,2,2,bounds.height-3); // Inner top
    g.drawLine(2,2,bounds.width-3,2); // Inner left

    g.setColor(new Color(192,192,192));
    g.drawLine(3,bounds.height-3,bounds.width-3,bounds.height-3); // Inner bot
    g.drawLine(bounds.width-3,3,bounds.width-3,bounds.height-3); // Inner right
    g.setColor(new Color(128,128,128));
    g.drawLine(2,bounds.height-2,bounds.width-2,bounds.height-2); // Mid bot
    g.drawLine(bounds.width-2,2,bounds.width-2,bounds.height-2); // Mid right
    g.setColor(new Color(64,64,64));
    g.drawLine(1,bounds.height-1,bounds.width-1,bounds.height-1); // Outer bot
    g.drawLine(bounds.width-1,1,bounds.width-1,bounds.height-1); // Outer right
  }

  /**
   * Paint the canvas window based on scroll bar position, etc.
   */
  public void paint(Graphics g) {
    Rectangle bounds = getWindowSize();

    // If database isn't open, display suitable warning!
    if(applet.indexDB == null) {
      paintWarning(g, "** ERROR **\t"
		   + "Database not Found!\t"
		   + "\t"
		   + "Your browser may not allow\t"
		   + "applets to open files."
		   );
      return;
    }

    paintBevels(g,bounds);  // Paint the bevels

    // Clip text drawing to fit within that bevel
    g.setColor(Color.black);
    g.clipRect(4,4,bounds.width-8,bounds.height-8);

    // Get basic font information.
    // The font height is stored in an instance variable.
    Font font = g.getFont();
    FontMetrics fontMetrics = g.getFontMetrics();
    fontHeight = fontMetrics.getHeight() + 5;
    fontAscent = fontMetrics.getMaxAscent();

    // If the file list has changed, reset the scrollbars
    // Note that the scrollbars are in units of 'pixels'.
    // The full range of the scrollbar is the total pixel height
    // of all of the filenames (fontHeight*myFileIds.length)
    // with some adjustments.
    boolean horzChanged = false;
    if(myFileIds != nextFileIds) {
      myFileIds = nextFileIds;
      selected = 0; // Reset selection
      // Reset vertical scrollbar now...
      vertScrollbar.setLineIncrement(fontHeight);
      vertScrollbar.setPageIncrement(bounds.height - 10 - fontHeight);
      vertScrollbar.setValues(0,               // Current: top of scrollbar
			      bounds.height-10,// S/B height = win height
			      0,               // Min position
			      fontHeight*(myFileIds.length+1));
      horzChanged = true; // Reset horizontal scrollbar at end of paint()
    }


    // Paint all of the strings within the current display
    int currentX = 5 - horzScrollbar.getValue(); 
    int currentFile = vertScrollbar.getValue() / fontHeight;
    int currentY = (currentFile+1)*fontHeight - vertScrollbar.getValue();
    while(    (currentY < bounds.height + fontHeight)
	   && (myFileIds != null)
	   && (currentFile < myFileIds.length)    ) {
      String filename = getFileCaption(myFileIds[currentFile]);
      // Measure width of string and possibly update horizontal scrollbar
      int thisWidth = fontMetrics.stringWidth(filename);
      if (thisWidth > currentMaxWidth) {
	currentMaxWidth = thisWidth;
	horzChanged = true;
      }
      if(currentFile == selected) {
	g.setColor(Color.red);
	g.drawString(filename, currentX, currentY);
      } else {
	g.setColor(Color.black);
	g.drawString(filename, currentX, currentY);
      }
      currentY += fontHeight; // Next line
      currentFile++; // Next file name
    }

    // If the horizontal scrollbar information has changed, update it
    // I defer this because the currentMaxWidth can potentially change
    // many times during the redraw, and I don't want to waste time
    // updating the scroll bar more than once.
    if(horzChanged) {
      int curvalue = horzScrollbar.getValue();
      horzScrollbar.setValues(curvalue,bounds.width-10,
			      0,currentMaxWidth+10);
      horzScrollbar.setLineIncrement(fontHeight);
      horzScrollbar.setPageIncrement(bounds.width - 2*fontHeight);
    }
  }
}

/***************************************************************************/
