/*

Copyright 1998, Tim Kientzle.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of Tim Kientzle may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR AUTHORS ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR AUTHORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

*/

import java.io.*;

/***************************************************************************/

/**
 * Utility class to hold statistics about a single word.
 */
public class SearchWord extends Object {
  String word;
  int[] counts;
  int countsLength;

  public int getCount(int i) {
    if(i<countsLength)  return counts[i];
    else return 0;
  }
	
  public int getNumberCounts() { return countsLength; }
  
  public int totalNonZero() { // Return number of non-zero entries
    int total = 0;
    for(int i=0; i<countsLength; i++)
      if(counts[i] > 0)	total ++;
    return total;
  };
  
  public SearchWord() {
    word = ""; countsLength = 1; counts = new int[countsLength];}

  public SearchWord(String w) { this(); word = w; }
  
  // Change size of counts[]
  private final void resize(int size) {
    int[] newCounts = new int[size];
    for(int i=0;i<size && i<countsLength;i++)
      newCounts[i] = counts[i];
    counts = newCounts;
    countsLength = size;
  }
  
  // Make sure counts[] is big enough
  // Note: doubling strategy ensures linear time for building array
  private final void ensure(int size) {
    if(size < countsLength) return;
    int newLength = countsLength;
    while(size >= newLength) newLength *= 2;
    resize(newLength);
  }
  
  
  // Create a new word from a word and a data input
  public SearchWord(String w, java.io.DataInput e) {
    word = w; // first token is word
    countsLength = 16;
    counts = new int[countsLength];
    int fileId = 0, count = 0, maxFileId = 0;
    while(true) {
      try { // Current strategy: each word in DB maps to a list of fileIds
	fileId = e.readShort();
	count = 1;
      } catch (IOException ex) {
	break;
      }
      ensure(fileId+1); // Make sure array is large enough
      counts[fileId] = count;
      if (fileId > maxFileId) maxFileId = fileId;
    }
    resize(maxFileId + 1); // Release unused array entries
  }
  
  // Update counts to reflect usage of both words
  // Equivalent to ``map min''
  public SearchWord and(SearchWord w) {
    int i;
    if(w.countsLength > countsLength) ensure(w.countsLength);
    for(i=0; i<countsLength && i<w.countsLength; i++)
      if(w.getCount(i) < counts[i]) counts[i] = w.getCount(i);
    for(; i<countsLength ; i++)
      counts[i] = 0;
    return this;
  }
  
  // OR: Update counts to reflect usage of either word
  // Equivalent to ``map max''
  public SearchWord or(SearchWord w) {
    if(w.countsLength > countsLength) ensure(w.countsLength);
    for(int i=0; i<countsLength && i<w.countsLength; i++)
      if(w.getCount(i) > counts[i]) counts[i] = w.getCount(i);
    return this;
  }

  // this := this "and not" w
  public SearchWord andNot(SearchWord w) {
    int i;
    if(w.countsLength > countsLength) ensure(w.countsLength);
    for(i=0; i<countsLength && i<w.countsLength; i++)
      if( (counts[i] > 0) && (w.getCount(i) > 0) )
	counts[i] = 0;
    return this;
  }

  /**
   * Return an int[] containing all of the file ids.
   */
  public int [] getFileIds() {
    int [] fileIds = new int[totalNonZero()];
    for(int i=0,index=0; i<getNumberCounts(); i++) {
      if(counts[i] > 0)
	fileIds[index++] = i;
    }
    return fileIds;
  }

}

/***************************************************************************/
