/*

Copyright 1998, Tim Kientzle.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of Tim Kientzle may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR AUTHORS ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR AUTHORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

*/

import java.io.*;
import java.awt.*;

/***************************************************************************/


/**
 * List of words and counts.  This list box receives a string from
 * the text field, parses it and looks up the relevant words in
 * the word database.  When the search results change, it hands
 * a list of file numbers to the file list box.
 */
public class SearchWordList extends List {
  Search applet;   // Handle on applet object

  public SearchWordList(Search a) {
    applet = a;
    if(applet.indexDB == null) {
      // If database isn't open, put an error message in the list box
      addItem("**************",0);
      addItem("** ERROR",1);
      addItem("**************",2);
      addItem("** COULDN'T",3);
      addItem("** OPEN",4);
      addItem("** INDEX",5);
      addItem("**************",6);
    } else {
      // If success, prime list box with empty total
      addItem("TOTAL: 0",0);
    }
  }

  // Break text into words, look up each new word in database,
  // notify file list box when finished.
  void UpdateWordList(String s) {
    int itemCount = countItems();
    // Don't bother if database open failed.
    if (applet.indexDB == null) return;

    SearchWord total = null; // Cumulative statistics
    int listPosition = 0;
    java.util.StringTokenizer t = new java.util.StringTokenizer(s);
    while(t.hasMoreTokens()) {
      // Retrieve next word info from database
      SearchWord word = LookupWord(CanonicalWord(t.nextToken()));
      
      // Build item for list, build result vector
      String listEntry; 
      if(word.totalNonZero() > 0) { // Only count words that exist
	listEntry = word.word + ": " + word.totalNonZero();
	if(total == null) { // Handle first one specially.
	  total = new SearchWord();
	  total.or(word);
	} else {
	  total.and(word);
	}
      } else {
	listEntry = word.word + ": (Not in Index)";
      }
      if(listPosition >= itemCount) {
	addItem(listEntry,listPosition);
	itemCount++;
      } else if(!getItem(listPosition).equals(listEntry)) {
	replaceItem(listEntry,listPosition);
      }
      listPosition++;

    }

    // Add 'TOTAL' entry to list
    String listEntry = "TOTAL: " + ((total == null)?0:total.totalNonZero());
    if(listPosition >= itemCount) { 
      addItem(listEntry,listPosition);
      itemCount++;
    } else if(!getItem(listPosition).equals(listEntry))
      replaceItem(listEntry,listPosition);
    listPosition++;


    // Clean up end of wordPanel list
    while(listPosition < itemCount) {
      delItem(listPosition);
      itemCount--;
    }

    // Copy file ids into an array
    // (Someday, sort this array to implement results ranking!)
    if(total != null) {
      applet.fileList.updateFileList(total.getFileIds());
    } else {
      applet.fileList.updateFileList(new int[0]);
    }
  }

  // Convert the word into a canonical format (e.g., uppercase)
  // This is also the place to implement grammatical knowledge.
  String CanonicalWord(String w) {
    StringBuffer b = new StringBuffer(w);
    return b.toString().toLowerCase();
  }
  
  /**
   * Cache words that have already been looked up
   */
  java.util.Hashtable currentWords = new java.util.Hashtable();

  /**
   * Find a word in the database and return a 'Word' object.
   * Maintain a cache to keep from going to disk too often.
   */
  SearchWord LookupWord(String w) {
    if(currentWords.containsKey(w)) {
      return (SearchWord)currentWords.get(w);
    }
    // If database open failed, punt...
    if(applet.indexDB == null) return new SearchWord(w);

    // Convert string into a byte sequence
    byte [] b = new byte[w.length()];
    for(int i=0; i<w.length(); i++) b[i] = (byte)w.charAt(i);
    // Look up key in database
    byte [] wordData;
    try {
      //System.out.println("Searching for word " + w + " in index.");
      wordData = applet.indexDB.search(b);
    } catch (IOException e) {
      System.err.println("*** Search for " + w + " generated IOException");
      return new SearchWord(w);
    }

    // Build a word object and cache it
    SearchWord word;
    if(wordData == null) {
      word = new SearchWord(w);
    } else {
      DataInput t = new DataInputStream(new ByteArrayInputStream(wordData));
      word = new SearchWord(w,t);

      // DEBUG: dump word stats
      String debugOut = "Word `" + w + "' appears in files ";
      int totalFiles = 0;
      for(int i=0; i<word.getNumberCounts(); i++) {
	if(word.getCount(i) > 0) {
	  totalFiles ++;
	  if(totalFiles < 10) debugOut = debugOut + " " + i;
	  else if (totalFiles < 11) debugOut = debugOut + " ... ";
	}
      }
      System.out.println(debugOut + " (" + totalFiles + " total)");
    }
    currentWords.put(w,word);
    return word;
  }
}


/***************************************************************************/

