
package dph.awt.lightweight;

import java.awt.*;
import java.awt.event.*;


/**
 * Defines a base class for encapsulating behaviors common to Buttons.  These are
 * mostly event-handling behaviors such as allowing listeners to register for
 * ActionEvents, processing mouse activity and calling actionPerformed.
 *
 * @author      Perelman-Hall, David
 */
public abstract class ButtonBase extends CommonBase {

    protected String label;
    protected String actionCommand;
    protected boolean pressed = false;
    protected ActionListener actionListener;


    /**
     * Constructs the ButtonBase by constructing its parent then storing the
     * label to use for the Button.
     *
     * @param  label  the String label to use for the button
     */
    ButtonBase( String label ) {
        super();
        this.label = label;
    }


    // -----[ Accessor Methods ]-----

    
    /**
     * Sets the label of the ButtonBase to the String newLabel argument.
     *
     * @param  newLabel  the String label to use for the button
     */
    public synchronized void setLabel( String newLabel ) {
        label = newLabel;
        this.invalidate();
        this.repaint();
    }

    /**
     * Gets the label of the ButtonBase.
     *
     * @return  the String label of the button
     */
    public String getLabel() {
        return label;
    }


    /**
     * Sets the actionCommand of the ButtonBase.
     *
     * @param  command  the String actionCommand of the button
     */
    public synchronized void setActionCommand( String command ) {
        actionCommand = command;
    }


    /**
     * Gets the actionCommand of the ButtonBase.
     *
     * @return  the String actionCommand of the button
     */
    public String getActionCommand() {
        return actionCommand==null ? label : actionCommand;
    }


 
    
    // -----[ add and remove actionListeners with AWTEventMulticaster ]-----
    
    
    /**
     * Adds the ActionListener as a registered listener of ActionEvents on
     * the button.
     *
     * @param al         the ActionListener to add
     */
    public void addActionListener( ActionListener al ) {
        actionListener = AWTEventMulticaster.add( actionListener, al );
    }


    /**
     * Removes the ActionListener as a registered listener of ActionEvents on
     * the button.
     *
     * @param al         the ActionListener to remove
     */
    public void removeActionListener( ActionListener al ) {
        actionListener = AWTEventMulticaster.remove( actionListener, al );
    }
    
    
   /**
    * Handle any mouse activity on the button, especially processing mouse
    * clicks to perform actionCommand.  Pass all other activity up the
    * inheritance chain.
    *
    *@param   me    the MouseEvent to process
    */
    public void processMouseEvent( MouseEvent me ) {
        if( isEnabled() ) {
            switch( me.getID() ) {
                case MouseEvent.MOUSE_CLICKED:
                        if( actionListener != null ) {
                            actionListener.actionPerformed( new ActionEvent (
                                this, ActionEvent.ACTION_PERFORMED, this.getActionCommand()
                            ) );
                        }
            }
        }
        super.processMouseEvent( me );
    }
    
}