
package dph.awt.lightweight;


/**
 * Defines a selected/unselected Checkbox class.  Listeners can register to listen
 * for ItemChanged Events on the Checkbox.
 *
 * @author      Perelman-Hall, David
 */
public class Checkbox extends CommonBase implements java.awt.ItemSelectable {

    protected String label;
    protected boolean state = false;
    protected java.awt.event.ItemListener itemListener;    



    /**
     * Constructs the Checkbox by calling the other Checkbox constructor
     * with an empty label and an initial state of unchecked (false).
     */
    public Checkbox(){
        this( "", false );
    }
    
    
    /**
     * Constructs the Checkbox by calling the other Checkbox constructor
     * with the label argument and an initial state of unchecked (false).
     *
     * @param  label   the String label of the Checkbox
     */
    public Checkbox( String label ) {
        this( label, false );
    }
    

    /**
     * Constructs the Checkbox, setting its label to the label argument 
     * and the state of cheked or unchecked to the state argument.
     *
     * @param  label   the String label of the Checkbox
     * @param  state   the checked state of the Checkbox
     */
    public Checkbox( String label, boolean state ) {
        super();
        this.label = label;
        this.state = state;
    }
    

    // -----[ Accessor Methods ]-----
 
    
    /**
     * Sets the label of the Checkbox to the newLabel argument.  Calls
     * <code>invalidate</code> to force the layout the Checkbox again.
     *
     * @param newLabel         the new String label to use
     */
    public synchronized void setLabel( String newLabel ) {
        label = newLabel;
        this.invalidate();
        this.repaint();
    }


    /**
     * Returns the String label of the Checkbox
     *
     * @return the label of the Checkbox
     */
    public String getLabel() {
        return label;
    }


    /**
     * Sets the state of the Checkbox to the newState argument.
     *
     * @param newState         the new boolean state to use
     */
    public void setState( boolean newState ) {
        this.state = newState;
        repaint();
    }
    
    /**
     * Returns the boolean state of the Checkbox
     *
     * @return the state of the Checkbox
     */
    public boolean getState() { return state; }



    // --------------------------------------
    // -----[ ItemSelectable Interface ]-----
    // --------------------------------------

    
    /**
     * Adds the ItemListener as a registered listener of ItemEvents on
     * the Checkbox.
     *
     * @param il         the ItemListener to add
     */
    public void addItemListener( java.awt.event.ItemListener il ) {
        itemListener = java.awt.AWTEventMulticaster.add( itemListener, il );
    }


    /**
     * Removes the ItemListener as a registered listener of ItemEvents on
     * the Checkbox.
     *
     * @param il         the ItemListener to remove
     */
    public void removeItemListener( java.awt.event.ItemListener il ) {
        itemListener = java.awt.AWTEventMulticaster.remove( itemListener, il );
    }
    
    
    /**
     * Part of the ItemListener interface, returns the current Checkbox.
     *
     * @return the current checkbox
     */
    public Object[] getSelectedObjects() {
        Object[] objs = { this };
        return objs;
    }


   /**
    * Gets the optimum size for this Checkbox, which will be calculated to encompass
    * any label and checkable box, with a few pixels extra on all four sides.
    *
    *@param   g    the Graphics context to use for figuring size
    *
	* @return  the Dimension representing the Checkbox's size
    */
   protected java.awt.Dimension figureMySize( java.awt.Graphics g ) {

      int width = 17;
      int height = 25;
      

      // Calculate actual dimensions
      if( g != null ) {
            g.setFont( new java.awt.Font( "dialog", java.awt.Font.PLAIN, 12 ) );
         java.awt.FontMetrics fm = g.getFontMetrics();

         // Calculate title dimensions
         if( label != null ) {

            // Get more accurate height
            height = fm.getHeight() + 10;

            // Calculate label width
            width = dph.util.Util.pixelsWideForString( label, fm );
         }
      }
      
      // add space for the checkbox itself
      width += 6 + height;
      return new java.awt.Dimension( width, height );
   }


    
   /**
    * Paints the Checkbox.
    *
    *@param   g    the Graphics context to use for painting
    */
    public void paint( java.awt.Graphics g ) {
        java.awt.Dimension mySize = getSize();
        g.setFont( new java.awt.Font( "dialog", java.awt.Font.PLAIN, 12 ) );
        g.setColor( java.awt.Color.lightGray.brighter() );
        g.drawLine( 1, 1, mySize.width-1, 1 );
        g.drawLine( 1, 1, 1, mySize.height-1 );

        g.setColor( java.awt.Color.black );
        g.drawLine( 2, mySize.height-2, mySize.width, mySize.height-2 );
        g.drawLine( 1, mySize.height-1, mySize.width-1, mySize.height-1 );
        g.drawLine( mySize.width-1, 1, mySize.width-1, mySize.height-1 );
      
        // Draw label over image
        if( label != null  &&  label.length() > 0 ) {

            // Right-justify label
            java.awt.FontMetrics fm = g.getFontMetrics();
            int x = mySize.width - dph.util.Util.pixelsWideForString( label, fm );
            x -= 6;

            int y = (mySize.height / 2 + fm.getHeight()/2) -1;
            if( isEnabled() )
                g.setColor( java.awt.Color.black );
            else    
                g.setColor( java.awt.Color.gray );
            g.drawString( label, x, y );
        }
      
        g.setColor( java.awt.Color.black );
        g.fillRect( 6, 6, mySize.height-12, mySize.height-12 );

        g.setColor( java.awt.Color.lightGray.brighter() );
        g.fillRect( 7, 7, mySize.height-15, mySize.height-15 );

        if( state ) {
            g.setColor( java.awt.Color.black );
            g.drawLine( 8, 12, 10, 16 );
            g.drawLine( 8, 11, 10, 15 );
            g.drawLine( 10, 16, 18, 8 );
            g.drawLine( 10, 15, 17, 8 );
        }
    }

    
    
   /**
    * Handle any mouse activity on this Checkbox, especially processing mouse
    * clicks to change the Checkbox state.  Pass all other activity up the
    * inheritance chain.
    *
    *@param   evt    the MouseEvent to process
    */
   public void processMouseEvent( java.awt.event.MouseEvent evt ) {
      switch( evt.getID() ) {
      case java.awt.event.MouseEvent.MOUSE_CLICKED:
         if( isEnabled() ) {
            state = !state;
                if( itemListener != null ) {
                    int stateChange = state ? java.awt.event.ItemEvent.SELECTED : java.awt.event.ItemEvent.DESELECTED;
                    itemListener.itemStateChanged( new java.awt.event.ItemEvent (
                        this, java.awt.event.ItemEvent.ITEM_STATE_CHANGED, this.label, stateChange
                    ) );
                }
            repaint();
         }
         break;
      default:
         super.processMouseEvent( evt );
      }
   }



    // -----[ Unit Test ]-----
    public static void main( String[] argsw ) {
        DoubleBufferedExitingFrame frame = new DoubleBufferedExitingFrame( "Testing lightweight checkbox" );
        frame.setLayout( new java.awt.FlowLayout() );
        frame.setBackground( java.awt.Color.blue );
        frame.add( new Checkbox( "Lightweight Checkbox" ) );
        frame.add( new RoundButton( "Lightweight RndBtn" ) );
        frame.add( new FlatButton( "Lightweight Flat Btn" ) );
        frame.setSize( 600, 500 );
        frame.setVisible( true );
    }
}
