
package dph.awt.lightweight;


/**
 * Abstract class that implements 3 basic behaviors: 1) Calls <code>super.addNotify</code> 
 * then tries to determine the size of this component.  If user set 
 * size with call to <code>setSize</code>, then will use that size, else 
 * will call figureMySize to determine size.  Waiting till after 
 * <code>super.addNotify</code> ensures there will be a graphics context 
 * for call to <code>figureMySize</code>.  2) Overrides <code>update</code>
 * to omit screen-erasing and only call <code>paint</code>.  3) Overrides
 * <code>enable/disable</code> to call repaint after change to enable state. 
 *
 * @author DPH
 * @version 1.0
 *
 * @see java.awt.Component
 */
public abstract class CommonBase extends java.awt.Component {


    /**
     * Force a call to the AWT Component constructor, then enable event handling
     * for mouse events.
     *
     * @see java.awt.Component
     */
    public CommonBase() {
        super();
        enableEvents( java.awt.AWTEvent.MOUSE_EVENT_MASK );
    }
    
    
    
    /**
     * Waits till after <code>super.addNotify</code> to ensure there will be a 
     * graphics context for call to <code>figureMySize</code>, then determines the
     * component's size.
     *
     * @see java.awt.Component.addNotify
     * @see java.awt.Component.setSize
     * @see figureMySize
     */
    public void addNotify() {
        super.addNotify();
        java.awt.Dimension d = getSize();
        if( d.width == 0  ||  d.height == 0 ) {
            setSize( this.figureMySize( this.getGraphics() ) );
        }
    }
    
    
    /**
     * Overridden to call repaint in case change of enabled
     * state affects the look of the lightweight component.
     *
     * @param state     new state of the enabled property
     */
    public void setEnabled( boolean state ) {
        super.setEnabled( state );
        repaint();
    }
  
  
    /**
     * Overridden to call paint without clearing the
     * screen so as to reduce flicker.
     *
     * @param g     Graphics context of this component
     */
    public void update( java.awt.Graphics g ) {
        paint( g );    
    }

    
    /**
     * Overridden to return the preferred size of the component.
     *
     * @return  Dimension representing the component's minimum size
     */
    public java.awt.Dimension getMinimumSize() {
        return getPreferredSize();
    }
    
    
    /**
     * Overridden to return either the user-established size
     * set by calling <code>setSize</code>, or the size as
     * determined by calling <code>figureMySize</code>.  Must
     * be called after component has had a graphics
     * context established for it.
     *
     * @see figureMySize
     * @see addNotify
     *
     * @return Dimension representing the component's preferred size
     */
    public java.awt.Dimension getPreferredSize() {
        if( getSize().width != 0  &&  getSize().height != 0 )
            return getSize();
        else
            return figureMySize( getGraphics() );
    }

    
    /**
     * Abstract method which must be defined in child classes to determine
     * their sizes.
     *
     * @see dph.awt.lightweight.Checkbox.figureMySize
     * @see dph.awt.lightweight.FlatButton.figureMySize
     * @see dph.awt.lightweight.RoundButton.figureMySize
     * @see dph.awt.lightweight.Spinner.figureMySize
     * @see getPreferredSize
     *
     * @param g     Graphics context of this component
     */
    protected abstract java.awt.Dimension figureMySize( java.awt.Graphics g );
}
