
package dph.awt.lightweight;

import java.awt.*;
import java.awt.event.*;

/**
 * Defines a Frame which implements off-screen Image buffering, and which
 * adds a WindowAdapter to listen to windowClosing events so that closing
 * when a user clicks on the window exit button is the default behavior.
 */
public class DoubleBufferedExitingFrame extends Frame {
    Image offScrImg;
    Graphics bufferedG;
    WindowAdapter closer;
    
    /**
     * Constructs by adding the <code>WindowAdapter</code> class which implements 
     * the default behavior for the windowClosing events.
     */
    public DoubleBufferedExitingFrame( String title ) {
        super( title );        
        this.addCloser();
    }
    
    protected void addCloser() {
        class WindowCloser extends WindowAdapter {
            public void windowClosing( WindowEvent evt ) {
                System.exit( 0 );
            }
        }
        this.closer = new WindowCloser();
        this.addWindowListener( closer );
    }
    
    
    public void setWindowCloser( WindowAdapter newCloser ) {
        this.removeWindowListener( this.closer );
        this.closer = newCloser;
        this.addWindowListener( this.closer );
    }
    
    /**
     * Invalidates the off-screen Image data member by nulling it whenever 
     * this component gets invalidated.
     */
    public void invalidate() {
        super.invalidate();
        offScrImg = null;
    }

    /**
     * Calls <code>paint</code> and does no other work.
     */
    public void update( Graphics g ) {
        paint( g );
    }
    
    /**
     * Tests for existence of the off-screen Image buffer and tries to
     * create one if it doesn't exist.  Returns without painting if
     * can't get the off-screen Image buffer, otherwise gets the Graphics
     * context associated with the Image, fills it with the background color, 
     * then calls <code>super.paint</code> on it.  Finally, disposes of the
     * Graphics context.
     */
    public void paint( Graphics g ) {
        if( offScrImg == null ) {
            offScrImg = this.createImage( this.getSize().width, this.getSize().height );
            bufferedG = offScrImg.getGraphics();
        }    
        if( offScrImg == null  ||  bufferedG == null )
            return;
        bufferedG.setColor( this.getBackground() );
        bufferedG.fillRect( 0, 0, this.getSize().width, this.getSize().height );
        super.paint( bufferedG );
        g.drawImage( offScrImg, 0, 0, null );
        //bufferedG.dispose();
    }
}