
package dph.awt.lightweight;

import dph.util.*;

/**
 * A class which extends ButtonBase and which displays with a flat appearance, optionally
 * with images or with a textual label or both.  The button pops up when the mouse
 * scrolls over it.
 *
 * @author DPH
 * @version 1.0
 *
 * @see ButtonBase
 */

public class FlatButton extends ButtonBase {

   // -----[ Member Data ]----------
   private String imageURL;               // filename URL for image
   private java.awt.Image image;          // the image to paint
   private boolean mouseIsIn=false;       // has the mouse entered the button
   private boolean mouseIsDown=false;     // has the user pressed the mouse button



   /**
    * Constructs a FlatButton with a textual label.
    *
    * @param label      the String text forming the label of the button
    */
   public FlatButton ( String label ) {
        super( label );
   }



   /**
    * Constructs a FlatButton with an image as a label.
    * Ensures image is loaded before using it.
    *
    * @param image      the Image to use as the label
    *
    * @see setup
    */
   public FlatButton ( java.awt.Image image ) {
        super( "" );
        this.image = image;
        this.setup();
   }



   /**
    * Constructs a FlatButton with the label formed out of both a String and an Image.
    * Ensures image is loaded before using it.
    *
    * @param  label         the String forming the textual part of the label
    * @param  image         the Image forming the graphical part of the label
    *
    * @see  setup
    */
   public FlatButton ( String label, java.awt.Image image ) {
        super( label );
        this.image = image;
        this.setup();
   }



    /**
     * Sets the Image portion of this FlatButton's label.
     * Ensures image is loaded before using it.
     *
     * @param  image    the Image to use on this FlatButton's label
     *
     * @see ensureImageLoaded
     */
    public void setImage( java.awt.Image image ) {
        this.image = image;
        if( image.getWidth(this) == -1  ||  image.getHeight(this) == -1 )
            this.ensureImageLoaded( image );
        invalidate();
        repaint();
    }



    /**
     * Sets the String to use as the URL of an Image which will be used on this
     * FlatButton's label. Ensures image is loaded before using it.
     *
     * @param  imageURL     the String forming the URL of an Image to use as label
     *
     * @see ensureImageLoaded
     */
    public void setImageURL( String imageURL ) {
        imageURL = imageURL;
        java.awt.Toolkit toolkit = this.getToolkit();
        image = toolkit.getImage( imageURL );
        this.ensureImageLoaded( image );
        invalidate();
        repaint();
    }


   /**
    * Handle any mouse activity on this FlatButton, especially processing mouse
    * entered, mouse exited, mouse pressed, and mouse released. Pass all other 
    * activity up the inheritance chain.
    *
    *@param   evt    the MouseEvent to process
    */
   public void processMouseEvent( java.awt.event.MouseEvent evt ) {
       if( isEnabled() ) {
         switch( evt.getID() ) {
         case java.awt.event.MouseEvent.MOUSE_ENTERED:
            mouseIsIn = true;
            repaint();
            break;
         case java.awt.event.MouseEvent.MOUSE_EXITED:
            mouseIsIn = false;
            repaint();
            break;
         case java.awt.event.MouseEvent.MOUSE_PRESSED:
            mouseIsDown = true;
            repaint();
            break;
         case java.awt.event.MouseEvent.MOUSE_RELEASED:
            mouseIsDown = false;
            repaint();
            break;
         }
      }
      super.processMouseEvent( evt );
   }



   /**
    * Gets the optimum size for this FlatButton, which will be calculated to encompass
    * any textual or image-based label with a few pixels extra on all four sides.
    *
    * @param g  the Graphics context to use in figuring size
    *
    * @return  the Dimension representing this FlatButton's size
    */
   protected java.awt.Dimension figureMySize( java.awt.Graphics g ) {
      int width = 40;
      int height = 20;

      // Calculate actual dimensions
      if( g != null ) {
         java.awt.FontMetrics fm = g.getFontMetrics();

         // Calculate title dimensions
         if( label != null ) {

            // Get more accurate height
            height = fm.getHeight() + 6;

            // Calculate label width
            width = Util.pixelsWideForString( label, fm );
            width += 6; // additional spacing around string and for border
         }

         // Calculate image dimensions
         if( this.image != null ) {
            int imageWidth = image.getWidth( this );        // image width
            imageWidth += 4;  // additional spacing around image and for border
            width = width > imageWidth ? width : imageWidth;
            int imageHeight = image.getHeight( this );         // image height
            imageHeight += 4;  // additional spacing around image and for border
            height = height > imageHeight ? height : imageHeight;
         }
      }
      return new java.awt.Dimension( width, height );
   }


    /**
     * Paints label over image, handles popping up, depressing, orientation and title position
     * issues.  Will not draw if this FlatButton has not been added to the screen.
     *
     * @param  g  the Graphics context to use in painting this FlatButton
     */
   public void paint( java.awt.Graphics g ) {
      if( isShowing() ) {
         if( !isEnabled() ) {
             mouseIsIn=false;
             mouseIsDown=false;
         }
          
         java.awt.Dimension mySize = getSize();
         
         // Color using desired background color
         g.setColor( getBackground() ); 
         g.fillRect( 0, 0, mySize.width-1, mySize.height-1 );

         // Draw a black outline around button
         g.setColor( java.awt.Color.black );
         g.drawRect( 0, 0, mySize.width-1, mySize.height-1 );

         java.awt.Color topLeft1=null;
         java.awt.Color topLeft2=null;
         java.awt.Color botRight1=null;
         java.awt.Color botRight2=null;
         boolean drawBorders = false;
         int padding = 0;
         if( mouseIsIn  &&  mouseIsDown ) {
            // Color for when mouse is over this and button has been depressed
            topLeft1 = getBackground().darker().darker();
            botRight1 = getBackground().brighter().brighter();
            drawBorders = true;
            padding = 1;
         }
         else if( mouseIsIn ){
            // Color for when mouse is over this button but not depressed
            topLeft1 = java.awt.Color.white;
            topLeft2 = getBackground().brighter();
            botRight1 = java.awt.Color.darkGray;
            botRight2 = getBackground().darker();
            drawBorders = true;
            padding = -1;
         }

            // Draw a unique border if one is needed because mouse has entered this
            // or user has pressed mouse button over this
            if( drawBorders ) {
            g.setColor( topLeft1 );
            g.drawLine( 0, 0, mySize.width, 0 );
            g.drawLine( 0, 0, 0, mySize.height-1 );
            g.setColor( botRight2 );
            g.drawLine( 1, mySize.height-2, mySize.width-2, mySize.height-2 );
            g.drawLine( mySize.width-2, mySize.height-2, mySize.width-2, 1 );
            
            // Give depressed look to button
            if( mouseIsDown ) {
                g.setColor( java.awt.Color.black );
                g.drawLine( 1, 1, mySize.width, 1 );
                g.drawLine( 1, 1, 1, mySize.height );
                g.setColor( java.awt.Color.white );
                g.drawLine( 1, mySize.height-1, mySize.width-1, mySize.height-1 );
                g.drawLine( mySize.width-1, 1, mySize.width-1, mySize.height-1 );
            }
         }


         // Draw image first
          int xPos = 2;
          int yPos = 2;
         java.awt.FontMetrics fm = g.getFontMetrics();
         if( image != null ) {

             // Center image if label is wider or higher than image
              if( mySize.width > (image.getWidth( this )+4) ){
                  xPos = (int)(((float)mySize.width - (float)image.getWidth( this )+4f) / 2f);
              }

             g.drawImage( image, xPos+padding, yPos+padding, this );
         }

         // Draw label over image
         if( label != null  &&  label.length() > 0 ) {

            // Draw title in foreground color
            if( isEnabled() )
                g.setColor( this.getForeground() );
            else
                g.setColor( java.awt.Color.gray );

                // Center label if image is wider than label
                if( mySize.width > Util.pixelsWideForString( label, fm ) +4 )
                    xPos = (int)(((float)mySize.width - (float)Util.pixelsWideForString( label, fm ) ) / 2f);

             yPos = fm.getHeight() /2 ;
             g.drawString( label, xPos+padding, mySize.height-yPos+padding );
         }
      }
   }




    // -----------------------------------------
    // -----[ Private Helper Methods ]----------
    // -----------------------------------------
    /**
     * Makes sure that any image associated with this FlatButton is loaded
     * so that its width and height have valid values.  Uses the MediaTracker
     * class to do so.
     *
     * @param  image   the Image object to be sure to load
     */
    private final void ensureImageLoaded( java.awt.Image image ) {
        try {
            java.awt.MediaTracker tracker = new java.awt.MediaTracker( this );
            tracker.addImage( image, 0 );
            tracker.waitForID( 0 );
            Util.assert( tracker.statusID(0, false) == java.awt.MediaTracker.COMPLETE, "tracker.statusID(0, false) == MediaTracker.COMPLETE" );
        } catch( InterruptedException ex ) {
            Util.debug( "Image loading interrupted" );
        }
    }


    /**
     * Do initialization with image before calculating size of FlatButton.
     */
    private final void setup() {
      if( image != null  &&  (image.getWidth(this)==-1  ||  image.getHeight(this)==-1) )
          this.ensureImageLoaded( image );
    }



   // -------------------------------
   // -----[ Unit Testing ]----------
   // -------------------------------
   public static void main( String[] args ) {
      java.awt.Image img = java.awt.Toolkit.getDefaultToolkit().getImage( "c:/dev/java/dph/java/awt/images/ButtonImage.gif" );
      java.awt.Frame f = new java.awt.Frame( "Test FlatButton" );
      f.setSize( 500, 450 );
      f.setBackground( java.awt.Color.blue );
      f.setLayout( new java.awt.FlowLayout() );

      final FlatButton btn1 = new FlatButton( "Exit" );
      FlatButton btn2 = new FlatButton( "This is a Button" );
      FlatButton btn3 = new FlatButton( "Hi Button" );
      FlatButton btn4 = new FlatButton( img );
      FlatButton btn5 = new FlatButton( "BUTTON 5", img );

      btn2.setImage( img );
      btn3.setImageURL( "c:/dev/java/dph/java/awt/images/BigBall.gif" );

      f.add( btn1 );
      f.add( btn2 );
      f.add( btn3 );
      f.add( btn4 );
      f.add( btn5 );

      f.addWindowListener( new java.awt.event.WindowAdapter() {
         public void windowClosing( java.awt.event.WindowEvent we ) {
            System.exit( 0 );
         }
      });


        // MouseHandler to print button label
      class MouseHandler1 extends java.awt.event.MouseAdapter {
         public void mouseClicked( java.awt.event.MouseEvent me ) {
             if( me.getSource() == btn1 )
                System.exit( 0 );
         }
      }


      btn1.addMouseListener( new MouseHandler1() );
      btn2.addMouseListener( new MouseHandler1() );
      btn3.addMouseListener( new MouseHandler1() );
      btn4.addMouseListener( new MouseHandler1() );

      f.setVisible( true );
   }

 }
