
package dph.awt.lightweight;



/**
 * A class that encapsulates a lightweight text Label, and allows for a
 * rectangular border around the text.  This class does not generate any
 * events of its own.
 *
 * @author DPH
 * @version 1.0
 *
 * @see java.awt.Label
 */
public class Label extends CommonBase {

    String text;
    boolean showBorder;
    

    /**
     * Constructs a Label with an empty title and defaults whether there should be 
     * a rectangular border around the text to false.
     *
     * @see Label( String text )
     */
    public Label() {
        this( "", false );
    }


    /**
     * Constructs a Label with a title and defaults whether there should be 
     * a rectangular border around the text to false.
     *
     * @param text          the text label of the Label
     *
     * @see Label( String text, boolean showBorder )
     */
    public Label( String text ) {
        this( text, false );
    }


    /**
     * Constructs a Label with a title and a variable indicating whether
     * or not there should be a rectangular border around the text.
     *
     * @param text          the text label of the Label
     * @param showBorder    the state of whether there should be a border
     *
     * @see paint
     */
    public Label( String text, boolean showBorder ) {
        super();
        this.text = text;
        this.showBorder = showBorder;
    }
    


    /**
     * Sets the textual label of the Label.  Calls <code>invalidate</code>
     * in case the label changed size and should be layed out again.
     *
     * @param text          the text label of the Label
     */
    public synchronized void setText( String newText ) {
        this.text = newText;
        invalidate();
        this.repaint();
    }
    
    

    /**
     * Returns the textual label of the Label.
     *
     * @return the text label of the Label
     */
    public String getText() {
        return text;
    }
    
    
    /**
     * Sets the state of whether or not there should be a rectangular
     * border around the Label.
     *
     * @param state          the state of whether there should a border
     */
    public void setShowBorder( boolean state ) {
        showBorder = state;
    }
    
    
    /**
     * Returns the state of whether there should a border around the Label.
     *
     * @return a boolean representing whether there should a border
     */
    public boolean getShowBorder() {
        return showBorder;
    }



    /**
     * Gets the optimum size for this Label.
     *
     *@param g      the graphics context for this Label
     *
     *@return  A Dimension representing the preferred size of this Label
     */
    protected java.awt.Dimension figureMySize( java.awt.Graphics g ) {
        if( text != null   &&  g != null ) {
            int stringWidth = dph.util.Util.pixelsWideForString( text, g.getFontMetrics() );
            int height = g.getFontMetrics().getHeight();
            return new java.awt.Dimension( stringWidth + 6, height + 5 );
        }
        return getSize();
    }


    
    /**
     * Draws this Label, including a rectangular border if there should be one.
     * Draws it only to fit the text of the Label, so calling <code>setSize</code>
     * could render it inappropriately.
     */
    public void paint( java.awt.Graphics g ) {
        
       g.setColor( java.awt.Color.black );
       if( showBorder ) {
          g.drawRect( 0, 0, getSize().width, getSize().height );
       }
       int y = (getSize().height / 2 + g.getFontMetrics().getHeight()/2) -2;
       g.drawString( text, 3, y );
    }
    

    // -----[ Unit Test ]-----
    public static void main( String[] argsw ) {
        DoubleBufferedExitingFrame frame = new DoubleBufferedExitingFrame( "Testing lightweight Label" );
        frame.setLayout( new java.awt.FlowLayout() );
        frame.setBackground( java.awt.Color.blue );
        frame.add( new Label( "Lightweight Label" ) );
        frame.setSize( 600, 500 );
        frame.setVisible( true );
    }
}
