
package dph.awt.lightweight;


/**
 * Defines a numeric Spinner class.  Listeners can register to listen
 * for ActionChanged Events on the Spinner.
 *
 * @author      Perelman-Hall, David
 */
public class Spinner extends CommonBase {

    protected int value;
    protected int minValue = Integer.MIN_VALUE, maxValue = Integer.MAX_VALUE;
    protected java.awt.event.ActionListener actionListener;
    protected java.awt.Polygon upArrow=new java.awt.Polygon();
    protected java.awt.Polygon dnArrow=new java.awt.Polygon();
    protected java.awt.Polygon upPressedArrow=new java.awt.Polygon();
    protected java.awt.Polygon dnPressedArrow=new java.awt.Polygon();
    protected boolean upPressed, dnPressed;
    protected boolean rollsOver;
    protected long oldMouseClickTime;
    private String actionCommand = "spin";

    private static long DELAY_PERIOD = 150;


    /**
     * Constructs a Spinner.
     */
    public Spinner() {
        super();
    }
    


    // -----[ Accessor Methods ]-----

    /**
     * Sets the value of the Spinner the newValue argument.
     *
     * @param  newValue   the int value to set the Spinner to
     */
    public synchronized void setValue( int newValue ) {
        this.value = newValue;
        this.repaint();
    }


    /**
     * Gets the value of the Spinner.
     *
     * @return  the int value of the Spinner
     */
    public int getValue() {
        return value;
    }
    
    /**
     * Sets the minimum value of the Spinner the value argument.
     *
     * @param  value   the minimum int value to set the Spinner to
     */
    public synchronized void setMinValue( int value ) {
        setValue( minValue = value );
    }
    
    /**
     * Gets the minimum value of the Spinner.
     *
     * @return  the int minimum value of the Spinner
     */
    public int getMinValue() {
        return minValue;
    }


    /**
     * Sets the maximum value of the Spinner the value argument.
     *
     * @param  value   the maximum int value to set the Spinner to
     */
    public synchronized void setMaxValue( int value ) {
        maxValue = value;
    }
    
    /**
     * Gets the maximum value of the Spinner.
     *
     * @return  the int maximum value of the Spinner
     */
    public int getMaxValue() {
        return maxValue;
    }

    /**
     * Sets whether the Spinner rolls over when the user spins
     * it past its minimum or maximum values.
     *
     * @param  state   the value of whether the Spinner rolls over or not
     */
    public void setRollsOver( boolean state ) {
        rollsOver = state;
    }
    
    /**
     * Gets the value of whether the Spinner rolls over or not.
     *
     * @return  the boolean value of whether the Spinner rolls over or not
     */
    public boolean getRollsOver() {
        return rollsOver;
    }

    /**
     * Sets the actionCommand of the Spinner.
     *
     * @param  newActionCommand  the String actionCommand of the Spinner
     */
    public void setActionCommand( String newActionCommand ) {
        actionCommand = newActionCommand;
    }
    
    /**
     * Gets the actionCommand of the Spinner.
     *
     * @return  the String actionCommand of the Spinner
     */
    public String getActionCommand() {
        return actionCommand;
    }
    
    
    // -----[ add and remove actionListeners with AWTEventMulticaster ]-----
   
    
    /**
     * Adds the ActionListener as a registered listener of ActionEvents on
     * the Spinner.
     *
     * @param al         the ActionListener to add
     */
    public void addActionListener( java.awt.event.ActionListener al ) {
        actionListener = java.awt.AWTEventMulticaster.add( actionListener, al );
    }


    /**
     * Removes the ActionListener as a registered listener of ActionEvents on
     * the Spinner.
     *
     * @param al         the ActionListener to remove
     */
    public void removeActionListener( java.awt.event.ActionListener al ) {
        actionListener = java.awt.AWTEventMulticaster.remove( actionListener, al );
    }


	/**
	 * Gets the optimum size for this Spinner.
	 *
	 * @param  g   the Graphics context to use for figuring the Spinner's size
	 *
	 * @return  the Dimension representing the Spinner's size
	 */
	protected java.awt.Dimension figureMySize( java.awt.Graphics g ) {
	    java.awt.Dimension mySize = new java.awt.Dimension( 40, 30 );
	    
	    // Unpressed polygons
		upArrow.addPoint( mySize.width-8, 4 );
		upArrow.addPoint( mySize.width-15, mySize.height/2 );
		upArrow.addPoint( mySize.width-2, mySize.height/2 );
		
		dnArrow.addPoint( mySize.width-8, mySize.height-5 );
		dnArrow.addPoint( mySize.width-14, mySize.height/2+1 );
		dnArrow.addPoint( mySize.width-2, mySize.height/2+1 );
		
		// Pressed polygons
		upPressedArrow.addPoint( mySize.width-8, 5 );
		upPressedArrow.addPoint( mySize.width-14, mySize.height/2-1 );
		upPressedArrow.addPoint( mySize.width-3, mySize.height/2-1 );
		
		dnPressedArrow.addPoint( mySize.width-8, mySize.height-4 );
		dnPressedArrow.addPoint( mySize.width-13, mySize.height/2+2 );
		dnPressedArrow.addPoint( mySize.width-3, mySize.height/2+2 );
		return mySize;
	}


    
	/**
	 * Paints the Spinner.
	 *
	 * @param  g   the Graphics context to use for painting the Spinner
	 */
    public void paint( java.awt.Graphics g ) {
        java.awt.Dimension mySize = getSize();
		g.setColor( java.awt.Color.lightGray.brighter() );
		g.drawLine( 0, 0, mySize.width-1, 0 );
		g.drawLine( 0, 0, 0, mySize.height );
		
		g.setColor( java.awt.Color.black );
		g.drawLine( 2, mySize.height-2, mySize.width, mySize.height-2 );
		g.drawLine( 1, mySize.height-1, mySize.width, mySize.height-1 );
		g.drawLine( mySize.width, 0, mySize.width, mySize.height );
		g.drawLine( mySize.width-1, 1, mySize.width-1, mySize.height );
		

		java.awt.Rectangle rect = new java.awt.Rectangle( 5, 5, mySize.height-10, mySize.height-10 );
		
		if( isEnabled() )
		    g.setColor( java.awt.Color.white );
		else    
		    g.setColor( java.awt.Color.gray );
		g.fillRect( rect.x, rect.y, rect.width, rect.height );
		
		g.setColor( java.awt.Color.black );
		g.drawLine( rect.x+1, rect.y+rect.height-1, rect.x+rect.width-1, rect.y+rect.height-1 );
		g.drawLine( rect.x+rect.width-1, rect.y+1, rect.x+rect.width-1, rect.y+rect.height-1 );
		if( !upPressed )
	        g.fillPolygon( upArrow );
	    else    
	        g.fillPolygon( upPressedArrow );
	    if( !dnPressed )    
	        g.fillPolygon( dnArrow );
	    else    
	        g.fillPolygon( dnPressedArrow );
				
		// Draw value
		g.setFont( new java.awt.Font( "SanSerif", java.awt.Font.BOLD, 12 ) );
		java.awt.FontMetrics fm = g.getFontMetrics();
        int stringWidth = dph.util.Util.pixelsWideForString( ""+value, fm );
	    int y = (mySize.height / 2 + fm.getHeight()/2) -1;
	    int x = rect.x + (rect.width - stringWidth) /2;
	    g.setColor( java.awt.Color.black );
	    g.drawString( ""+value, x, y );
    }
    

    /**
     * Helper method used for calling actionPerformed on the ActionListeners
     * for this Spinner.
     */
    private void notifyListeners() {
        if( actionListener != null ) {
            actionListener.actionPerformed( new java.awt.event.ActionEvent (
                this, java.awt.event.ActionEvent.ACTION_PERFORMED, this.getActionCommand()
            ) );
        }
    }
    
    
    /**
     * Increments the value of this Spinner. Tests whether needs to rollover.
     * Notifies listeners when increment occurs.
     */
    public void increment() {
        boolean valueChanged = false;
        if( value < maxValue ) {
            ++value;
            valueChanged = true;
        }
        else if( rollsOver ) {
            value = minValue;
            valueChanged = true;
        }
        if( valueChanged ) {
            notifyListeners();
        }
    }

    
    /**
     * Decrements the value of this Spinner. Tests whether needs to rollover.
     * Notifies listeners when decrement occurs.
     */
    public void decrement() {
        boolean valueChanged = false;
        if( value > minValue ) {
            --value;
            valueChanged = true;
        }
        else if( rollsOver ) {
            value = maxValue;
            valueChanged = true;
        }
        if( valueChanged ) {
            notifyListeners();
        }
    }
    
    
    /**
     * Defines an anonymous nested class for spinning a Thread off 
     * to handle the mouse clicked activity on this Spinner.  The spun-off
     * Tyread calls either increment of decrement.
     */
    public void spin() {
        
        // can try to set flag when spinning and use finalize to unset
        // or can try to catch thread death to unset flag
        class SpinThread extends Thread {
            
            SpinThread() {
                super();
                this.setPriority( Thread.MIN_PRIORITY );
            }
            
            public void run() {
                try {
                    if( upPressed )
                        increment();
                    if( dnPressed )
                        decrement();
                    repaint();
                    Thread.sleep( 1250 );
                    while( upPressed || dnPressed ) {
                        if( upPressed )
                            increment();
                        if( dnPressed )
                            decrement();
                        repaint();    
                        Thread.sleep( 400 );
                    }
                } catch( InterruptedException ex ) {
                    // ignore it
                }
            }
        }
        
        new SpinThread().start();
    }

    
    
   /**
    * Handle any mouse activity on this Spinner, especially processing mouse
    * downs to spin the Spinner and mouse ups to stop the Spinner.
    * Pass all other activity up the inheritance chain.
    *
    *@param   evt    the MouseEvent to process
    */
	public void processMouseEvent( java.awt.event.MouseEvent evt ) {
	    if( isEnabled() ) {
    		switch( evt.getID() ) {
        		case java.awt.event.MouseEvent.MOUSE_PRESSED:
        		    // Do not let user start up too many spin threads in a short time
        		    long newMouseClickTime = System.currentTimeMillis();
        		    if( newMouseClickTime - oldMouseClickTime > DELAY_PERIOD ){
            		    if( upArrow.contains( new java.awt.Point(evt.getX(), evt.getY()) ) ) {
            		        upPressed = true;
        		            spin();
            		    }    
            		    else if( dnArrow.contains( new java.awt.Point(evt.getX(), evt.getY()) ) ) {
            		        dnPressed = true;
        		            spin();
            		    }
        		    }
        		    oldMouseClickTime = newMouseClickTime;
        		    break;
        		case java.awt.event.MouseEvent.MOUSE_RELEASED:
        		    if( upPressed == true )
        		        upPressed = false;
        		    else if( dnPressed == true )
        		        dnPressed = false;
        		    repaint();
        		    break;
    		}
		}
		super.processMouseEvent( evt );
	}


    // -----[ Unit Test ]-----
    public static void main( String[] argsw ) {
        DoubleBufferedExitingFrame frame = new DoubleBufferedExitingFrame( "Testing lightweight Spinner" );
        frame.setLayout( new java.awt.FlowLayout() );
        frame.setBackground( java.awt.Color.blue );
        Spinner spinner = new Spinner();
        spinner.setMaxValue( 222 );
        spinner.setMinValue( -10 );
        frame.add( spinner );
        frame.add( new RoundButton( "Lightweight RndBtn" ) );
        frame.add( new FlatButton( "Lightweight Flat Btn" ) );
        frame.setSize( 600, 500 );
        frame.setVisible( true );
    }
}