
package dph.awt.lightweight;

import dph.util.Util;
import java.awt.*;
import java.awt.event.*;

/**
 * Defines a simple tabbed container with a simple interface that allows clients to
 * add tabbed contents with each tab taking the form of:
 * <ul>
     * <li>a <code>Panel</code> containing the components for that tab
     * <li>a <code>String</code> label to appear in the tab for the panel
     * </ul>
     * <p>
 * For each tab to be added the user creates a Panel and then adds the panel
 * along with the tab's label using the <code>addTabPanel</code> method.  The size
 * of this TabPanel container accommodates both the row of tabs and the panel area
 * below the tabs.  Tabs can appear only across the top.
 *
 * @author      Perelman-Hall, David
 */
public class TabPanel extends Container {

    String[] tabLabels;     // the tabs' labels
    Rectangle[] tabs;       // the screen regions of the tabs
    int selectedTab = -1;   // the currently selected tab
    Panel currentPanel = new DoubleBufferedPanel();
    java.util.Hashtable cards = new java.util.Hashtable();


    /**
     * Constructs the TabPanel class by enabling mouse events, nulling out the
     * current Layout, then adding the currentPanel to this container, but defers
     * positioning or sizing it until the <code>addNotify</code> call.
     */
    public TabPanel() {
        enableEvents( AWTEvent.MOUSE_EVENT_MASK );
        this.add( currentPanel );
        cards.put( "", currentPanel );
    }


    /**
     * Positions and sizes the currentPanel so it is just below the row of tabs.
     */
    public void addNotify() {
        super.addNotify();
        currentPanel.setLocation( 2, 27 );
        currentPanel.setSize( this.getPanelSize() );
    }


    /**
     * Adds a tab to end of row of tabs and adds its associated Panel.  Calls
     * <code>invalidate</code> to layout the container again.
     *
     * @param tabLabel      the label to be drawn on the tab
     * @param panel         the panel to be drawn when this tab is selected
     */
    public synchronized void addTabPanel( String tabLabel, Panel panel ) {
        cards.put( tabLabel, panel );
        this.addTab( tabLabel );
        invalidate();
        repaint();
    }


    /**
     * Adds a tab to row of tabs at specified tabNum position,
     * and adds its associated Panel.  Valid tab number range is dependent on
     * the current selected tab.  If no tab is the current selected tab then the
     * only valid value for <code>tabNum</code> is 0.  Else the only valid values range from
     * 0 to highest tab position + 1;
     *
     * @param tabLabel      the label to be drawn on the tab
     * @param panel         the panel to be drawn when this tab is selected
     * @param tabNum        the zero-based position of the tab
     *
     * @throws              IllegalArgumentException if <code>tabNum</code> is not valid
     */
    public synchronized void addTabPanel( String tabLabel, Panel panel, int tabNum )
    throws IllegalArgumentException {

        // See if tabNum is valid
        if( selectedTab == -1  ) {
            if( tabNum != 0 )
                throw new IllegalArgumentException( tabNum + " out of range" );

            this.addTabPanel( tabLabel, panel );
        }
        else {
            if( tabNum > tabLabels.length  ||  tabNum < 0 )
                throw new IllegalArgumentException( tabNum + " out of range" );

            // load up array of existing panels
            java.util.Enumeration labels = cards.keys();
            Component[] oldPanels = new Component[cards.size()];
            int idx=0;
            while( labels.hasMoreElements() )
                oldPanels[idx++] = (Component)cards.get( labels.nextElement() );

            // new Hashtable to hold panels
            cards = new java.util.Hashtable();

            // New array of labels
            String[] newLabels = new String[ tabLabels.length+1 ];

            // Store panels with lower number than tabNum
            for( int panelNum=0; panelNum<tabNum; ++panelNum ) {
                cards.put( tabLabels[panelNum], oldPanels[panelNum] );
                newLabels[panelNum] = tabLabels[panelNum];
            }

            // Add the new panel
            newLabels[tabNum] = tabLabel;
            cards.put( tabLabel, panel );

            // Add rest of existing panels
            for( int panelNum=tabNum; panelNum<oldPanels.length; ++panelNum ) {
                newLabels[panelNum+1] = tabLabels[panelNum];
                cards.put( tabLabels[panelNum], oldPanels[panelNum] );
            }
            tabLabels = newLabels;
            tabs = new Rectangle[ tabLabels.length ];
            invalidate();
            repaint();
        }
    }


    /**
     * Removes a tab and its panel using the label as the value of the tab to be removed.
     *
     * @param tabLabel      the label of the panel to be removed
     */
    public synchronized void removeTabPanel( String tabLabel ) {
        for( int i=0; i<tabLabels.length; ++i ) {
            if( tabLabel.equals( tabLabels[i] ) ) {
                this.removeTabPanel( i );
                break;
            }
        }
    }


    /**
     * Removes a tab and its panel using the tab number as the index of the tab to be removed.
     *
     * @param tabNum      the index of the panel to be removed
     */
    public synchronized void removeTabPanel( int  tabNum ) {
        if( tabNum < tabLabels.length ) {

            // New array of labels
            String[] tmpLabels = new String[tabLabels.length -1];
            int idx = 0;

            // Store labels except tabNum
            for( int i=0; i<tabLabels.length; ++i ) {
                if( i != tabNum )
                    tmpLabels[idx++] = tabLabels[i];
                else {
                    // remove this one from the cards
                    cards.remove( tabLabels[i] );
                    // see if have to decrement selectedTab number
                    if( selectedTab >= tabNum )
                        --selectedTab;
                }
            }
            tabLabels = tmpLabels;
        }
        setSelectedTab( selectedTab );
        invalidate();
    }


    /**
     * Adds the tab's label to the String[] of tab labels, and increments the size of
     * the <code>Rectangle</code> array.  Rectangle is not actaully created until this
     * TabPanel is drawn. This is a private helper method.
     *
     * @param label         the label to be drawn for a tab
     */
    private void addTab( String label ) {
        // Add space in array for the Rectangle
        // but populate it during draw
        Rectangle[] tmpRects = new Rectangle[ tabs==null?1:tabs.length+1 ];
        if( tabs != null )
            System.arraycopy( tabs, 0, tmpRects, 0, tabs.length );
        tabs = tmpRects;

        // Add the String
        String[] tmpLabels = new String[ tabLabels==null?1:tabLabels.length+1 ];
        int idx = 0;
        if( tabLabels != null ) {
            System.arraycopy( tabLabels, 0, tmpLabels, 0, tabLabels.length );
            idx = tabLabels.length;
        }
        tmpLabels[idx] = label;
        tabLabels = tmpLabels;
    }



    /**
     * Looks for a mouse event indicating user clicked the mouse.  When a mouse press is
     * detected, determines which tab was clicked on, if any, and then calls
     * <code>setSelectedTab</code> to cause the new tab to become the selected one.
     *
     * @param evt         the MouseEvent to examine to see if it is a mouse press
     *
     * @see setselectedTab
     */
    public void processMouseEvent( MouseEvent evt ) {
        switch( evt.getID() ) {
            case MouseEvent.MOUSE_PRESSED:
                if( tabs != null ) {
                    for( int tabNum=0; tabNum<tabs.length; ++tabNum ) {
                        if( tabs[tabNum].contains( new Point( evt.getX(), evt.getY() ) ) ) {
                            if( tabNum != selectedTab ) {
                                setSelectedTab( tabNum );
                            }
                            break;
                        }
                    }
                }
                break;
        }
        super.processMouseEvent( evt );
    }


    /**
     * Returns an int designating which tab is the currently-selected tab by index.
     *
     * @return the index of the selectedTab
     */
    public int getSelectedTab() {
        return selectedTab;
    }


    /**
     * Returns the currently selected tab label.
     *
     * @return the label of the selectedTab
     */
    public String getSelectedTabLabel() {
        return selectedTab == -1 ? null : tabLabels[selectedTab];
    }



    /**
     * Sets the label of the tab indicated by tabNum to the new String label.
     *
     * @param tabNum         the index of the tab to receive the new label
     * @param newLabel       the new String value to set the label to
     */
    public void setSelectedTabLabel( int tabNum, String newLabel ) {
        if( tabNum < tabLabels.length )
            tabLabels[tabNum] = newLabel;
        repaint();
    }


    /**
     * Position components in a Panel which has a null layout.  This is a hack
     * which sets a null layout to FlowLayout, then nulls the layout from FlowLayout
     * back to null layout, and then calls setLocation to be sure to position
     * Components correctly.  Without this, if a user built a Panel with null
     * layout, its components weren't correctly positioned in the TabPanel.
     */
    private void validateNullLayoutContainer() {
        if( selectedTab > -1 ) {
            Component c = (Component)cards.get(tabLabels[selectedTab]);
            if( c instanceof Container ) {
                Container container = (Container) c;
                if( container.getLayout() == null ) {

                    // Get the Components
                    Component[] cc = container.getComponents();

                    // Record their positions
                    Point[] pList = new Point[ cc.length ];
                    for( int i=0; i<cc.length; ++i ) {
                        pList[i] = cc[i].getLocation();
Util.out( cc[i] + " has location " + pList[i] );
                    }

                    // Diddle with the layout of the Container
                    container.setLayout( new FlowLayout() );
                    container.setLayout( null );

                    // Re-set the locations of the Components
                    for( int i=0; i<cc.length; ++i ) {
                        cc[i].setLocation( pList[i] );
                    }
                }
            }
        }
    }


    /**
     * Sets the tab indicated by tabNum to be the currently-selected tab.
     *
     * @param tabNum         the index of the tab to be selected
     */
    public void setSelectedTab( int tabNum ) {
        if( tabNum >= 0  &&  tabNum < tabLabels.length ) {
            this.remove( currentPanel );
            selectedTab = tabNum;
            currentPanel = (Panel)cards.get( tabLabels[tabNum] );
            this.add( currentPanel );
            currentPanel.setLocation( 2, 27 );
            currentPanel.setSize( this.getPanelSize() );
            validateNullLayoutContainer();
            currentPanel.doLayout();
        }
        repaint();
    }


    /**
     * Returns the Dimension of the Panel area of the TabPanel.
     *
     * @return  a Dimension representing the size of the Panel portion of the TabPanel
     */
    public Dimension getPanelSize() {
        return new Dimension( this.getSize().width-4, this.getSize().height-29 );
    }


    /**
     * Returns the Dimension of the TabPanel.
     *
     * @return  a Dimension representing the size of the TabPanel
     */
    public Dimension getPreferredSize() {
        if( getSize().width < 45  ||  getSize().height < 35 )
            return new Dimension( 45, 35 );
        return getSize();
    }


    /**
     * Omit the default screen-clearing bahvior of update.
     *
     * @param g         the Graphics context to use for painting
     */
    public void update( Graphics g ) {
        paint( g );
    }


    /**
     * Paint the Panel portion then the Tab portion of the TabPanel.
     *
     * @param g         the Graphics context to use for painting
     */
    public void paint( Graphics g ) {
        super.paint( g );
        this.paintTabPanel( g );
        this.paintTabs( g );
    }


    /**
     * Paint the Panel portion of the TabPanel.
     *
     * @param g         the Graphics context to use for painting
     */
    private void paintTabPanel( Graphics g ) {
        // Find out my size
        Dimension d = this.getSize();

        // Account for tabs
        int x = 0;
        int y = 25;

        // My color
        Color interior = getBackground();

        // draw the square panel
        g.setColor( interior );
        g.fillRect( 0, 0, d.width, d. height );

        // draw the top border
        g.setColor( Color.lightGray );
        g.drawLine( x, y, d.width, y );
        g.drawLine( x, y, x, d.height );
        g.drawLine( x, y+1, d.width-1, y+1 );
        g.drawLine( x+1, y, x+1, d.height-1 );

        // draw the bottom border
        g.setColor( Color.darkGray );
        g.drawLine( x+1, d.height-1, d.width, d.height-1 );
        g.drawLine( x+2, d.height-2, d.width, d.height-2 );
        g.drawLine( d.width-1, d.height-1, d.width-1, y+1);
        g.drawLine( d.width-2, d.height-1, d.width-2, y+2);
    }


    /**
     * Paint the Tab portion of the TabPanel.
     *
     * @param g         the Graphics context to use for painting
     */
    private void paintTabs( Graphics g ) {
        Font f = getFont();
        FontMetrics fm = null;
        if( f != null ) {
            fm = getFontMetrics( f );
            if( fm == null )
                return;
            if( tabLabels != null  &&  tabLabels.length > 0  ) {
                int oldW=0, y=0, h=25, xSpacer=10, ySpacer=0;
                for( int tabNum=0; tabNum<tabLabels.length; ++tabNum ) {

                    // Find the starting positon and width of tab
                    int x = oldW;
                    int w = 20 + dph.util.Util.pixelsWideForString( tabLabels[tabNum], fm );

                    // See if working with currently-selected tab
                    if( selectedTab == tabNum ) {

                        // Use a bold font--may have to resize tab for larger bolded font
                        g.setFont( new Font( f.getName(), Font.BOLD, f.getSize()+1 ) );
                        int stringWidth = dph.util.Util.pixelsWideForString( tabLabels[tabNum], getFontMetrics(g.getFont()) );
                        h = 30;
                        y = 0;
                        xSpacer = (w - stringWidth) / 2;
                        if( tabNum==0 )
                            ySpacer = 4;
                        else
                            ySpacer = 2;
                    }

                    // Not currently-selected tab
                    else {
                        g.setFont( f );
                        h = 23;
                        y = 2;
                        xSpacer = 10;
                        ySpacer = 0;
                    }

                    // populate tabs Rectangle array so we can call
                    // contains method on it
                    tabs[tabNum] = new Rectangle( x, y, w, h );


                    // Color in tabs
                    g.setColor( getBackground() );
                    g.fillRoundRect( x, y, w, h, 6, 6 );
                    g.setColor( getForeground() );
                    g.drawString( tabLabels[tabNum], x+xSpacer, 16 );

                    // Border tab top-left
                    g.setColor( Color.lightGray );
                    g.drawLine( x, y+3, x, y+24+ySpacer );
                    g.drawLine( x+3, y, x+w-3, y );
                    g.drawLine( x, y+3, x+3, y );

                    g.drawLine( x+1, y+3, x+1, y+24+ySpacer );
                    g.drawLine( x+3, y+1, x+w-3, y+1 );
                    g.drawLine( x+1, y+3, x+3, y+1 );

                    // Border tab top-right
                    g.setColor( Color.darkGray );
                    g.drawLine( x+w-1, y+3, x+w-1, y+22+ySpacer );
                    g.drawLine( x+w, y+3, x+w, y+22+ySpacer );
                    g.drawLine( x+w, y+3, x+w-3, y );
                    g.drawLine( x+w-2, y, x+w, y+2 );

                    oldW = x + w;
                }
            }
        }
        // Restore the font
        g.setFont( f );
    }


    // -----[ Unit Test ]-----
    public static void main( String[] args ) {
        Frame f = new DoubleBufferedExitingFrame( "Test LightWeight TabPanel" );
        f.setLayout( new FlowLayout() );
        f.setSize( 520, 490 );
        f.setBackground( Color.gray );
        
        final TabPanel tabPanel = new TabPanel();
        tabPanel.setSize( 400, 340 );
        tabPanel.setBackground( Color.red );
        f.add( tabPanel );

        RoundButton DBtn = new RoundButton( "My Round Button" );
        DBtn.setBackground( Color.yellow );
        DBtn.setForeground( Color.blue );

        FlatButton DBtn2 = new FlatButton( "My Flat Button" );
        DBtn2.setBackground( Color.cyan );
        DBtn2.setForeground( Color.darkGray );

        DoubleBufferedPanel p = new DoubleBufferedPanel();
        p.setLayout( new FlowLayout() );
        p.add( DBtn );
        p.add( DBtn2 );
        final Checkbox chbx = new Checkbox( "Lightweight CHBX" );
        p.add( chbx );
        tabPanel.addTabPanel( "Flow Layout", p );

        p = new DoubleBufferedPanel();
        p.setLayout( null );
        FlatButton fb = new FlatButton( "Flat button" );
        p.add( fb );
        fb.setLocation( 300, 70 );
        Button button = new Button( "Regular Button" );
        p.add( button );
        button.setLocation( 300, 40 );
        TextField tf = new TextField( "This is a text field" );
        p.add( tf );
        tf.setLocation( 100, 200 );
        tabPanel.addTabPanel( "Null Layout", p );


        p = new DoubleBufferedPanel();
        p.setLayout( new GridLayout( 2, 2, 15, 15 ) );
        RoundButton b = new RoundButton( "Round Button" );
        p.add( b );
        p.add( new FlatButton( "Flat Button" ) );
        List list = new List( 5, true );
        list.addItem( "first list item" );
        list.addItem( "second list item" );
        list.addItem( "third list item" );
        list.addItem( "fourth list item" );
        list.addItem( "fifth list item" );
        list.addItem( "sixth list item" );
        list.addItem( "eight list item" );
        list.addItem( "ninth list item" );
        list.addItem( "tenth list item" );
        list.addItem( "eleventh list item" );
        list.addItem( "twelfth list item" );
        list.addItem( "thirteenthg list item" );
        list.addItem( "fourteenth list item" );
        list.addItem( "fifteenth list item" );
        list.addItem( "sixteenth list item" );
        list.addItem( "seventeenth list item" );
        list.addItem( "eighteenth list item" );
        list.addItem( "ninteenth list item" );
        list.addItem( "twentieth list item" );
        p.add( list );
        ScrollPane sp = new ScrollPane();

        class MyCanvas extends Canvas {
            Image image;

            public void addNotify() {
                super.addNotify();
                this.setupGraphics();
            }

            private void setupGraphics() {
                if( image == null )
                    image = this.createImage( getSize().width, getSize().height );
                Graphics imgG = image.getGraphics();
                if( imgG != null ) {
                    this.setBackground( Color.cyan );
                    this.setForeground( Color.black );
                    Font font = new Font( "TimesRoman", Font.BOLD, 70 );
                    Color color = Color.black;
                    int remainder50 = 0;
                    int remainder100 = 0;
                    for( int y=25; y<=875; y+=25 ) {
                        remainder50 = y % 50;
                        remainder100 = y % 100;
                        if( remainder100 == 0 ){
                            font = new Font( "SansSerif", Font.BOLD+Font.ITALIC, 70 );
                            color = Color.green;
                        }
                        else if ( remainder50 == 0 ){
                            font = new Font( "Dialog", Font.PLAIN, 70 );
                            color = Color.white;
                        }
                        else {
                            font = new Font( "TimesRoman", Font.BOLD, 70 );
                            color = Color.black;
                        }
                        imgG.setFont( font );
                        imgG.setColor( color );
                        imgG.drawString( "Canvas in ScrollPane", 50, y );
                    }
                }
            }


            public void paint( Graphics g ) {
                if( image != null )
                    g.drawImage( image, 0, 0, this );
            }
        }

        Canvas c = new MyCanvas();
        c.setSize( 900, 900 );
        sp.add( c );
        sp.setSize( 50, 50 );
        p.add( sp );
        tabPanel.addTabPanel( "Grid Layout", p );

        p = new DoubleBufferedPanel();
        p.setLayout( new BorderLayout() );
        Label l = new Label( "This is a Very Long Regular Label Shown Here in South" );
        l.setBackground( Color.gray );
        l.setForeground( Color.black );
        p.add( "South", l );
        l = new Label( "The Northern Label is HERE!" );
        p.add( "North", l );
        p.add( "East", new TextArea( "Start of text area here to the east...", 12, 20 ) );
        p.add( "West", new TextArea( 12, 15 ) );
        p.add( "Center", new Button( "center button" ) );
        tabPanel.addTabPanel( "Border Layout", p );

        tabPanel.setSelectedTab( 3 );
        f.setResizable( false );
        f.setVisible( true );
    }
}