/*
 * IrTestApp.c
 *     9/17/1998 - AJ Musgrove
 *
 * Show the IR functionality of the Palm platform
 */
 
#include <Pilot.h>
#include <irlib.h>
#include "IrTestApp_res.h"

/*
 * Prototypes
 */
static Boolean StartApplication(void);
static void StopApplication(void);
static void AppEventLoop(void);
static Boolean AppHandleEvent(EventPtr event);
static Boolean MainFormEventHandler(EventPtr event);
static void callback(IrConnect *con, IrCallBackParms *parms);
static void ShowResult(CharPtr func, IrStatus stat);
static void StoreDiscovery(IrDeviceList* deviceList);
static IrStatus SendData();
static void SetData(BytePtr buf, Word len);

/*
 * Globals
 */
static Boolean bound = false;
static IrConnect connect;
static Boolean packet_in_use = false;
static IrPacket packet;
static UInt irref;
static IrDeviceAddr dev;

/*
 * Implementation
 */

/*
 * Entry point for every pilot program.
 * We will only respond to the normal launch command
 */
DWord PilotMain(Word cmd, Ptr cmdBPB, Word launchFlags)
{
	if (cmd == sysAppLaunchCmdNormalLaunch)
	{
		if (!StartApplication())
			return 0;
		AppEventLoop();
		StopApplication();
	}
	
	return 0;
}

/*
 * Locate the IR library, Open IR library, select LMP
 * connections, setup initial IR packet and
 * load the main form
 */
static Boolean StartApplication(void )
{
	if (SysLibFind(irLibName,&irref) != 0)
	{
		FrmCustomAlert(IRErrorAlert,"SysLibFind fails","","");
		return false;
	}
	else
	{
		if (IrOpen(irref,irOpenOptSpeed115200) != 0)
		{
			FrmCustomAlert(IRErrorAlert,"IrOpen fails","","");
			return false;
		}
	}
	
	IrSetConTypeLMP(&connect);
	packet.buff = (BytePtr)"Data";
	packet.len = 4;
	
	FrmGotoForm(MainForm);
	
	return true;
}

/*
 * Shut down connections, close the library
 */
static void StopApplication(void)
{
	if (IrIsIrLapConnected(irref))
		IrDisconnectIrLap(irref);
	IrClose(irref);
}

/*
 * main event dispatch
 */
static void AppEventLoop(void)
{
	EventType event;
	Word error;
	do
	{
		EvtGetEvent(&event, evtWaitForever);
		if (SysHandleEvent(&event))
			continue;
		if (AppHandleEvent(&event))
			continue;
		FrmDispatchEvent(&event);
	} while (event.eType != appStopEvent);
}

/*
 * handle application events. The only
 * one we should get is to load the main
 * form. All we do is set the event handler
 */
static Boolean AppHandleEvent(EventPtr event)
{
	FormPtr form;
	Word formId;
	Boolean handled = false;
	
	if (event->eType == frmLoadEvent)
	{
		formId = event->data.frmLoad.formID;
		form = FrmInitForm(formId);
		FrmSetActiveForm(form);
		
		switch (formId)
		{
		case MainForm:
			FrmSetEventHandler(form,MainFormEventHandler);
			break;
		}
		
		handled = true;
	}
	
	return handled;
}

/*
 * handle events for the main form. All
 * this really involves is drawing the
 * form and reacting to buttons
 */
static Boolean MainFormEventHandler(EventPtr event)
{
	Boolean handled = false;
	
	switch (event->eType)
	{
	case frmOpenEvent:
		FrmDrawForm(FrmGetActiveForm());
		handled = true;
		break;
	case ctlSelectEvent:
		switch (event->data.ctlSelect.controlID)
		{
		case MainBindButton:
			if (bound)
			{
				ShowResult("Unbind",IrUnbind(irref,&connect));
				bound = false;
			}
			else
			{
				ShowResult("Bind",IrBind(irref,&connect,callback));
				IrSetDeviceInfo(irref,&connect.lLsap,1);
				bound = true;
			}
			handled = true;
			break;
		case MainDiscoveryButton:
			ShowResult("DiscoverReq",IrDiscoverReq(irref,&connect));
			handled = true;
			break;
		case MainIrLAPButton:
			if (IrIsIrLapConnected(irref))
				ShowResult("DisconnectIrLap",IrDisconnectIrLap(irref));
			else
				ShowResult("ConnectIrLap",IrConnectIrLap(irref,dev));
			handled = true;
			break;
		case MainLMPRequestButton:
			ShowResult("ConnectReq",IrConnectReq(irref,&connect,&packet,0));
			handled = true;
			break;
		case MainDataButton:
			ShowResult("DataReq",SendData());
			handled = true;
			break;
		case MainTestButton:
			ShowResult("TestReq",IrTestReq(irref,dev,&connect,&packet));
			handled = true;
			break;
		}
		break;
	}
	
	return handled;
}

/*
 * IR callback functions. Generally, all
 * we do is print the event recieved. If
 * we get a DATA_IND, we put the data into
 * the recieved data field. If we get
 * an incoming LMP connection request, we
 * automatically accept it
 */
static void callback(IrConnect *con, IrCallBackParms *parms)
{
	CharPtr event;
	
	switch (parms->event)
	{
	case LEVENT_DISCOVERY_CNF: 
		event = "DISCOVERY_CNF";
		StoreDiscovery(parms->deviceList); break;
	case LEVENT_LAP_CON_CNF: 
		event = "LAP_CON_CNF"; break;
	case LEVENT_LAP_CON_IND:
		event = "LAP_CON_IND"; break;
	case LEVENT_LAP_DISCON_IND:
		event = "LAP_DISCON_IND"; break;
	case LEVENT_LM_CON_CNF:
		event = "LM_CON_CNF"; break;
	case LEVENT_LM_CON_IND:
		event = "LM_CON_IND"; break;
	case LEVENT_LM_DISCON_IND:
		event = "LM_DISCON_IND"; break;
	case LEVENT_PACKET_HANDLED:
		packet_in_use = false;
		event = "PACKET_HANDLED"; break;
	case LEVENT_DATA_IND:
		event = "DATA_IND";
		SetData(parms->rxBuff,parms->rxLen); break;
	case LEVENT_STATUS_IND:
		switch (parms->status)
		{
		case IR_STATUS_NO_PROGRESS:
			event = "S_NO_PROGRESS"; break;
		case IR_STATUS_LINK_OK:
			event = "S_LINK_OK"; break;
		case IR_STATUS_MEDIA_NOT_BUSY:
			event = "S_MEDIA_NOT_BUSY";	break;
		default:
			event = "S_UNKNOWN";
		}
		break;
	case LEVENT_TEST_CNF:
		switch (parms->status)
		{
		case IR_STATUS_SUCCESS:
			event = "TEST_SUCCESS"; break;
		case IR_STATUS_FAILED:
			event = "TEST_FAILED"; break;
		}
		break;
	case LEVENT_TEST_IND:
		event = "TEST_IND"; break;
	default: event = "UNKNOWN";
	}
	
	FldSetTextPtr(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainEventField)),
	event);
	FldDrawField(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainEventField)));
	
	if (parms->event == LEVENT_LM_CON_IND)
		ShowResult("ConnectRsp",IrConnectRsp(irref,&connect,&packet,0));
}

/*
 * puts the function name and result
 * of anything executed in the library onto
 * the main form
 */
void ShowResult(CharPtr func, IrStatus stat)
{
	CharPtr status;
	
	switch (stat)
	{
	case IR_STATUS_SUCCESS:
		status = "SUCCESS";	break;
	case IR_STATUS_FAILED:
		status = "FAILED"; break;
	case IR_STATUS_PENDING:
		status = "PENDING";	break;
	case IR_STATUS_MEDIA_BUSY:
		status = "MEDIA_BUSY"; break;
	case IR_STATUS_NO_IRLAP:
		status = "NO_IRLAP"; break;
	default:
		status = "UNKNOWN";
	}
	
	FldSetTextPtr(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainFunctionField)),
	func);
	FldDrawField(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainFunctionField)));
		
	FldSetTextPtr(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainStatusField)),
	status);
	FldDrawField(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainStatusField)));
}

/*
 * records information we found
 * during discovery
 */
void StoreDiscovery(IrDeviceList* deviceList)
{
	int i;
	char id[10];
	char len[10];
	
	for (i = 0; i < deviceList->nItems; i++)
	{
		dev = deviceList->dev[i].hDevice;
		connect.rLsap = deviceList->dev[i].xid[0];
	}
}

/*
 * Send data from the Data field to the peer.
 * Since the packet is owned by the stack during
 * its communication, we can only use one copy. We
 * use packet_in_use to regulate this
 */
IrStatus SendData()
{
	IrStatus ret;

	if (packet_in_use)
		return IR_STATUS_FAILED;
			
	packet_in_use = true;
	
	packet.buff = (BytePtr)FldGetTextPtr
		(FrmGetObjectPtr(FrmGetActiveForm(),
		FrmGetObjectIndex(FrmGetActiveForm(),MainDataField)));
	packet.len = FldGetTextLength(FrmGetObjectPtr
		(FrmGetActiveForm(),
		FrmGetObjectIndex(FrmGetActiveForm(),MainDataField)));
	
	if ((ret = IrDataReq(irref,&connect,&packet))==IR_STATUS_FAILED)
		packet_in_use = false;
		
	return ret;
}

/*
 * This will take a received data, assume
 * it is a string, and put it into the DataRec
 * field.
 */
void SetData(BytePtr buf, Word len)
{
	char str[80];
	MemMove(str,buf,len);
	str[len] = 0;
	
	FldSetTextPtr(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainDataRecField)),
	str);
	FldDrawField(FrmGetObjectPtr(FrmGetActiveForm(),
	FrmGetObjectIndex(FrmGetActiveForm(),MainDataRecField)));
	
}