################################################################################
#
#  Author:  Troy Neeriemer
#           troy@intraware.com
#
#  File:  mg-admin.pl
#
#  Useage:  Be sure to modify the host, port, and search base variables to 
#           match your environment.  This CGI also requires that mailing
#           groups be enabled in the directory server, this happens by default
#           if Netscape Messaging Server is installed and told about the 
#           Directory Server.
#
#           The source code is broken up into four sections.  The first part
#           is essentially the main loop, and global variables.  The second
#           portion are subroutines that do the primary processing; the third
#           section is LDAP specific routines; and the fourth are HTML output.
#
#           Be aware that the directory server password for a user can be passed
#           in clear text by this CGI so using SSL is encouraged.
#
################################################################################

#!usr/local/bin/perl

use strict;
use CGI;
use FileHandle;
use Mozilla::LDAP::Conn;

# modify the following variables to point to the appropriate directory server
my $ldap_host = "heron.boston.intraware.com";
my $ldap_port = "389";
my $search_base = "o=heron.intraware.com";

# do HTTP streaming as they call it in NAS lingo
STDOUT->autoflush;

# set up the CGI stuff (basically as a global variable)
my $cgi = new CGI;
print $cgi->header();

# check the state of the application and display the appropriate page
unless ($cgi->param()) {
    html_display_login();
} elsif ($cgi->param("exit") eq "Exit") {
    html_display_login();
} elsif ($cgi->param("mode") eq "login") {
    process_login();
} elsif ($cgi->param("mode") eq "submit") {
    process_submit();
}


###############################################################################
###   Subroutines
###############################################################################

# process_login is responsible for binding to the directory server and getting
# the list of mail groups

sub process_login {
    # variables that are specific to the user that's logging in
    my $user = $cgi->param("user");
    my $password = $cgi->param("password");
    my $bind_dn;
    my $name;
    my $bind_entry;

    ($bind_dn, $name) = ldap_get_user_info($user);

    # get_user_info will exit with $bind_dn not being defined if there is a 
    # problem so we need to check to make sure all is well
    if (! $bind_dn) {
        html_display_error("The user name you entered does not appear to 
                            exist in the directory or the directory server 
                            is not responding.");
        exit(0);
    }


    my @mailGroup;

    ldap_get_mail_groups($bind_dn, $password, \@mailGroup);
    

    # now we have the information we need, let's display it.
    html_mg_header($bind_dn, $name, $user, $password);

    my $i;
    my $description;
    my $groupName;

    # for each mailGroup we need to check to see whether the user is a member
    # or not, and print out the information accordingly
    foreach (@mailGroup){
        $groupName = $_->{"cn"}[0];
        $description = $_->{"description"}[0];
        my $isMember = 0;
        
        # this is where we figure out whether they are a member or not
        my $ct = $_->size("uniquemember");
        for ($i = 0; $i < $ct ; $i++) {
            if ($bind_dn eq $_->{"uniquemember"}[$i]) {
                $isMember=1;
                last;
            }
        }

        html_row($isMember, $groupName, $ct, $description);
    }

    html_mg_footer();
    
}

# process_submit is responsible for processing changes in mail group membership
# and displaying confirmation of the changes

sub process_submit {
    my @key = $cgi->param();
    my $bind_dn = $cgi->param("bind_dn");
    my $password = $cgi->param("password");
    my $name = $cgi->param("name");
    my $user = $cgi->param("user");
    my $mode = $cgi->param("mode");

    my @selected;

    html_results_header();

    # really sloppy way of getting the CGI param list pared down to just the 
    # names of mailing groups
    foreach (@key) {
        next if ($_ eq "bind_dn");
        next if ($_ eq "password");
        next if ($_ eq "name");
        next if ($_)eq "user";
        next if ($_ eq "mode");
        push(@selected, $_);
        
    }

    my @mailGroup;
    my @results;
    ldap_get_mail_groups($bind_dn, $password, \@mailGroup);

    foreach (@mailGroup) {
        my $is_member = 0;
        my $is_selected = 0;
        my $group = $_;
        my $group_name = $group->{"cn"}[0];
        my $group_email = $group->{"mail"}[0];
        
        #figure out if the user is a member of this group
        my $size = $group->size("uniquemember");
        my $i;
        for ($i = 0 ; $i < $size ; $i++) {
            if ($bind_dn eq $group->{"uniquemember"}[$i]) {
                $is_member = 1;
                last;
            }
        }

        #figure out if the user has selected this group or not
        foreach (@selected) {
            if ($_ eq $group_name) {
                $is_selected = 1;
                last;
            }
        }
        
        my $action;
        
        # figure out what action to take
        if ($is_member) {
            if ($is_selected) {
                $action = "keep";
            } else {
                $action = "remove";
            }
        } else {
            if ($is_selected) {
                $action = "add";
            } else {
                $action = "none";
            }
        }

        if ($action eq "add") {
            ldap_add_to_group($bind_dn, $password, $group_name);
        } elsif ($action eq "remove") {
            ldap_remove_from_group($bind_dn, $password, $group_name);
        }

        push(@results, "$group_name:$action:$group_email");

        
    }

    html_results(\@results);

    html_results_footer($bind_dn, $name, $user, $password);

    
}

################################################################################
###  LDAP Subroutines
################################################################################

# ldap_get_user_info
# parameters:
#     $user = simple user name
# returns:
#     $bind_dn = full distinguished name for entry with a uid of $user
#     $cn = the common name (usually full name) of user with uid of $user

sub ldap_get_user_info {
    my $user = $_[0];
    my $bind_dn;
    my $cn;
    
    my $anon_conn = new Mozilla::LDAP::Conn($ldap_host, $ldap_port, "", "", "") 
        || die "Can't connect to $ldap_host.\n";

    my $entry = $anon_conn->search($search_base, "sub", "(uid=$user)");

    if (! $entry) {
        # we didn't get any results back
        return;
    } else {
        my $i = 0;
        while($entry) {
            $i++;
            $bind_dn = $entry->getDN();
            if ($entry->exists("cn")) {
                $cn = $entry->{"cn"}[0];
            } else {
                # if there isn't a "cn" or common name then the entry is
                # probably corrupted so we should exit
                return;
            }
                        
            $entry = $anon_conn->nextEntry();
        }
        # if there is more than one entry with the same uid the directory 
        # server is in a bad state so we should exit
        if ($i > 1) {
            return;
        }
    }    

    $anon_conn->close();
    
    return ($bind_dn, $cn);
}

# ldap_get_mail_groups
# parameters:
#     $bind_dn = full distinguished name
#     $password = password for $bind_dn
#     $mgref = a reference to an array that will store Entry objects
# returns:
#     nothing explicitly, but the contents of $mgref should be populated

sub ldap_get_mail_groups {
    my $bind_dn = $_[0];
    my $password = $_[1];
    my $mgref = $_[2];
    
    my $conn = new Mozilla::LDAP::Conn($ldap_host, $ldap_port, $bind_dn, 
                                       $password, "");
    my $entry;
    my $cn;

    if (! $conn) {
        html_display_error( "Couldn't connect to the directory server.");
        exit(0);
    } else {
        $entry = $conn->search($search_base, "sub", "objectclass=mailGroup");
        
        # make sure that we've got some mailGroup's to work with.
        if (! $entry) {
            html_display_error("There aren't any mailing lists on this server.");
            exit(0);
        }

        # loop through all the entries and add them to the mailGroup array
        while($entry) {
            $cn = $entry->{"cn"}[0];
            # skip the Postmaster group since we don't really want people 
            # adding themselves to this group
            if ($cn ne "Postmaster") {
                push(@$mgref,$entry);
            }
            $entry = $conn->nextEntry();
        }
        
        $conn->close();
    }
}

# ldap_add_to_group
# parameters:
#     $bind_dn = full distinguished name
#     $password = password for $bind_dn
#     $mg_cn = cn entry for the mailing group
# returns:
#     nothing.  There should probably be some additional error checking done
#     in this subroutine.  There should be an additional uniquemember attribute
#     added to the mailing group at the end of this routine.

sub ldap_add_to_group {
    my $bind_dn = $_[0];
    my $password = $_[1];
    my $mg_cn = $_[2];

    my $conn = new Mozilla::LDAP::Conn($ldap_host, $ldap_port, $bind_dn, 
                                       $password, "");
    
    if (!$conn) {
        html_display_error( "Couldn't connect to the directory server.");
        exit(0);
    } else {
        my $entry = $conn->search($search_base, "sub", "(cn=$mg_cn)");
        if (!$entry) {
            html_display_error( "Couldn't find $mg_cn.");
            exit(0);
        }
        $entry->addValue("uniquemember", $bind_dn);
        $conn->update($entry);

        $conn->close();

    }
}

# ldap_remove_from_group
# parameters:
#     $bind_dn = full distinguished name
#     $password = password for $bind_dn
#     $mg_cn = cn entry for the mailing group
# returns:
#     nothing.  There should probably be some additional error checking done
#     in this subroutine.  There should be an additional uniquemember attribute
#     removed from the mailing group at the end of this routine.


sub ldap_remove_from_group {
    my $bind_dn = $_[0];
    my $password = $_[1];
    my $mg_cn = $_[2];


    my $conn = new Mozilla::LDAP::Conn($ldap_host, $ldap_port, $bind_dn, 
                                       $password, "");
    my $rentry;
    
    if (!$conn) {
        html_display_error( "Couldn't connect to the directory server.");
        exit(0);
    } else {
        $rentry = $conn->search($search_base, "sub", "(cn=$mg_cn)");
        if (!$rentry) {
            html_display_error( "Couldn't find $mg_cn.");
            exit(0);
        }
        my $success = $rentry->removeValue("uniquemember", $bind_dn);
        
        $conn->update($rentry);
        if (!$success) {
            html_display_error("Unable to modify $mg_cn.");
            exit(0);
        }
        $conn->close();

    }
}


################################################################################
###  HTML Output Subroutines
################################################################################

# html_display_login provide a user name & password form so that a user can
# bind to the directory server

sub html_display_login {
    my $script = $cgi->script_name();
    print qq!<HTML><HEAD>
<TITLE>Mailing Group Self-Administration</TITLE>
<BODY BGCOLOR=ffffff>

\&nbsp;
<P>

<CENTER>
<TABLE BORDER CELLPADDING=3 CELLSPACING=3>
<TR>
<TD ALIGN=center BGCOLOR=D5CCBB>
<P><FONT FACE=geneva,arial,helvetica SIZE=+1><B>Mailing Group Membership 
<BR>Self-Administration</B>

<TABLE>
<FORM METHOD=POST ACTION="$script">
<INPUT TYPE="hidden" NAME="mode" VALUE="login">
<TR>
    <TD ALIGN=right>
        <FONT FACE=geneva,arial,helvetica SIZE=2> User name:</FONT>
    </TD>
    <TD>
        <FONT FACE=geneva,arial,helvetica SIZE=2>  
        <INPUT TYPE="text" NAME="user" SIZE=10>  </FONT>
    </TD>
</TR>
<TR>
    <TD ALIGN=right>
          <FONT FACE=geneva,arial,helvetica SIZE=2>Password:</FONT>
    </TD>
    <TD>
        <FONT FACE=geneva,arial,helvetica SIZE=2>
        <INPUT TYPE="password" NAME="password" SIZE=10></FONT>
    </TD>
</TR>

<TR>
    <TD COLSPAN=2 ALIGN=right>
        <FONT FACE=geneva,arial,helvetica SIZE=2>
        <INPUT TYPE="submit" VALUE="Login"></FONT>
    </TD>
</TR>

</FORM>
</TABLE>
</TD></TR></TABLE>
</CENTER>
</BODY>
</HTML>!
}

# html_display_error displays a formatted error message.
sub html_display_error {
    my $message = $_[0];

    print "<P><B>$message<P>Please contact your system adminstrator.\n";
}

# html_mg_header is the top of the page that displays the mailing groups and
# lets users subcribe to groups.
# parameters:
#     $bind_dn = full distinguished name of user
#     $name = full name of user, used for personalization
#     $user = actual uid of user, could be extracted from bind_dn
#     $password = password for $bind_dn
# returns:
#     nothing

sub html_mg_header {
    my $bind_dn = $_[0];
    my $name = $_[1];
    my $user = $_[2];
    my $password = $_[3];
    my $script = $cgi->script_name();

    print qq!<HEAD><TITLE>Update Mailing Group Membership</TITLE></HEAD>
<BODY BGCOLOR=ffffff>
<CENTER><BR><BR>

    
<TABLE WIDTH=600>
<FORM METHOD=POST ACTION="$script">
<INPUT TYPE=hidden NAME=mode VALUE="submit">
<INPUT TYPE=hidden NAME=bind_dn VALUE="$bind_dn">
<INPUT TYPE=hidden NAME=name VALUE="$name">
<INPUT TYPE=hidden NAME=user VALUE="$user">
<INPUT TYPE=hidden NAME=password VALUE="$password">


<TR BGCOLOR=408080>
    <TD COLSPAN=4 ALIGN=center>
        <FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=ffffff>
        <B>Mailing Group Self-Administration for $name.</B></FONT>
    </TD>
</TR>
<TR>
    <TD COLSPAN=4>
        <BR><FONT FACE=geneva,arial,helvetica SIZE=2>
        The checkbox in the Member column shows whether you are a member of that 
        group or not.  To change your membership, merely toggle the checkbox. 
        Once  you are a member of a group, you will recieve e-mail addressed to
        the group name.  After you are satisfied with your selections click the
        "Update" button.

        <BR><BR>
    </TD>
</TR>
<TR BGCOLOR=D5CCBB>
    <TD>
        <FONT FACE=geneva,arial,helvetica SIZE=2><B>Member</B></FONT>
    </TD>
    <TD>
        <FONT FACE=geneva,arial,helvetica SIZE=2><B>Group Name</B></FONT>
    </TD>
    <TD>
        <FONT FACE=geneva,arial,helvetica SIZE=2><B>Description</B></FONT>
    </TD>
    <TD>
        <FONT FACE=geneva,arial,helvetica SIZE=2><B># of Members</B></FONT>
    </TD>
</TR>!;
}

# html_mg_footer displays the footer for the mailing group subscription page

sub html_mg_footer {
    print qq!<TR>
    <TD COLSPAN=4 ALIGN=center>
        <FONT FACE=geneva,arial,helvetica SIZE=2>
        <BR><BR><INPUT TYPE=submit VALUE=Update> 
        <INPUT TYPE=reset VALUE=Revert> 
        <INPUT TYPE=submit Name="exit" VALUE="Exit">
        </FONT>
    </TD>
</TR>
</FORM>
</TABLE>
</BODY></HTML>
!;

}

# html_row displays an mailing group entry as part of a HTML table
# parameters:
#    $checked = used to determine initial state of the subscription checkbox
#    $name = name of mail group
#    $num_members = number of members in the mail group
#    $description = description of the mail group
# returns:
#     nothing

sub html_row {
    my $checked = $_[0];
    my $name = $_[1];
    my $num_members = $_[2];
    my $description = $_[3];

    my $box_state;

    if ($checked) {
        $box_state="CHECKED";
    }

    print qq!<TR>
    <TD ALIGN=center VALIGN=top BGCOLOR=EAE6DD>
        <FONT FACE=geneva,arial,helvetica SIZE=2>
        <INPUT TYPE=checkbox NAME=$name $box_state></FONT>
    </TD>
    <TD VALIGN=top BGCOLOR=EAE6DD>
        <FONT FACE=geneva,arial,helvetica SIZE=2><B>$name</B></FONT>
    </TD>
    <TD VALIGN=top BGCOLOR=EAE6DD>
        <FONT FACE=geneva,arial,helvetica SIZE=2>$description</FONT>
    </TD>
    <TD VALIGN=top ALIGN=center BGCOLOR=EAE6DD>
        <FONT FACE=geneva,arial,helvetica SIZE=2>$num_members</FONT>
    </TD>
</TR>!
}

# html_results_header displays the top of the page for the results of changes
# in subsciption pages

sub html_results_header {
    print qq!<HEAD><TITLE>Update Results</TITLE></HEAD><BODY BGCOLOR=ffffff>
<CENTER><BR><BR>
<TABLE WIDTH=600>!;
}

# html_results is responsible for printing out the update results after a user
# has made changes (or not) to their subscriptions
# parameters:
#     $results = a reference to an array which contains the group name, action
#                performed on the group, and the e-mail address of the group.
#                The information for each group is colon delimited.
# returns:
#     nothing

sub html_results {
    my $results = $_[0];
    my $group;
    my $action;
    my $email;
    my $ct = 0;

    my @message;

    print "<TR BGCOLOR=408080>\n";
    print "<TD ALIGN=center COLSPAN=2><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=ffffff><B>You are a member of the following Mailing Groups.</B></FONT></TD>\n";
    print "</TR>\n";
    print "<TR BGCOLOR=D5CCBB><TD><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=000000><B>Group Name</B></FONT></TD>";
    print "<TD><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=000000><B>Group E-Mail Address</B></FONT></TD></TR>\n";

    foreach(@$results) {
        ($group, $action, $email) = split(/:/,$_);
        if (($action eq "keep") || ($action eq "add")) {
            $ct++;
            print "<TR>\n";
            print "<TD VALIGN=top BGCOLOR=EAE6DD><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=000000>$group</FONT></TD>\n";
            print "<TD VALIGN=top BGCOLOR=EAE6DD><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=000000><A HREF=\"mailto:$email\">$email</A></FONT></TD>\n";
            print "</TR>\n";
        }
    }
    
    if ($ct == 0) {
        print "<TR><TD VALIGN=top BGCOLOR=EAE6DD COLSPAN=2><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=000000>None</FONT></TD></TR>\n";
    }
    

    foreach(@$results) {
        ($group, $action, $email) = split(/:/,$_);
        if ($action eq "add") {
            push(@message, "You have been added to <B>$group</B>.");
        } elsif ($action eq "remove") {
            push(@message, "You have been removed from <B>$group</B>.");
        }
    }
    
    if ($#message != -1) {
        print "<TR><TD COLSPAN=2><BR></TD</TR>\n";
        print "<TR BGCOLOR=408080>\n";
        print "<TD ALIGN=center COLSPAN=2><FONT FACE=geneva,arial,helvetica SIZE=2 COLOR=ffffff><B>Results of update.</B></FONT></TD>\n";
        print "</TR>\n";
        print "<TR><TD><FONT FACE=geneva,arial,helvetica SIZE=2 COLSPAN=2>\n";
        foreach(@message) {
            print "<BR>$_\n";
        }
        print "</TD></TR>\n";
    }

}

# html_results_footer is responsible for displaying the footer on the results
# page.  It has some dynamic form elements that it is responsible for.
# parameters:
#     $bind_dn = full distinguished name
#     $name = name of user
#     $user = uid of user
#     $password = password of user
# returns:
#     nothing

sub html_results_footer {
    my $bind_dn = $_[0];
    my $name = $_[1];
    my $user = $_[2];
    my $password = $_[3];
    my $script = $cgi->script_name();

    print qq!<TR><TD ALIGN=center COLSPAN=2><FONT FACE=geneva,arial,helvetica SIZE=2>
<FORM METHOD=POST ACTION=\"$script\">
<INPUT TYPE=hidden NAME=mode VALUE=\"login\">
<INPUT TYPE=hidden NAME=bind_dn VALUE=\"$bind_dn\">
<INPUT TYPE=hidden NAME=name VALUE=\"$name\">
<INPUT TYPE=hidden NAME=user VALUE=\"$user\">
<INPUT TYPE=hidden NAME=password VALUE=\"$password\">
<BR><BR><INPUT TYPE=submit VALUE=\"Re-Edit\"> 
<INPUT TYPE=submit Name=\"exit\" VALUE=\"Exit\">

</FORM>
</TD></TR></TABLE>
</BODY></HTML>
!;
}











