// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called PETE (Portable Expression Template Engine) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-99-5.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No.  W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE, and to allow others to do so.
// The public may copy and use this SOFTWARE, FOR NONCOMMERCIAL USE ONLY,
// without charge, provided that this Notice and any statement of
// authorship are reproduced on all copies.  Neither the Government nor
// the University makes any warranty, express or implied, or assumes any
// liability or responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about PETE, send e-mail to pete@acl.lanl.gov,
// or visit the PETE web page at http://www.acl.lanl.gov/pete/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef PETE_PETE_TYPECOMPUTATIONS_H
#define PETE_PETE_TYPECOMPUTATIONS_H

//-----------------------------------------------------------------------------
//
// CLASS NAME
//    Type2Index<T>
//
// DESCRIPTION
//    This template describes a set of trait classes that associate an
//    index with a type. Each concrete class---a type that is designed
//    to be used like a built-in type---must have a specialization
//    of this class that provides a unique index. This index is used
//    to figure out the default return type of a binary/trinary operation.
//    Specifically, the largest index is chosen.
//
//-----------------------------------------------------------------------------

template<class Type>
struct Type2Index {
  enum { val = 666666 };
};

// Apparently KCC requires this for types like 'const int' to work correctly.

template<class T>
struct Type2Index<const T> {
  enum {val = Type2Index<T>::val };
};

template<> struct Type2Index<bool> {
  enum { val = 1 };
};

template<> struct Type2Index<char> {
  enum { val = 2 };
};

template<> struct Type2Index<short> {
  enum { val = 3 };
};

template<> struct Type2Index<int> {
  enum { val = 4 };
};

template<> struct Type2Index<long> {
  enum { val = 5 };
};

template<> struct Type2Index<float> {
  enum { val = 6 };
};

template<> struct Type2Index<double> {
  enum { val = 7 };
};


//-----------------------------------------------------------------------------
//
// CLASS NAME
//    UnaryReturn<T, Op>
//
// DESCRIPTION
//    This template describes the default mechanism for calculating the
//    return type to a unary expression given the argument type T and
//    the operation type Op.
//
//    There are two sensible things one can do automatically:
//      o (Op::tag == UnaryPassThruTag) make the return type 
//        the same as the argument to the function/operation. 
//        For example, operator-(T) should return a T.
//      o (Op::tag != UnaryPassThruTag) return a type based entirely on 
//        the operation. For example, operator! always returns a bool.
//    To figure out which approach to take, PETEUnaryReturn uses the
//    tag from the operator and another template, ComputeUnaryType.
//
//    Consider negation (unary minus). The operator would be formed like:
//      struct OpUnaryMinus {
//        enum { tag = UnaryPassThruTag };
//      };
//    Logical negation (not) would be formed like:
//      struct OpNot {
//        enum { tag = Type2Index<bool> };
//        typedef bool type;
//      };
//    The minor redundancy (specifying the tag and the type) is required to 
//    allow easy support for compilers that may or may not support partial
//    specialization.
//
//    Special cases can be handled by directly specializing PETEUnaryReturn.
//    For example, the abs function typically returns a double when the
//    argument is a complex<double>. The appropriate specialization here
//    would be:
//      template<> struct PETEUnaryReturn< complex<double>, FnAbs > {
//        typedef double type;
//      };
//
//-----------------------------------------------------------------------------

const int UnaryPassThruTag = 0;

template<class T, class Op, int OpTag>
struct ComputeUnaryType {
  typedef typename Op::Type_t Type_t;
};

template<class T, class Op>
struct ComputeUnaryType<T, Op, UnaryPassThruTag> {
  typedef T Type_t;
};

template<class T, class Op>
struct UnaryReturn {
  typedef typename ComputeUnaryType<T, Op, Op::tag>::Type_t Type_t;
};


//-----------------------------------------------------------------------------
//
// CLASS NAME
//    BinaryReturn<T1, T2, Op>
//
// DESCRIPTION
//    This template describes the default mechanism for calculating the
//    return type to a binary expression given the argument types T1 and
//    T2 and the operation type Op.
//
//    There are four sensible things one can do automatically:
//      o (Op::tag == BinaryPromoteTag) make the return type by 
//        promoting/converting the "simpler" type into the more "complex." 
//        For example, we typically want to do this with addition. 
//      o (Op::tag == BinaryUseLeftTag) return the type of the 
//        left-hand operand. For example, this is what happens with operator<<.
//      o (Op::tag == BinaryUseRightTag) return the type of the 
//        right-hand operand. 
//      o (otherwise) return a type based entirely on the operation. 
//        For example, operator!= always returns a bool.
//    To figure out which approach to take, BinaryReturn uses the
//    tag from the operator and another template, ComputeBinaryType.
//
//    Consider addition. The operator would be formed like:
//      struct OpAdd {
//        enum { tag = BinaryPromoteTag };
//      };
//
//    Special cases can be handled by directly specializing BinaryReturn.
//    For example, the multiplication between a matrix and a vector might do a
//    matrix/vector product, thereby returning a vector. The appropriate 
//    specialization here would be:
//      struct BinaryReturn< Mat<double,3>, Vec<float,3>, OpMultiply > {
//        typedef Vector<double,3> Type_t;
//      };
//    Notice how the element type is promoted.
//
//-----------------------------------------------------------------------------

const int BinaryPromoteTag = -2;
const int BinaryUseLeftTag = -1;
const int BinaryUseRightTag = 0;

// This is still harder than it has to be. There are bugs in
// the EDG front end.

template<class T1, class T2, class Op, int op>
struct ComputeBinaryType {
  typedef typename Op::Type_t Type_t;
};

template<class T1, class T2, class Op>
struct ComputeBinaryType<T1, T2, Op, BinaryUseLeftTag> {
  typedef T1 Type_t;
};

template<class T1, class T2, class Op>
struct ComputeBinaryType<T1&, T2, Op, BinaryUseLeftTag> {
  typedef T1 Type_t;
};

template<class T1, class T2, class Op>
struct ComputeBinaryType<T1, T2, Op, BinaryUseRightTag> {
  typedef T2 Type_t;
};

template<class T1, class T2, class Op>
struct ComputeBinaryType<T1, T2&, Op, BinaryUseRightTag> {
  typedef T2 Type_t;
};

template<class T1, class T2, int lr>
struct ComputePromoteType {
};

template<class T1, class T2>
struct ComputePromoteType<T1, T2, 1> {
  typedef T1 Type_t;
};

template<class T1, class T2>
struct ComputePromoteType<T1, T2, 0> {
  typedef T2 Type_t;
};

template<class T1, class T2, int t1, int t2>
struct ComputePromoteType2 {
  enum { order = (t1>=t2) };
  typedef typename
    ComputePromoteType<T1, T2,order>::Type_t Type_t;
};

template<class T1, class T2, class Op>
struct ComputeBinaryType<T1, T2, Op, BinaryPromoteTag> {
  typedef typename ComputePromoteType2<T1, T2, 
    Type2Index<T1>::val, Type2Index<T2>::val>::Type_t Type_t;
};

template<class T1, class T2, class Op>
struct BinaryReturn {
  typedef typename ComputeBinaryType<T1, T2, Op, Op::tag>::Type_t Type_t;
};


//-----------------------------------------------------------------------------
//
// CLASS NAME
//    TrinaryReturn<T1, T2, T3, Op>
//
// DESCRIPTION
//    This template describes the default mechanism for calculating the
//    return type to a trinary expression given the argument types T1, T2, and
//    T3 and the operation type Op. The only trinary expression supported
//    in C++ is the ?: operation. In this case, T1 should end up being bool
//    and the result of the calculation is of type Binary_Promotion(T2,T3)
//    with the value being that associated with T2 if T1's associated value 
//    turns out to be true and T3 if T1's associated value turns out to be 
//    false.
//
//-----------------------------------------------------------------------------

template<class T1, class T2, class T3, class Op>
struct TrinaryReturn {
  typedef typename ComputeBinaryType<T2, T3, Op, Op::tag>::Type_t Type_t;
};


#endif // TYPE_COMPUTATIONS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TypeComputations.h,v $   $Author: sa_smith $
// $Revision: 1.3 $   $Date: 1999/02/03 21:41:44 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
