// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called PETE (Portable Expression Template Engine) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-99-5.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No.  W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE, and to allow others to do so.
// The public may copy and use this SOFTWARE, FOR NONCOMMERCIAL USE ONLY,
// without charge, provided that this Notice and any statement of
// authorship are reproduced on all copies.  Neither the Government nor
// the University makes any warranty, express or implied, or assumes any
// liability or responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about PETE, send e-mail to pete@acl.lanl.gov,
// or visit the PETE web page at http://www.acl.lanl.gov/pete/.
// ----------------------------------------------------------------------
// ACL:license

// Include files

#include "Tools/ClassDescriptor.h"
#include "Tools/Header.h"
#include "Tools/OperatorDescriptor.h"
#include "Tools/Parser.h"
#include "Tools/PrintOperators.h"
#include "Tools/PrintFunctions.h"
#include "Tools/PrintList.h"
#include "Tools/Options.h"
#include "Tools/PeteOps.h"

#include <fstream.h>
#include <iostream.h>
#include <string>
#include <vector>
using std::string;
using std::vector;
using std::copy;
using std::back_inserter;

#if defined(macintosh)
#include <console.h>
#endif

int main(int argc, char *argv[])
{
#if defined(macintosh)
  argc = ccommand(&argv);
#endif

  if (flagOption(argc,argv,"--help") || flagOption(argc,argv,"--pete-help"))
  {
    cout << "MakeOperators produces global functions for C++  " << endl;
    cout << "operators (+ - * etc.) that create expression trees." << endl;
    cout << "Global assignment operators may be produced as well." << endl;
    cout << "This function can also produce operator tag structs." << endl;
    cout << "" << endl;
    cout << "Options:" << endl;
    cout << "--help:           Print this message." << endl;
    cout << "--pete-help:      Print this message." << endl;
    cout << "--classes file:   Read the class descriptors from file." << endl;
    cout << "                  If no class file is provided, then" << endl;
    cout << "                  no operators or assignment operators" << endl;
    cout << "                  are produced." << endl;
    cout << "--o file:         Name of file to write operator info to." << endl;
    cout << "                  If not specified, output goes to the terminal."
         << endl;
    cout << "--operators file: Read the operator descriptors from file." << endl;
    cout << "                  If no operator file is provided, then" << endl;
    cout << "                  the standard set of PETE operators is" << endl;
    cout << "                  used (most of the C++ operators)." << endl;
    cout << "--pete-ops:       Add the set of PETE operators to those" << endl;
    cout << "                  input from the operator file." << endl;
    cout << "--guard string:   Use string for the include guard" << endl;
    cout << "                  (defaults to GENERATED_OPERATORS_H)." << endl;
    cout << "--scalars:        If this flag is present, only generate" << endl;
    cout << "                  operators involving the user-defined scalars."
	 << endl;
    cout << "--extra-classes:  If this flag is present, only generate" << endl;
    cout << "                  operators involving the extraClasses."
	 << endl;
    cout << "--no-expression:  If this flag is present, don't generate"
	 << endl;
    cout << "                  operators involving Expression<T>" << endl;
    cout << "--assign-ops:     If this flag is present, generate the" << endl;
    cout << "                  assignment operators which call evaluate()"
	 << endl;
    cout << "--op-tags:        If this flag is present, generate the" << endl;
    cout << "                  operator tag structs" << endl;
    cout << "--no-shift-guard: If this flag is present, don't put guards"
	 << endl;
    cout << "                  around the scalar << class and scalar >> class"
	 << endl;
    cout << "                  operators (they can get confused with stream"
	 << endl;
    cout << "                  operations)." << endl;
    cout << endl;
    cout << "These two options are used internally by PETE:" << endl;
    cout << "--insert-op:      Used to build the file" << endl;
    cout << "                  src/Tools/PeteOps.cpp." << endl;
    cout << "--lanl-boilerplate:  Includes the standard ACL header and" << endl;
    cout << "                  footer." << endl;
    return 0;
  }

  vector<string> filesUsed;
  filesUsed.push_back("MakeOperators");

  bool printOperators = flagOption(argc,argv,"--classes");
  string cls = stringOption(argc,argv,"--classes","");
  string ofile = stringOption(argc, argv, "--o", "");
  string guardDef(printOperators ?
		  "GENERATED_OPERATORS_H" : "OPERATOR_TAGS_H");
  string includeGuard = stringOption(argc,argv,"--guard",guardDef);
  bool justScalars = flagOption(argc,argv,"--scalars");
  bool justExtraClasses = flagOption(argc,argv,"--extra-classes");
  bool expression = !flagOption(argc,argv,"--no-expression");
  bool assignment = flagOption(argc,argv,"--assign-ops");
  bool printTags = flagOption(argc,argv,"--op-tags");
  bool shiftGuard = !flagOption(argc,argv,"--no-shift-guard");
  bool insertOp = flagOption(argc,argv,"--insert-op");
  bool addPeteOps = (!flagOption(argc,argv,"--operators")) ||
    flagOption(argc,argv,"--pete-ops");
  bool lanlBoilerplate = flagOption(argc,argv,"--lanl-boilerplate");

  // Input the operator descriptors.
  
  map<string, vector<OperatorDescriptor> > mOps;
  map<string, vector<OperatorDescriptor> > inputOps;  

  if (flagOption(argc,argv,"--operators"))
  {
    string ops = stringOption(argc,argv,"--operators","");
    ifstream fOps(ops.c_str());
    filesUsed.push_back(ops);
    if (!fOps)
    {
      cerr << "ERROR: couldn't open operator description file \""
	   << ops << "\"" << endl;
      return -1;
    }
    Parser<OperatorDescriptor> pOps(fOps, ops, 
				    "TAG", "FUNCTION", "EXPR", mOps);
    pOps.parse();
  }
  inputOps = mOps;
  
  if (addPeteOps)
  {
    peteOps(mOps);
  }
  
  // Create vectors for unary operators.
  
  vector<OperatorDescriptor> unaryOps(mOps["unaryOps"]);
  copy(mOps["unaryBoolOps"].begin(), mOps["unaryBoolOps"].end(),
       back_inserter(unaryOps));
  copy(mOps["unarySpecialOps"].begin(), mOps["unarySpecialOps"].end(),
       back_inserter(unaryOps));
  
  vector<OperatorDescriptor> &unaryCastOps = mOps["unaryCastOps"];

  // Create vectors for binary operators.
  
  vector<OperatorDescriptor> binaryOps(mOps["binaryOps"]);
  copy(mOps["binaryBoolOps"].begin(), mOps["binaryBoolOps"].end(),
       back_inserter(binaryOps));
  //  copy(mOps["binaryLeftOps"].begin(), mOps["binaryLeftOps"].end(),
  //       back_inserter(binaryOps));
  copy(mOps["binarySpecialOps"].begin(), mOps["binarySpecialOps"].end(),
       back_inserter(binaryOps));
    
  vector<OperatorDescriptor> binaryLeftOps(mOps["binaryLeftOps"]);

  // Create reference for trinary operators.
  
  vector<OperatorDescriptor> &trinaryOps = mOps["trinaryOps"];
    
  // Create vector for assignment operators.
  
  vector<OperatorDescriptor> assignOps(mOps["assignOp"]);
  copy(mOps["binaryAssignOps"].begin(),
       mOps["binaryAssignOps"].end(),
       back_inserter(assignOps));
  copy(mOps["binaryAssignBoolOps"].begin(),
       mOps["binaryAssignBoolOps"].end(),
       back_inserter(assignOps));

  // Input the class descriptors.

  map<string, vector<ClassDescriptor> > mClasses;  

  vector<ClassDescriptor> classes;
  vector<ClassDescriptor> extraClasses;
  vector<ClassDescriptor> scalars;
  vector<ClassDescriptor> userClasses;

  if (printOperators)
  {
    ifstream fClasses(cls.c_str());
    filesUsed.push_back(cls);
    if (justScalars)
    {
      filesUsed.push_back("(Only operations with scalars were printed.)");
    }
    
    if (!fClasses)
    {
      cerr << "ERROR: couldn't open class description file \""
	   << cls << "\"" << endl;
      return -1;
    }
    
    Parser<ClassDescriptor> pClasses(fClasses, cls,
				     "ARG", "CLASS", "", mClasses);
    pClasses.parse();
  
    classes = mClasses["classes"];
    extraClasses = mClasses["extraClasses"];
    scalars = mClasses["scalars"];

    userClasses = classes;
    if (expression)
    {
      classes.push_back(ClassDescriptor("class T[n]","Expression<T[n]>"));
    }
    if (!justScalars)
    {
      scalars.push_back(ClassDescriptor("class T[n]","T[n]"));
    }
  }

  // Set up streams for printing.
  
  ostream *ofl = &cout;
  if (ofile != string(""))
    {
      ofl = new ofstream(ofile.c_str());
      if (ofl == NULL || !*ofl)
        {
          cerr << "ERROR: couldn't open output file \""
               << ofile << "\"." << endl;
          return -1;
        }
    }
        
  // Print header.
  
  printHeader(*ofl,includeGuard,filesUsed,lanlBoilerplate);

  // The following code is used when we generate PeteOps.cpp from
  // PeteOps.in.  Users should never use the --insert-ops option.

  if (insertOp)
  {
    *ofl << "#include \"Tools/OperatorDescriptor.h\"" << endl
	 << "#include <vector.h>" << endl
	 << "#include <map.h>" << endl;
    *ofl << endl
	 << "void peteOps(map<string,vector<OperatorDescriptor> > &m)" << endl
	 << "{" << endl;

    map<string,vector<OperatorDescriptor> >::iterator opvec;

    for(opvec = mOps.begin();opvec != mOps.end();++opvec)
    {
      printList(*ofl,InsertOp(opvec->first),opvec->second);
    }

    *ofl << "}" << endl;
  }

  // Print operator tags.
  
  if (printTags)
  {
    printList(*ofl,UnaryOp(),           inputOps["unaryOps"]);
    printList(*ofl,UnaryBoolOp(),       inputOps["unaryBoolOps"]);
    printList(*ofl,UnaryCastOp(),       inputOps["unaryCastOps"]);
    printList(*ofl,UnarySpecialOp(),    inputOps["unarySpecialOps"]);
    printList(*ofl,BinaryOp(),          inputOps["binaryOps"]);
    printList(*ofl,BinaryBoolOp(),      inputOps["binaryBoolOps"]);
    printList(*ofl,BinaryLeftOp(),      inputOps["binaryLeftOps"]);
    printList(*ofl,BinarySpecialOp(),   inputOps["binarySpecialOps"]);
    printList(*ofl,BinaryAssignOp(),    inputOps["binaryAssignOps"]);
    printList(*ofl,BinaryAssignOp(),    inputOps["assignOp"]);
    printList(*ofl,BinaryAssignBoolOp(),inputOps["binaryAssignBoolOps"]);
    printList(*ofl,TrinaryOp(),         inputOps["trinaryOps"]);
  }

  // Print operators.

  if (printOperators)
  {
    if (!justScalars)
    {
      if (!justExtraClasses)
      {
	printList(*ofl,UnaryFunction(),unaryOps,classes);
	printList(*ofl,UnaryCastFunction(),unaryCastOps,classes);
	printList(*ofl,BinaryFunction(),binaryOps,classes,classes);
	printList(*ofl,BinaryFunction(),binaryLeftOps,classes,classes);
      }
      else
      {
	printList(*ofl,UnaryFunction(),unaryOps,extraClasses);
	printList(*ofl,UnaryCastFunction(),unaryCastOps,extraClasses);
	printList(*ofl,BinaryFunction(),binaryOps,extraClasses,extraClasses);
	printList(*ofl,BinaryFunction(),binaryLeftOps,extraClasses,extraClasses);
	printList(*ofl,BinaryFunction(),binaryOps,classes,extraClasses);
	printList(*ofl,BinaryFunction(),binaryLeftOps,classes,extraClasses);
	printList(*ofl,BinaryFunction(),binaryOps,extraClasses,classes);
	printList(*ofl,BinaryFunction(),binaryLeftOps,extraClasses,classes);
      }
    }

    if (!justExtraClasses)
    {
      printList(*ofl,BinaryFunction(),binaryOps,classes,scalars);
      printList(*ofl,BinaryFunction(),binaryLeftOps,classes,scalars);
      printList(*ofl,BinaryFunction(),binaryOps,scalars,classes);
    }
    else
    {
      printList(*ofl,BinaryFunction(),binaryOps,extraClasses,scalars);
      printList(*ofl,BinaryFunction(),binaryLeftOps,extraClasses,scalars);
      printList(*ofl,BinaryFunction(),binaryOps,scalars,extraClasses);
    }

    // The following flag covers the common situation where you define
    // ostream << class.  Some compilers define cout to be a class that
    // inherits from ostream, so the compiler would use the PETE shift
    // operators T << class which defines shift for scalars and the user
    // class.  Since this shift operration is pretty bizzare, and stream
    // output is pretty common, the default behaviour of PETE is to turn
    // off the operators that allow for scalar << container and
    // scalar << expression.

    if (shiftGuard)
    {
      *ofl << "#ifdef PETE_ALLOW_SCALAR_SHIFT" << endl;
    }
    if (!justExtraClasses)
    {
      printList(*ofl,BinaryFunction(),binaryLeftOps,scalars,classes);
    }
    else
    {
      printList(*ofl,BinaryFunction(),binaryLeftOps,scalars,extraClasses);
    }
    if (shiftGuard)
    {
      *ofl << "#endif // PETE_ALLOW_SCALAR_SHIFT" << endl;
    }

    if (!justScalars)
    {
      if (!justExtraClasses)
      {
	printList(*ofl,TrinaryFunction(),trinaryOps,classes,classes,classes);
      }
      else
      {
	printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,extraClasses,extraClasses);
	printList(*ofl,TrinaryFunction(),trinaryOps,classes,extraClasses,extraClasses);
	printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,classes,extraClasses);
	printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,extraClasses,classes);
	printList(*ofl,TrinaryFunction(),trinaryOps,classes,classes,extraClasses);
	printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,classes,classes);
	printList(*ofl,TrinaryFunction(),trinaryOps,classes,extraClasses,classes);
      }
    }
    if (!justExtraClasses)
    {
      printList(*ofl,TrinaryFunction(),trinaryOps,classes,classes,scalars);
      printList(*ofl,TrinaryFunction(),trinaryOps,classes,scalars,classes);
      printList(*ofl,TrinaryFunction(),trinaryOps,classes,scalars,scalars);
    }
    else
    {
      printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,extraClasses,scalars);
      printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,scalars,extraClasses);
      printList(*ofl,TrinaryFunction(),trinaryOps,classes,extraClasses,scalars);
      printList(*ofl,TrinaryFunction(),trinaryOps,classes,scalars,extraClasses);
      printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,classes,scalars);
      printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,scalars,classes);
      printList(*ofl,TrinaryFunction(),trinaryOps,extraClasses,scalars,scalars);
    }
  }

  // Print assignment operators.
  
  if (assignment)
  {
    if (!justExtraClasses)
    {
      printList(*ofl,AssignFunctionForClass(),assignOps,userClasses);
    }
    else
    {
      printList(*ofl,AssignFunctionForClass(),assignOps,extraClasses);
    }
  }

  // Print footer.
  
  printFooter(*ofl,includeGuard,lanlBoilerplate);
  
  // Clean up.
  
  if (ofile != string(""))
    delete ofl;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MakeOperators.cpp,v $   $Author: sa_smith $
// $Revision: 1.13 $   $Date: 1999/02/03 21:40:04 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
