// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called PETE (Portable Expression Template Engine) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-99-5.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No.  W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE, and to allow others to do so.
// The public may copy and use this SOFTWARE, FOR NONCOMMERCIAL USE ONLY,
// without charge, provided that this Notice and any statement of
// authorship are reproduced on all copies.  Neither the Government nor
// the University makes any warranty, express or implied, or assumes any
// liability or responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about PETE, send e-mail to pete@acl.lanl.gov,
// or visit the PETE web page at http://www.acl.lanl.gov/pete/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef PETE_SRC_TOOLS_PARSER_H
#define PETE_SRC_TOOLS_PARSER_H

#include <iostream.h>
using std::getline;
#include <map>
using std::map;
#include <string>
using std::string;
#include <vector.h>

//-----------------------------------------------------------------------------
//
// ENUMERATION NAME
//    TokenType
//
// DESCRIPTION
//    The Token types for our little parser:
//      o four "keyword" tokens
//      o a group name token
//      o a separator token
//      o an equals sign token
//      o an end-o-line token
//      o an end-o-file token
//      o no token
//
//-----------------------------------------------------------------------------

enum TokenType { KEY1, KEY2, KEY3, GROUP, 
  SEP, EQUALS, STRING, EOL, EOFile, NOTOKEN };


//-----------------------------------------------------------------------------
//
// CLASS NAME
//    Token
//
// DESCRIPTION
//    The Tokens returned by our parser. Contains fields for getting the
//    text, type, and line number associated with the token.
//
//-----------------------------------------------------------------------------

class Token {
public:

  //---------------------------------------------------------------------------
  // Constructors and initialization functions.

  Token(const string &str)
  : str_m(str), pos_m(0), len_m(0), type_m(NOTOKEN), line_m(0)
  { }
  
  Token(const string &str, int pos, int len, TokenType type, int line)
  : str_m(str), pos_m(pos), len_m(len), type_m(type), line_m(line)
  { }

  void set(int pos, int len,  TokenType type, int line)
  {
    pos_m = pos;
    len_m = len;
    type_m = type;
    line_m = line;
  }
  
  //---------------------------------------------------------------------------
  // Return information about the token.

  string text() const
  {
    return str_m.substr(pos_m, len_m);
  }
  
  TokenType type() const
  {
    return type_m;
  }
  
  int line() const
  {
    return line_m;
  }
  
private:

  const string &str_m;
  int pos_m, len_m, line_m;
  TokenType type_m;
};

inline ostream &operator<<(ostream &os, const Token &tok)
{
  os << tok.type() << ": " << tok.text();
  
  return os;
}


//-----------------------------------------------------------------------------
//
// CLASS NAME
//    Lexer
//
// DESCRIPTION
//    A tiny lexical analyzer that turns text from an istream into tokens.
//
//-----------------------------------------------------------------------------

class Lexer {
public:

  //---------------------------------------------------------------------------
  // Constructor. Initializes the lexer and reads text into a string for future
  // processing.

  Lexer(istream &is, const string &fn, const string &kw1, const string &kw2, 
    const string &kw3)
  : file_m(fn), kw1_m(kw1), kw2_m(kw2), kw3_m(kw3), tok_m(str_m)
  {
    line_m = 1;
    pos_m = 0;

    // Read input file.
        
    string buffer;  
    while (getline(is, buffer) || is.gcount())
      {
        if (is.eof())
          {
            str_m += buffer;
          }
        else if (is.fail())
          {
            str_m += buffer;
            is.clear(is.rdstate() & ~ios::failbit);
          }
        else
          {
            str_m += buffer;
            str_m += '\n';
          }
      }
  }
  
  //---------------------------------------------------------------------------
  // Returns the next token.

  const Token &nextToken()
  {
    while (pos_m < str_m.size() && 
      (str_m[pos_m] == ' ' || str_m[pos_m] == '\t'))
        pos_m++;
        
    if (pos_m == str_m.size())
      {
        tok_m.set(pos_m, 0, EOFile, line_m);
      }
    else
      {
        char c = str_m[pos_m];
        if (c == '\n')
          {
            tok_m.set(pos_m++, 1, EOL, line_m++);
          }
        else if (c == '-')
          {
            int pos = pos_m;
            do {
              pos_m++;
            } while (pos_m < str_m.size() && str_m[pos_m] == '-');
            tok_m.set(pos, pos_m - pos, SEP, line_m);
          }
        else if (c == '=')
          {
            tok_m.set(pos_m++, 1, EQUALS, line_m);
          }
        else if (c == '\"')
          {
            int pos = pos_m;
            do {
              pos_m++;
            } while (pos_m < str_m.size() && 
                     str_m[pos_m] != '\"' /* && str_m[pos_m] != '\n' */);
            if (pos_m == str_m.size() /* || str_m[pos_m] == '\n' */)
              {
                cerr << "ERROR: untermininated string.\n"
                  << "File: \"" << file_m << "\"; line: " << line_m
                  << endl;
                exit(1);
              }
            else
              {
                tok_m.set(pos + 1, ++pos_m - pos - 2, STRING, line_m);
              }
          }
        else if (isalpha(c))
          {
            int pos = pos_m;
            do {
              pos_m++;
            } while (pos_m < str_m.size() && isalpha(str_m[pos_m]));
            if (str_m.substr(pos, pos_m - pos) == kw1_m)
              {
                tok_m.set(pos, pos_m - pos, KEY1, line_m);
              }
            else if  (str_m.substr(pos, pos_m - pos) == kw2_m)
              {
                tok_m.set(pos, pos_m - pos, KEY2, line_m);
              }
            else if  (str_m.substr(pos, pos_m - pos) == kw3_m)
              {
                tok_m.set(pos, pos_m - pos, KEY3, line_m);
              }
            else
              {
                tok_m.set(pos, pos_m - pos, GROUP, line_m);
              }
          }
        else
          {
            cerr << "ERROR: unrecognizable input found.\n"
              << "File: \"" << file_m << "\"; line: " << line_m
              << endl;
            exit(2);
          }
      }

    return tok_m;
  }            

  //---------------------------------------------------------------------------
  // Returns the current token.

  const Token &currentToken()
  {
    return tok_m;
  }
  
  //---------------------------------------------------------------------------
  // Matches the current token or else generates an error.

  void matchCurrent(TokenType type)
  {
    if (tok_m.type() != type)
      {
        cerr << "ERROR: syntax error.\n"
          << "File: \"" << file_m << "\"; line: " << tok_m.line()
          << endl;
        exit(3);
      }
  }
  
  //---------------------------------------------------------------------------
  // Matches the next token or else generates an error.

  void matchNext(TokenType type)
  {
    nextToken();
    matchCurrent(type);
  }
    
private:
  
  string file_m, kw1_m, kw2_m, kw3_m, str_m;
  Token tok_m;
  int pos_m, line_m;
};
            

//-----------------------------------------------------------------------------
//
// CLASS NAME
//    Parser
//
// DESCRIPTION
//    A tiny parser that reads our input file and stores data for each group
//    of descriptors into a map.
//
//-----------------------------------------------------------------------------

template<class Descriptor>
class Parser {
public:

  //---------------------------------------------------------------------------
  // Constructor.

  Parser(istream &is, const string &fn, const string &kw1, const string &kw2, 
    const string &kw3,
    map<string, vector<Descriptor> > &glist)
  : lexer_m(is, fn, kw1, kw2, kw3), glist_m(glist)
  { }
  
  //---------------------------------------------------------------------------
  // Recursive descent parse functions. Stores information into glist_m.

  void parse()
  {
    lexer_m.nextToken();
    while (lexer_m.currentToken().type() != EOFile)
      if (lexer_m.currentToken().type() == EOL)
        lexer_m.nextToken();
      else
        group();
  }
  
  void group()
  {
    lexer_m.matchCurrent(GROUP);
    group_m = lexer_m.currentToken().text();
    lexer_m.matchNext(EOL);

    lexer_m.nextToken();    
    dataList();
  }
  
  void dataList()
  {
    while (lexer_m.currentToken().type() == SEP)
      {
        lexer_m.matchNext(EOL);
        while (true)
          {
            const Token &tok = lexer_m.nextToken();
            if (tok.type() == KEY1 || tok.type() == KEY2 ||
                tok.type() == KEY3)
              {
                data(tok.type());
              }
            else
              break;
          }

        glist_m[group_m].push_back(Descriptor(d1_m, d2_m, d3_m));          
      }
  }
  
  void data(TokenType kw)
  {
    lexer_m.matchNext(EQUALS);
    lexer_m.matchNext(STRING);
    if (kw == KEY1)
      d1_m = lexer_m.currentToken().text();
    else if (kw == KEY2)
      d2_m = lexer_m.currentToken().text();
    else
      d3_m = lexer_m.currentToken().text();
    lexer_m.matchNext(EOL);
  }
  
private:

  Lexer lexer_m;
  string group_m, d1_m, d2_m, d3_m;
  map<string, vector<Descriptor> > &glist_m;
  
};
  
#endif // PETE_SRC_TOOLS_PARSER_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Parser.h,v $   $Author: sa_smith $
// $Revision: 1.6 $   $Date: 1999/02/03 21:40:04 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

    
