/*  Abstract Machine for the Small compiler
 *
 *  Copyright T. Riemersma, 1997, 1998
 */
#include <assert.h>
#include <limits.h>
#include <stdarg.h>
#include <string.h>
#include "amx.h"


#define CHKBOUNDS               /* remove this definition for a modest speedup*/
#define AMX_VERSION     0       /* current version */

typedef unsigned char uchar;

#define _namemax        16
typedef struct {
  cell address;
  char name[_namemax+1];
} AMX_FUNCSTUB;

typedef enum {
  OP_NONE,              /* invalid opcode */
  OP_LOAD_PRI,
  OP_LOAD_ALT,
  OP_LOAD_S_PRI,
  OP_LOAD_S_ALT,
  OP_LOAD_I,
  OP_LREF_PRI,
  OP_LREF_ALT,
  OP_LREF_S_PRI,
  OP_LREF_S_ALT,
  OP_CONST_PRI,
  OP_CONST_ALT,
  OP_ADDR_PRI,
  OP_ADDR_ALT,
  OP_STOR_PRI,
  OP_STOR_ALT,
  OP_STOR_S_PRI,
  OP_STOR_S_ALT,
  OP_STOR_I,
  OP_SREF_PRI,
  OP_SREF_ALT,
  OP_SREF_S_PRI,
  OP_SREF_S_ALT,
  OP_LIDX,
  OP_IDXADDR,
  OP_LCRTL,
  OP_SCRTL,
  OP_MOVE_PRI,
  OP_MOVE_ALT,
  OP_XCHG,
  OP_PUSH_PRI,
  OP_PUSH_ALT,
  OP_PUSH_C,
  OP_PUSH,
  OP_PUSH_S,
  OP_POP_PRI,
  OP_POP_ALT,
  OP_STACK,
  OP_HEAP,
  OP_PROC,
  OP_RET,
  OP_RETN,
  OP_CALL,
  OP_CALL_I,
  OP_JUMP,
  OP_JREL,
  OP_JZER,
  OP_JNZ,
  OP_JEQ,
  OP_JNEQ,
  OP_JLESS,
  OP_JLEQ,
  OP_JGRTR,
  OP_JGEQ,
  OP_JSLESS,
  OP_JSLEQ,
  OP_JSGRTR,
  OP_JSGEQ,
  OP_SHL,
  OP_SHR,
  OP_SSHR,
  OP_SHL_C_PRI,
  OP_SHL_C_ALT,
  OP_SHR_C_PRI,
  OP_SHR_C_ALT,
  OP_SMUL,
  OP_SDIV,
  OP_SDIV_ALT,
  OP_UMUL,
  OP_UDIV,
  OP_UDIV_ALT,
  OP_ADD,
  OP_SUB,
  OP_SUB_ALT,
  OP_AND,
  OP_OR,
  OP_XOR,
  OP_NOT,
  OP_NEG,
  OP_INVERT,
  OP_ADD_C,
  OP_SMUL_C,
  OP_ZERO_PRI,
  OP_ZERO_ALT,
  OP_ZERO,
  OP_ZERO_S,
  OP_SIGN_PRI,
  OP_SIGN_ALT,
  OP_EQ,
  OP_NEQ,
  OP_LESS,
  OP_LEQ,
  OP_GRTR,
  OP_GEQ,
  OP_SLESS,
  OP_SLEQ,
  OP_SGRTR,
  OP_SGEQ,
  OP_EQ_C_PRI,
  OP_EQ_C_ALT,
  OP_INC_PRI,
  OP_INC_ALT,
  OP_INC,
  OP_INC_S,
  OP_INC_I,
  OP_DEC_PRI,
  OP_DEC_ALT,
  OP_DEC,
  OP_DEC_S,
  OP_DEC_I,
  OP_MOVS,
  OP_CMPS,
  OP_FILL,
  OP_HALT,
  OP_BOUNDS,
  OP_SYSREQ_PRI,
  OP_SYSREQ_C,
  OP_FILE,
  OP_LINE,
  OP_SYMBOL,
} OPCODE;

int amx_Version(AMX *amx, int *fileversion, int *requiredversion, int *curversion)
{
  if (amx!=NULL) {
    AMX_HEADER *hdr=(AMX_HEADER *)amx->base;
    if (fileversion!=NULL)
      *fileversion=hdr->file_version;
    if (requiredversion!=NULL)
      *requiredversion=hdr->amx_version;
  } /* if */
  if (curversion!=NULL)
    *curversion=AMX_VERSION;
  return AMX_ERR_NONE;
}

int amx_Callback(AMX *amx, cell index, cell *result, cell *params)
{
  AMX_HEADER *hdr=(AMX_HEADER *)amx->base;
  AMX_FUNCSTUB *func=(AMX_FUNCSTUB *)(amx->base+(int)hdr->natives+(int)index*sizeof(AMX_FUNCSTUB));
  AMX_NATIVE f=(AMX_NATIVE)func->address;
  assert(f!=NULL);
  assert(index<hdr->num_natives);

  /* Note:
   *   params[0] == number of parameters passed to the native function
   *   params[1] == first argument
   *   etc.
   */

  amx->error=AMX_ERR_NONE;
  *result = f(amx,params);
  return amx->error;
}

#pragma argsused
int amx_Debug(AMX *amx, int code, cell p1, cell p2, char *name)
{
  return AMX_ERR_NONE;
}

int amx_Init(AMX *amx,void *program)
{
  AMX_HEADER *hdr;

  memset(amx,0,sizeof(AMX));
  hdr=(AMX_HEADER *)program;
  if ((unsigned short)hdr->magic!=0xf1e0 || hdr->defsize!=sizeof(AMX_FUNCSTUB))
    return AMX_ERR_FORMAT;
  if (hdr->amx_version>AMX_VERSION)
    return AMX_ERR_VERSION;
  assert(hdr->hea == hdr->size);
  if (hdr->stp>INT_MAX || hdr->stp<=0)
    return AMX_ERR_FORMAT;

  amx->base=program;

  /* Set a zero cell at the top of the stack, which functions
   * as a sentinel for strings.
   */
  hdr->stp -= sizeof(cell);
  * (cell *)(amx->base+(int)hdr->stp) = 0;

  /* set initial values */
  amx->hea=hdr->hea - hdr->dat; /* stack and heap relative to data segment */
  amx->stp=hdr->stp - hdr->dat;
  amx->stk=amx->stp;
  amx->callback=amx_Callback;
  amx->debug=amx_Debug;
  amx->curline=0;
  amx->curfile=0;

  return AMX_ERR_NONE;
}

int amx_Release(AMX *amx)
{
  if (amx->base!=NULL) {
    amx->base=NULL;
    return AMX_ERR_NONE;
  } else {
    return AMX_ERR_FORMAT;
  } /* if */
}

int amx_NameLength(AMX *amx, int *length)
{
  AMX_HEADER *hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  *length=hdr->defsize - sizeof(cell);
  return AMX_ERR_NONE;
}

int amx_NumModules(AMX *amx, int *number)
{
  AMX_HEADER *hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  *number=hdr->num_modules;
  return AMX_ERR_NONE;
}

int amx_GetModule(AMX *amx, int index, char *modulename)
{
  AMX_HEADER *hdr;
  AMX_FUNCSTUB *mod;

  hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  if (index>=hdr->num_modules)
    return AMX_ERR_INDEX;

  mod=(AMX_FUNCSTUB *)(amx->base+(int)hdr->modules+index*sizeof(AMX_FUNCSTUB));
  strcpy(modulename,mod->name);
  return AMX_ERR_NONE;
}


int amx_NumPublics(AMX *amx, int *number)
{
  AMX_HEADER *hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  *number=hdr->num_publics;
  return AMX_ERR_NONE;
}

int amx_GetPublic(AMX *amx, int index, char *funcname)
{
  AMX_HEADER *hdr;
  AMX_FUNCSTUB *func;

  hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  if (index>=hdr->num_publics)
    return AMX_ERR_INDEX;

  func=(AMX_FUNCSTUB *)(amx->base+(int)hdr->publics+index*sizeof(AMX_FUNCSTUB));
  strcpy(funcname,func->name);
  return AMX_ERR_NONE;
}

int amx_GetUserData(AMX *amx, int index, void **ptr)
{
  if (index<0 || index>=AMX_USERNUM)
    return AMX_ERR_INDEX;
  *ptr=amx->userdata[index];
  return AMX_ERR_NONE;
}

int amx_SetUserData(AMX *amx, int index, void *ptr)
{
  if (index<0 || index>=AMX_USERNUM)
    return AMX_ERR_INDEX;
  amx->userdata[index]=ptr;
  return AMX_ERR_NONE;
}

static int findmodule(AMX *amx,char *name)
{
  AMX_HEADER *hdr;
  AMX_FUNCSTUB *mod;
  int i;

  hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  mod=(AMX_FUNCSTUB *)(amx->base+(int)hdr->modules);
  for (i=0; i<hdr->num_modules; i++) {
    if (strcmp(mod->name,name)==0)
      break;
    mod++;
  } /* for */
  if (i==hdr->num_modules)
    return -1;                  /* nothing to do (this is no error) */
  return i;
}

int amx_Register(AMX *amx, char *modulename, int (*callback)(char*,AMX_NATIVE*))
{
  AMX_FUNCSTUB *mod, *func;
  AMX_HEADER *hdr;
  int i,mod_id,err;
  AMX_NATIVE funcptr;

  hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  mod_id=findmodule(amx,modulename);
  assert(mod_id==-1 || mod_id>=0 && mod_id<hdr->num_modules);
  if (mod_id==-1)
    return AMX_ERR_NONE;        /* nothing to do (this is no error) */

  err=0;
  func=(AMX_FUNCSTUB *)(amx->base+(int)hdr->natives);
  for (i=0; i<hdr->num_natives; i++) {
    if (func->address==mod_id) {
      if (callback(func->name,&funcptr)==AMX_ERR_NONE)
        func->address=(ucell)funcptr;
      else
        err++;
    } /* if */
    func++;
  } /* for */

  if (err)
    return AMX_ERR_NOTFOUND;
  mod=(AMX_FUNCSTUB *)(amx->base+(int)hdr->modules+mod_id*sizeof(AMX_FUNCSTUB));
  mod->address=1;              /* this module is relocated */
  return AMX_ERR_NONE;
}

int amx_Unregister(AMX *amx, char *modulename, int (*callback)(char*,AMX_NATIVE*))
{
  AMX_FUNCSTUB *mod;
  AMX_HEADER *hdr;
  int mod_id;
  AMX_NATIVE funcptr;
  cell params[1] = { 0 };       /* dummy parameters */

  hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  mod_id=findmodule(amx,modulename);
  assert(mod_id==-1 || mod_id>=0 && mod_id<hdr->num_modules);
  mod=(AMX_FUNCSTUB *)(amx->base+(int)hdr->modules+mod_id*sizeof(AMX_FUNCSTUB));
  if (mod->address!=1)
    return AMX_ERR_NONE;        /* this module is not relocated */

  if (callback("",&funcptr)==AMX_ERR_NONE)
    funcptr(amx,params);
  return AMX_ERR_NONE;
}

int amx_CheckModules(AMX *amx)
{
  AMX_FUNCSTUB *mod;
  AMX_HEADER *hdr;
  int i;

  hdr=(AMX_HEADER *)amx->base;
  assert(hdr!=NULL);
  mod=(AMX_FUNCSTUB *)(amx->base+(int)hdr->modules);
  for (i=0; i<hdr->num_modules; i++) {
    if (mod->address==0)
      return AMX_ERR_MODULE;
    mod++;
  } /* for */
  return AMX_ERR_NONE;
}

#define GETPARAM(v)     ( v=*(cell *)(code+(int)cip), cip+=sizeof(cell) )
#define PUSH(v)         ( stk-=sizeof(cell), *(cell *)(data+(int)stk)=v )
#define POP(v)          ( v=*(cell *)(data+(int)stk), stk+=sizeof(cell) )

#if defined CHKBOUNDS
  #define STKMARGIN     (16*sizeof(cell))
  #define CHKMARGIN()   if (hea+STKMARGIN>stk) return AMX_ERR_STACKERR
  #define CHKSTACK()    if (stk>hdr->stp) return AMX_ERR_STACKLOW
  #define CHKHEAP()     if (hea<0) return AMX_ERR_HEAPLOW
#else
  #define CHKMARGIN()
  #define CHKSTACK()
  #define CHKHEAP()
#endif

int amx_DebugBrowse(AMX *amx)
{
  AMX_HEADER *hdr;
  uchar *code;
  cell cip;
  long codesize;
  OPCODE op;

  hdr=(AMX_HEADER *)amx->base;
  code=amx->base+(int)hdr->cod;
  codesize=hdr->dat - hdr->cod;

  /* sanity checks */
  assert(OP_PUSH_PRI==30);
  assert(OP_PROC==39);
  assert(OP_SHL==58);
  assert(OP_SMUL==65);
  assert(OP_EQ==88);
  assert(OP_INC_PRI==100);
  assert(OP_MOVS==110);
  assert(OP_SYMBOL==119);

  /* start browsing code */
  for (cip=0; cip<codesize; ) {
    op=(OPCODE) code[(int)cip];
    cip++;
    switch (op) {
    case OP_LOAD_PRI:   /* instructions with 1 parameter */
    case OP_LOAD_ALT:
    case OP_LOAD_S_PRI:
    case OP_LOAD_S_ALT:
    case OP_LREF_PRI:
    case OP_LREF_ALT:
    case OP_LREF_S_PRI:
    case OP_LREF_S_ALT:
    case OP_CONST_PRI:
    case OP_CONST_ALT:
    case OP_ADDR_PRI:
    case OP_ADDR_ALT:
    case OP_STOR_PRI:
    case OP_STOR_ALT:
    case OP_STOR_S_PRI:
    case OP_STOR_S_ALT:
    case OP_SREF_PRI:
    case OP_SREF_ALT:
    case OP_SREF_S_PRI:
    case OP_SREF_S_ALT:
    case OP_LCRTL:
    case OP_SCRTL:
    case OP_PUSH_C:
    case OP_PUSH:
    case OP_PUSH_S:
    case OP_STACK:
    case OP_HEAP:
    case OP_CALL:
    case OP_JUMP:
    case OP_JREL:
    case OP_JZER:
    case OP_JNZ:
    case OP_JEQ:
    case OP_JNEQ:
    case OP_JLESS:
    case OP_JLEQ:
    case OP_JGRTR:
    case OP_JGEQ:
    case OP_JSLESS:
    case OP_JSLEQ:
    case OP_JSGRTR:
    case OP_JSGEQ:
    case OP_SHL_C_PRI:
    case OP_SHL_C_ALT:
    case OP_SHR_C_PRI:
    case OP_SHR_C_ALT:
    case OP_ADD_C:
    case OP_SMUL_C:
    case OP_ZERO:
    case OP_ZERO_S:
    case OP_EQ_C_PRI:
    case OP_EQ_C_ALT:
    case OP_INC:
    case OP_INC_S:
    case OP_DEC:
    case OP_DEC_S:
    case OP_MOVS:
    case OP_CMPS:
    case OP_FILL:
    case OP_HALT:
    case OP_BOUNDS:
    case OP_SYSREQ_C:
      cip+=sizeof(cell);
      break;

    case OP_LOAD_I:     /* instructions without parameters */
    case OP_STOR_I:
    case OP_LIDX:
    case OP_IDXADDR:
    case OP_MOVE_PRI:
    case OP_MOVE_ALT:
    case OP_XCHG:
    case OP_PUSH_PRI:
    case OP_PUSH_ALT:
    case OP_POP_PRI:
    case OP_POP_ALT:
    case OP_PROC:
    case OP_RET:
    case OP_RETN:
    case OP_CALL_I:
    case OP_SHL:
    case OP_SHR:
    case OP_SSHR:
    case OP_SMUL:
    case OP_SDIV:
    case OP_SDIV_ALT:
    case OP_UMUL:
    case OP_UDIV:
    case OP_UDIV_ALT:
    case OP_ADD:
    case OP_SUB:
    case OP_SUB_ALT:
    case OP_AND:
    case OP_OR:
    case OP_XOR:
    case OP_NOT:
    case OP_NEG:
    case OP_INVERT:
    case OP_ZERO_PRI:
    case OP_ZERO_ALT:
    case OP_SIGN_PRI:
    case OP_SIGN_ALT:
    case OP_EQ:
    case OP_NEQ:
    case OP_LESS:
    case OP_LEQ:
    case OP_GRTR:
    case OP_GEQ:
    case OP_SLESS:
    case OP_SLEQ:
    case OP_SGRTR:
    case OP_SGEQ:
    case OP_INC_PRI:
    case OP_INC_ALT:
    case OP_INC_I:
    case OP_DEC_PRI:
    case OP_DEC_ALT:
    case OP_DEC_I:
    case OP_SYSREQ_PRI:
      break;

    case OP_FILE: {
      int num=code[(int)cip];
      cell offs;
      cip+=num+1;
      GETPARAM(offs);
      amx->debug(amx, DBG_FILE, offs, 0, (char *)(code+(int)cip-num-sizeof(cell)));
      break;
    } /* case */
    case OP_LINE:
      cip+=2*sizeof(cell);
      break;
    case OP_SYMBOL: {
      unsigned char t, g;
      int num=code[(int)cip];
      cell offs;
      cip+=num+1;
      GETPARAM(offs);
      t = code[(int)cip++];
      g = code[(int)cip++];
      if (g==1)         /* do global symbols only */
        amx->debug(amx, DBG_SYMBOL, offs, t + 256*g,
                   (char *)(code+(int)cip-num-2-sizeof(cell)));
      break;
    } /* case */
    default:
      return AMX_ERR_INVINSTR;
    } /* switch */
  } /* for */
  return AMX_ERR_NONE;
}

int amx_Exec(AMX *amx, cell *retval, int index, int numparams, ...)
{
  AMX_HEADER *hdr;
  AMX_FUNCSTUB *func;
  uchar *code, *data;
  cell pri,alt,stk,frm,hea,cip;
  cell offs;
  int num,i;
  OPCODE op;
  va_list ap;

  /* get the start address */
  hdr=(AMX_HEADER *)amx->base;
  if (index==AMX_EXEC_MAIN) {
    cip=hdr->cip;
  } else if (index<0 && index!=AMX_EXEC_CONT) {
    return AMX_ERR_INDEX;
  } else {
    if (index>=hdr->num_publics)
      return AMX_ERR_INDEX;
    func=(AMX_FUNCSTUB *)(amx->base + (int)hdr->publics + index*sizeof(AMX_FUNCSTUB));
    cip=func->address;
  } /* if */

  /* set up the registers */
  code=amx->base+(int)hdr->cod;
  data=amx->base+(int)hdr->dat;
  hea=amx->hea;
  stk=amx->stk;

  /* sanity checks */
  assert(amx_CheckModules(amx)==AMX_ERR_NONE);
  assert(OP_PUSH_PRI==30);
  assert(OP_PROC==39);
  assert(OP_SHL==58);
  assert(OP_SMUL==65);
  assert(OP_EQ==88);
  assert(OP_INC_PRI==100);
  assert(OP_MOVS==110);
  assert(OP_SYMBOL==119);
  assert(sizeof(cell)==4);

  /* push the parameters to the stack (in reverse order) */
  stk-=numparams*sizeof(cell);
  va_start(ap,numparams);
  for (i=0; i<numparams; i++)
    *(cell *)(data+(int)stk+i)=va_arg(ap,cell);
  va_end(ap);
  PUSH(numparams*sizeof(cell));
  PUSH(0);      /* zero return address */

  /* start running */
  frm=0;        /* just to avoid compiler warnings */
  for ( ;; ) {
    op=(OPCODE) code[(int)cip];
    cip++;
    switch (op) {
    case OP_LOAD_PRI:
      GETPARAM(offs);
      pri= * (cell *)(data+(int)offs);
      break;
    case OP_LOAD_ALT:
      GETPARAM(offs);
      alt= * (cell *)(data+(int)offs);
      break;
    case OP_LOAD_S_PRI:
      GETPARAM(offs);
      pri= * (cell *)(data+(int)frm+(int)offs);
      break;
    case OP_LOAD_S_ALT:
      GETPARAM(offs);
      alt= * (cell *)(data+(int)frm+(int)offs);
      break;
    case OP_LOAD_I:
      /* verify address */
      if (pri>=hea && pri<stk || pri>=amx->stp)
        return AMX_ERR_MEMACCESS;
      pri= * (cell *)(data+(int)pri);
      break;
    case OP_LREF_PRI:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)offs);
      pri= * (cell *)(data+(int)offs);
      break;
    case OP_LREF_ALT:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)offs);
      alt= * (cell *)(data+(int)offs);
      break;
    case OP_LREF_S_PRI:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)frm+(int)offs);
      pri= * (cell *)(data+(int)offs);
      break;
    case OP_LREF_S_ALT:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)frm+(int)offs);
      alt= * (cell *)(data+(int)offs);
      break;
    case OP_CONST_PRI:
      GETPARAM(pri);
      break;
    case OP_CONST_ALT:
      GETPARAM(alt);
      break;
    case OP_ADDR_PRI:
      GETPARAM(pri);
      pri+=frm;
      break;
    case OP_ADDR_ALT:
      GETPARAM(alt);
      alt+=frm;
      break;
    case OP_STOR_PRI:
      GETPARAM(offs);
      *(cell *)(data+(int)offs)=pri;
      break;
    case OP_STOR_ALT:
      GETPARAM(offs);
      *(cell *)(data+(int)offs)=alt;
      break;
    case OP_STOR_S_PRI:
      GETPARAM(offs);
      *(cell *)(data+(int)frm+(int)offs)=pri;
      break;
    case OP_STOR_S_ALT:
      GETPARAM(offs);
      *(cell *)(data+(int)frm+(int)offs)=alt;
      break;
    case OP_STOR_I:
      /* verify address */
      if (alt>=hea && alt<stk || alt>=amx->stp)
        return AMX_ERR_MEMACCESS;
      *(cell *)(data+(int)alt)=pri;
      break;
    case OP_SREF_PRI:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)offs);
      *(cell *)(data+(int)offs)=pri;
      break;
    case OP_SREF_ALT:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)offs);
      *(cell *)(data+(int)offs)=alt;
      break;
    case OP_SREF_S_PRI:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)frm+(int)offs);
      *(cell *)(data+(int)offs)=pri;
      break;
    case OP_SREF_S_ALT:
      GETPARAM(offs);
      offs= * (cell *)(data+(int)frm+(int)offs);
      *(cell *)(data+(int)offs)=alt;
      break;
    case OP_LIDX:
      offs=pri*sizeof(cell) + alt;
      /* verify address */
      if (offs>=hea && offs<stk || offs>=amx->stp)
        return AMX_ERR_MEMACCESS;
      pri= * (cell *)(data+(int)offs);
      break;
    case OP_IDXADDR:
      pri=pri*sizeof(cell)+alt;
      break;
    case OP_LCRTL:
      GETPARAM(offs);
      switch (offs) {
      case 0:
        pri=hdr->cod;
        break;
      case 1:
        pri=hdr->dat;
        break;
      case 2:
        pri=hea;
        break;
      case 3:
        pri=amx->stp;
        break;
      case 4:
        pri=stk;
        break;
      case 5:
        pri=frm;
        break;
      case 6:
        pri=cip;
        break;
      } /* switch */
      break;
    case OP_SCRTL:
      GETPARAM(offs);
      switch (offs) {
      case 0:
      case 1:
      case 3:
        /* cannot change these parameters */
        break;
      case 2:
        hea=pri;
        break;
      case 4:
        stk=pri;
        break;
      case 5:
        frm=pri;
        break;
      case 6:
        cip=pri;
        break;
      } /* switch */
      break;
    case OP_MOVE_PRI:
      pri=alt;
      break;
    case OP_MOVE_ALT:
      alt=pri;
      break;
    case OP_XCHG:
      offs=pri;         /* offs is a temporary variable */
      pri=alt;
      alt=offs;
      break;
    case OP_PUSH_PRI:
      PUSH(pri);
      break;
    case OP_PUSH_ALT:
      PUSH(alt);
      break;
    case OP_PUSH_C:
      GETPARAM(offs);
      PUSH(offs);
      break;
    case OP_PUSH:
      GETPARAM(offs);
      PUSH(* (cell *)(data+(int)offs));
      break;
    case OP_PUSH_S:
      GETPARAM(offs);
      PUSH(* (cell *)(data+(int)frm+(int)offs));
      break;
    case OP_POP_PRI:
      POP(pri);
      break;
    case OP_POP_ALT:
      POP(alt);
      break;
    case OP_STACK:
      GETPARAM(offs);
      alt=stk;
      stk+=offs;
      CHKMARGIN();
      CHKSTACK();
      if (offs>0)
        amx->debug(amx, DBG_CLRSYM, stk, 0, NULL);
      break;
    case OP_HEAP:
      GETPARAM(offs);
      alt=hea;
      hea+=offs;
      CHKMARGIN();
      CHKHEAP();
      break;
    case OP_PROC:
      PUSH(frm);
      frm=stk;
      break;
    case OP_RET:
      POP(frm);
      POP(cip);
      if (cip==0) {
        /* entry function returns -> end of program */
        *retval=pri;
        amx->debug(amx, DBG_TERMINATE, 0, 0, NULL);
        return AMX_ERR_NONE;
      } else {
        amx->debug(amx, DBG_RETURN, 0, 0, NULL);
      } /* if */
      break;
    case OP_RETN:
      POP(frm);
      POP(cip);
      if (cip==0) {
        /* entry function returns -> end of program */
        *retval=pri;
        amx->debug(amx, DBG_TERMINATE, 0, 0, NULL);
        return AMX_ERR_NONE;
      } else {
        amx->debug(amx, DBG_RETURN, 0, 0, NULL);
      } /* if */
      stk+= *(cell *)(data+(int)stk) + 4;
      break;
    case OP_CALL:
      PUSH(cip+4);      /* skip address */
      cip= * (cell *)(code+(int)cip);   /* jump to the address */
      amx->debug(amx, DBG_CALL, 0, 0, NULL);
      break;
    case OP_CALL_I:
      PUSH(cip);
      cip=pri;
      amx->debug(amx, DBG_CALL, 0, 0, NULL);
      break;
    case OP_JUMP:
      /* since the GETPARAM() macro modifies cip, you cannot
       * do GETPARAM(cip) directly */
      cip= * (cell *)(code+(int)cip);
      break;
    case OP_JREL:
      cip+= * (cell *)(code+(int)cip) + sizeof(cell);
      break;
    case OP_JZER:
      if (pri==0)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JNZ:
      if (pri!=0)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JEQ:
      if (pri==alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JNEQ:
      if (pri!=alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JLESS:
      if ((ucell)pri < (ucell)alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JLEQ:
      if ((ucell)pri <= (ucell)alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JGRTR:
      if ((ucell)pri > (ucell)alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JGEQ:
      if ((ucell)pri >= (ucell)alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JSLESS:
      if (pri<alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JSLEQ:
      if (pri<=alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JSGRTR:
      if (pri>alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_JSGEQ:
      if (pri>=alt)
        cip= * (cell *)(code+(int)cip);
      else
        cip+=sizeof(cell);
      break;
    case OP_SHL:
      pri<<=alt;
      break;
    case OP_SHR:
      pri=(ucell)pri >> (ucell)alt;
      break;
    case OP_SSHR:
      pri>>=alt;
      break;
    case OP_SHL_C_PRI:
      GETPARAM(offs);
      pri<<=offs;
      break;
    case OP_SHL_C_ALT:
      GETPARAM(offs);
      alt<<=offs;
      break;
    case OP_SHR_C_PRI:
      GETPARAM(offs);
      pri=(ucell)pri >> (ucell)offs;
      break;
    case OP_SHR_C_ALT:
      GETPARAM(offs);
      alt=(ucell)alt >> (ucell)offs;
      break;
    case OP_SMUL:
      pri*=alt;
      break;
    case OP_SDIV:
      /* divide must always round down; this is a bit
       * involved to do in a machine-independent way.
       */
      offs=(pri % alt + alt) % alt;     /* true modulus */
      pri=(pri - offs) / alt;           /* division result */
      alt=offs;
      break;
    case OP_SDIV_ALT:
      /* divide must always round down; this is a bit
       * involved to do in a machine-independent way.
       */
      offs=(alt % pri + pri) % pri;     /* true modulus */
      pri=(alt - offs) / pri;           /* division result */
      alt=offs;
      break;
    case OP_UMUL:
      pri=(ucell)pri * (ucell)alt;
      break;
    case OP_UDIV:
      offs=(ucell)pri % (ucell)alt;     /* temporary storage */
      pri=(ucell)pri / (ucell)alt;
      alt=offs;
      break;
    case OP_UDIV_ALT:
      offs=(ucell)alt % (ucell)pri;     /* temporary storage */
      pri=(ucell)alt / (ucell)pri;
      alt=offs;
      break;
    case OP_ADD:
      pri+=alt;
      break;
    case OP_SUB:
      pri-=alt;
      break;
    case OP_SUB_ALT:
      pri=alt-pri;
      break;
    case OP_AND:
      pri&=alt;
      break;
    case OP_OR:
      pri|=alt;
      break;
    case OP_XOR:
      pri^=alt;
      break;
    case OP_NOT:
      pri=!pri;
      break;
    case OP_NEG:
      pri=-pri;
      break;
    case OP_INVERT:
      pri=~pri;
      break;
    case OP_ADD_C:
      GETPARAM(offs);
      pri+=offs;
      break;
    case OP_SMUL_C:
      GETPARAM(offs);
      pri*=offs;
      break;
    case OP_ZERO_PRI:
      pri=0;
      break;
    case OP_ZERO_ALT:
      alt=0;
      break;
    case OP_ZERO:
      GETPARAM(offs);
      *(cell *)(data+(int)offs)=0;
      break;
    case OP_ZERO_S:
      GETPARAM(offs);
      *(cell *)(data+(int)frm+(int)offs)=0;
      break;
    case OP_SIGN_PRI:
      if ((pri & 0xff)>=0x80)
        pri|= ~ (ucell)0xff;
      break;
    case OP_SIGN_ALT:
      if ((alt & 0xff)>=0x80)
        alt|= ~ (ucell)0xff;
      break;
    case OP_EQ:
      pri= pri==alt ? 1 : 0;
      break;
    case OP_NEQ:
      pri= pri!=alt ? 1 : 0;
      break;
    case OP_LESS:
      pri= (ucell)pri < (ucell)alt ? 1 : 0;
      break;
    case OP_LEQ:
      pri= (ucell)pri <= (ucell)alt ? 1 : 0;
      break;
    case OP_GRTR:
      pri= (ucell)pri > (ucell)alt ? 1 : 0;
      break;
    case OP_GEQ:
      pri= (ucell)pri >= (ucell)alt ? 1 : 0;
      break;
    case OP_SLESS:
      pri= pri<alt ? 1 : 0;
      break;
    case OP_SLEQ:
      pri= pri<=alt ? 1 : 0;
      break;
    case OP_SGRTR:
      pri= pri>alt ? 1 : 0;
      break;
    case OP_SGEQ:
      pri= pri>=alt ? 1 : 0;
      break;
    case OP_EQ_C_PRI:
      GETPARAM(offs);
      pri= pri==offs ? 1 : 0;
      break;
    case OP_EQ_C_ALT:
      GETPARAM(offs);
      pri= alt==offs ? 1 : 0;
      break;
    case OP_INC_PRI:
      pri++;
      break;
    case OP_INC_ALT:
      alt++;
      break;
    case OP_INC:
      GETPARAM(offs);
      *(cell *)(data+(int)offs) += 1;
      break;
    case OP_INC_S:
      GETPARAM(offs);
      *(cell *)(data+(int)frm+(int)offs) += 1;
      break;
    case OP_INC_I:
      *(cell *)(data+(int)pri) += 1;
      break;
    case OP_DEC_PRI:
      pri--;
      break;
    case OP_DEC_ALT:
      alt--;
      break;
    case OP_DEC:
      GETPARAM(offs);
      *(cell *)(data+(int)offs) -= 1;
      break;
    case OP_DEC_S:
      GETPARAM(offs);
      *(cell *)(data+(int)frm+(int)offs) -= 1;
      break;
    case OP_DEC_I:
      *(cell *)(data+(int)pri) -= 1;
      break;
    case OP_MOVS:
      GETPARAM(offs);
      memcpy(data+(int)alt, data+(int)pri, (int)offs);
      break;
    case OP_CMPS:
      GETPARAM(offs);
      pri=memcmp(data+(int)alt, data+(int)pri, (int)offs);
      break;
    case OP_FILL:
      GETPARAM(offs);
      for (i=(int)alt; offs>=sizeof(cell); i+=sizeof(cell), offs-=sizeof(cell))
        *(cell *)(data+i) = pri;
      break;
    case OP_HALT:
      GETPARAM(offs);
      *retval=pri;
      return (int)offs;
    case OP_BOUNDS:
      GETPARAM(offs);
      if (pri>offs || pri<0)
        return AMX_ERR_BOUNDS;
      break;
    case OP_SYSREQ_PRI:
      if (amx->callback==NULL)
        return AMX_ERR_CALLBACK;
      /* save a few registers */
      amx->cip=cip;
      amx->hea=hea;
      amx->frm=frm;
      amx->stk=stk;
      num=amx->callback(amx,pri,&pri,(cell *)(data+(int)stk));
      if (num!=AMX_ERR_NONE)
        return num;
      /* copy cip back */
      cip=amx->cip;
      break;
    case OP_SYSREQ_C:
      GETPARAM(offs);
      if (amx->callback==NULL)
        return AMX_ERR_CALLBACK;
      /* save a few registers */
      amx->cip=cip;
      amx->hea=hea;
      amx->frm=frm;
      amx->stk=stk;
      num=amx->callback(amx,offs,&pri,(cell *)(data+(int)stk));
      if (num!=AMX_ERR_NONE)
        return num;
      /* copy cip back */
      cip=amx->cip;
      break;
    case OP_FILE:
      num=code[(int)cip];
      cip+=num+1;
      GETPARAM(offs);
      assert(0);        /* this code should not occur during execution */
      amx->debug(amx, DBG_FILE, offs, 0, (char *)(code+(int)cip-num-sizeof(cell)));
      break;
    case OP_LINE:
      GETPARAM(amx->curline);
      GETPARAM(amx->curfile);
      amx->debug(amx, DBG_LINE, amx->curline, amx->curfile, NULL);
      break;
    case OP_SYMBOL: {
      unsigned char t, g;
      num=code[(int)cip];
      cip+=num+1;
      GETPARAM(offs);
      t = code[(int)cip++];
      g = code[(int)cip++];
      assert(g==0);     /* local symbols only */
      amx->debug(amx, DBG_SYMBOL, offs, t + 256*g,
                 (char *)(code+(int)cip-num-2-sizeof(cell)));
      break;
    } /* case */
    default:
      return AMX_ERR_INVINSTR;
    } /* switch */
  } /* for */
}

int amx_SetCallback(AMX *amx,AMX_CALLBACK callback)
{
  amx->callback=callback;
  return AMX_ERR_NONE;
}

int amx_SetDebugProc(AMX *amx, AMX_DEBUG debug)
{
  amx->debug=debug;
  return AMX_ERR_NONE;
}

int amx_RaiseError(AMX *amx, int error)
{
  assert(error>0);
  amx->error=error;
  return AMX_ERR_NONE;
}

int amx_GetAddr(AMX *amx,cell v,cell **addr)
{
  AMX_HEADER *hdr=(AMX_HEADER *)amx->base;

  if (v>=amx->hea && v<amx->stk || v>=amx->stp)
    return AMX_ERR_MEMACCESS;
  *addr=(cell *)(amx->base + (int)(hdr->dat + v));
  return AMX_ERR_NONE;
}

