/*  Abstract Machine for the Small compiler
 *
 *  Copyright T. Riemersma, 1997, 1998
 */

typedef long cell;
typedef unsigned long ucell;

struct __amx;
typedef cell (*AMX_NATIVE)(struct __amx *amx, cell *params);
typedef int (*AMX_CALLBACK)(struct __amx *amx, cell index, cell *result,
                            cell *params);
typedef int (*AMX_DEBUG)(struct __amx *amx, int code, cell p1, cell p2,
                         char *name);

#pragma pack(1)         /* structures must be packed (byte-aligned) */
#if defined __TURBOC__
  #pragma option -a-    /* "pack" pragma for older Borland compilers */
#endif

#define AMX_USERNUM     4

/* The AMX structure is the internal structure for many functions. Not all
 * fields are valid at all times; many fields are cached in local variables.
 */
typedef struct __amx {
  unsigned char *base;
  AMX_CALLBACK callback;
  AMX_DEBUG debug;
  void *userdata[AMX_USERNUM];
  /* for assertions */
  cell curline, curfile;
  /* native functions can raise an error */
  short error;
  /* for exernal functions, a few registers must be accessible from the outside */
  cell cip, frm, hea, stk, stp;
} AMX;

/* The AMX_HEADER structure is both the memory format as the file format. The
 * structure is used internaly.
 */
typedef struct {
  long size;            /* size of the "file" */
  short magic;          /* signature */
  char file_version;    /* file format version */
  char amx_version;     /* required version of the AMX */
  short flags;
  short defsize;
  long cod;             /* initial value of COD - code block */
  long dat;             /* initial value of DAT - data block */
  long hea;             /* initial value of HEA - start of the heap */
  long stp;             /* initial value of STP - stack top */
  long cip;             /* initial value of CIP - the instruction pointer */
  short num_publics;    /* number of items in the "public functions" table */
  long publics;         /* offset to the "public functions" table */
  short num_natives;    /* number of items in the "native functions" table */
  long natives;         /* offset to the "native functions" table */
  short num_modules;    /* number of items in the "module table" */
  long modules;         /* offset to the "module table" */
} AMX_HEADER;

enum {
  AMX_ERR_NONE,
  /* reserve the first 15 error codes for exit codes of the abstract machine */
  AMX_ERR_EXIT,         /* forced exit */
  AMX_ERR_ASSERT,       /* assertion failed */
  AMX_ERR_STACKERR,     /* stack/heap collision */
  AMX_ERR_BOUNDS,       /* index out of bounds */
  AMX_ERR_MEMACCESS,    /* invalid memory access */
  AMX_ERR_INVINSTR,     /* invalid instruction */
  AMX_ERR_STACKLOW,     /* stack underflow */
  AMX_ERR_HEAPLOW,      /* heap underflow */
  AMX_ERR_CALLBACK,     /* no callback, or invalid callback */
  AMX_ERR_NATIVE,       /* native function failed */

  AMX_ERR_MEMORY = 16,  /* out of memory */
  AMX_ERR_FORMAT,       /* invalid file format */
  AMX_ERR_VERSION,      /* file is for a newer version of the AMX */
  AMX_ERR_NOTFOUND,     /* function not found */
  AMX_ERR_INDEX,        /* invalid index parameter */
  AMX_ERR_MODULE,       /* unknown or unregistered module */
};

enum {
  DBG_FILE,             /* file number in p1, filename in name */
  DBG_LINE,             /* line number in p1, file number in p2 */
  DBG_SYMBOL,           /* address in p1, class/type in p2 */
  DBG_CLRSYM,           /* stack address below which locals should be removed */
  DBG_CALL,             /* function call */
  DBG_RETURN,           /* function returns */
  DBG_TERMINATE,        /* program ends */
};

#define AMX_EXEC_MAIN   -1      /* start at program entry point */
#define AMX_EXEC_CONT   -2      /* continue from last address */

int amx_Callback(AMX *amx, cell index, cell *result, cell *params);
int amx_CheckModules(AMX *amx);
int amx_Core(char *name, AMX_NATIVE *func);
int amx_Debug(AMX *amx, int code, cell p1, cell p2, char *name); /* ??? should be implemented by user program */
int amx_DebugBrowse(AMX *amx);
int amx_Exec(AMX *amx, cell *retval, int index, int numparams, ...);
int amx_GetAddr(AMX *amx,cell v,cell **addr);
int amx_GetModule(AMX *amx, int index, char *modulename);
int amx_GetPublic(AMX *amx, int index, char *funcname);
int amx_GetUserData(AMX *amx, int index, void **ptr);
int amx_Init(AMX *amx,void *program);
int amx_NameLength(AMX *amx, int *length);
int amx_NumModules(AMX *amx, int *number);
int amx_NumPublics(AMX *amx, int *number);
int amx_RaiseError(AMX *amx, int error);
int amx_Register(AMX *amx, char *modulename,
                 int (*callback)(char*,AMX_NATIVE*));
int amx_Release(AMX *amx);
int amx_SetCallback(AMX *amx, AMX_CALLBACK callback);
int amx_SetDebugProc(AMX *amx, AMX_DEBUG debug);
int amx_SetUserData(AMX *amx, int index, void *ptr);
int amx_Unregister(AMX *amx, char *modulename,
                   int (*callback)(char*,AMX_NATIVE*));
int amx_Version(AMX *amx, int *fileversion, int *requiredversion,
                int *curversion);

