/* Since some of these routines go farther than those of standard C,
 * they cannot always be implemented with portable C functions. In
 * other words, these routines must be ported to other environments.
 */
#include <conio.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "amx.h"

static int printstring(AMX *amx,cell *cstr,cell *params,int num);

static int dochar(AMX *amx,char ch,cell param)
{
  cell *cptr;

  switch (ch) {
  case '%':
    putchar(ch);
    return 0;
  case 'c':
    amx_GetAddr(amx,param,&cptr);
    putchar((int)*cptr);
    return 1;
  case 'd':
    amx_GetAddr(amx,param,&cptr);
    printf("%ld",(long)*cptr);
    return 1;
  case 's':
    amx_GetAddr(amx,param,&cptr);
    printstring(amx,cptr,NULL,0);
    return 1;
  } /* switch */
  /* error in the string format, try to repair */
  putchar(ch);
  return 0;
}

static int printstring(AMX *amx,cell *cstr,cell *params,int num)
{
  int len;
  int informat=0,paramidx=0;

  /* check the length of the string */
  for (len=0; cstr[len]!=0; len++)
    /* nothing */;
  /* check whether this is a packed string */
  if ((cstr[0] & 0xff00)!=0 && len>0) {
    int i,j;
    int c;
    /* the string is packed */
    for (i=0; i<len; i++) {
      for (j=0; j<sizeof(cell); j++) {
        c=(int)((cstr[i] >> 8*j) & 0xff);
        if (c==0)
          break;
        if (informat) {
          assert(params!=NULL);
          paramidx+=dochar(amx,c,params[paramidx]);
          informat=0;
        } else if (params!=NULL && c=='%') {
          informat=1;
        } else {
          putchar(c);
        } /* if */
      } /* for */
    } /* for */
  } else {
    int i;
    /* the string is unpacked */
    for (i=0; i<len; i++) {
      if (informat) {
        assert(params!=NULL);
        paramidx+=dochar(amx,(int)cstr[i],params[paramidx]);
        informat=0;
      } else if (params!=NULL && (int)cstr[i]=='%') {
        if (paramidx<num)
          informat=1;
        else
          amx_RaiseError(amx, AMX_ERR_NATIVE);
      } else {
        putchar((int)cstr[i]);
      } /* if */
    } /* for */
  } /* if */
  return paramidx;
}

static cell _print(AMX *amx,cell *params)
{
  cell *cstr;

  /* do the colour codes with ANSI strings */
  if (params[2]>=0)
    printf("\x1b[%dm",params[2]+30);
  if (params[3]>=0)
    printf("\x1b[%dm",params[3]+40);

  amx_GetAddr(amx,params[1],&cstr);
  printstring(amx,cstr,NULL,0);

  /* reset the colours */
  if (params[2]>=0 || params[3]>=0)
    printf("\x1b[37;40m");

  fflush(stdout);
  return 0;
}

static cell _printf(AMX *amx,cell *params)
{
  cell *cstr;

  amx_GetAddr(amx,params[1],&cstr);
  printstring(amx,cstr,params+2,(int)(params[0]/sizeof(cell))-1);
  fflush(stdout);
  return 0;
}

static void acceptchar(int c,int *num)
{
  switch (c) {
  case '\b':
    putchar('\b');
    *num-=1;
    #if defined __BORLANDC__
      /* the backspace key does not erase the
       * character, so do this explicitly */
      putchar(' ');     /* erase */
      putchar('\b');    /* go back */
    #endif
    break;
  case '\r':
    putchar('\n');
    *num+=1;
    break;
  default:
    putchar(c);
    *num+=1;
  } /* switch */
}

static int inlist(AMX *amx,int c,cell *params,int num)
{
  int i, key;

  for (i=0; i<num; i++) {
    if (i==0) {
      /* first key is passed by value, others are passed by reference */
      key = (int)params[i];
    } else {
      cell *cptr;
      amx_GetAddr(amx,params[i],&cptr);
      key=(int)*cptr;
    } /* if */
    if (c==key || c==-key)
      return key;
  } /* for */
  return 0;
}

#pragma argsused
static cell _getvalue(AMX *amx,cell *params)
{
  cell value;
  int base,sign,c,d;
  int chars,n;

  base=(int)params[1];
  if (base<2 || base>36)
    return 0;

  chars=0;
  value=0;

  c=getch();
  while (c!=EOF) {
    /* check for sign (if any) */
    if (chars==0) {
      if (c=='-') {
        sign=-1;
        acceptchar(c,&chars);
        c=getch();
      } else {
        sign=1;
      } /* if */
    } /* if */

    /* check end of input */
    if ((chars>1 || chars>0 && sign>0)
        && (n=inlist(amx,c,params+2,(int)params[0]/sizeof(cell)-1))!=0)
    {
      if (n>0)
        acceptchar(c,&chars);
      break;
    } /* if */

    /* get value */
    d=base;     /* by default, do not accept the character */
    if (c>='0' && c<='9') {
      d=c-'0';
    } else if (c>='a' && c<='z') {
      d=c-'a'+10;
    } else if (c>='A' && c<='Z') {
      d=c-'A'+10;
    } else if (c=='\b') {
      if (chars>0) {
        value/=base;
        acceptchar(c,&chars);
      } /* if */
    } /* if */
    if (d<base) {
      acceptchar(c,&chars);
      value=value*base + d;
    } /* if */
    c=getch();
  } /* while */
  return sign*value;
}

int amx_Console(char *name, AMX_NATIVE *func)
{
static struct {
  char *name;
  AMX_NATIVE func;
} funclist[] = {
  { "getvalue",  _getvalue },
  { "print",     _print },
  { "printf",    _printf },
  { NULL, NULL }        /* terminator */
};
  int i;

  assert(func!=NULL);
  *func=NULL;
  for (i=0; funclist[i].name!=NULL; i++) {
    if (strcmp(name,funclist[i].name)==0) {
      assert(*func==NULL);      /* otherwise, there is a double entry */
      *func=funclist[i].func;
      #if defined NDEBUG
	return AMX_ERR_NONE;    /* no need to look further (but do so any
				 * in debug mode */
      #endif
    } /* if */
  } /* for */
  #if !defined NDEBUG
    if (*func!=NULL)
      return AMX_ERR_NONE;      /* function found, return OK */
  #endif
  return AMX_ERR_NOTFOUND;       /* function not found */
}
