/*  Core module for the Small AMX
 *
 *  Copyright T. Riemersma, 1997, 1998
 */
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#include "amx.h"

#define TRUE    1
#define FALSE   0

#define CHARMASK        (0xff << 8*sizeof(char))
#define CELLCHARS       (sizeof(cell) / sizeof(char))


#if !defined NOPROPLIST
typedef unsigned char uchar;
typedef struct _property_list {
  cell id;
  char *name;
  cell value;
  struct _property_list *next;
} proplist;

static int is_init=FALSE;
static proplist proproot;

static proplist *list_additem(proplist *root)
{
  proplist *item;

  assert(root!=NULL);
  if ((item=(proplist *)malloc(sizeof(proplist)))==NULL)
    return NULL;
  item->name=NULL;
  item->id=0;
  item->value=0;
  item->next=root->next;
  root->next=item;
  return item;
}
static void list_delete(proplist *pred,proplist *item)
{
  assert(pred!=NULL);
  assert(item!=NULL);
  pred->next=item->next;
  assert(item->name!=NULL);
  free(item->name);
  free(item);
}
static void list_setitem(proplist *item,cell id,char *name,cell value)
{
  char *ptr;

  assert(item!=NULL);
  if ((ptr=(char *)malloc(strlen(name)+1))==NULL)
    return;
  if (item->name!=NULL)
    free(item->name);
  strcpy(ptr,name);
  item->name=ptr;
  item->id=id;
  item->value=value;
}
static proplist *list_finditem(proplist *root,cell id,char *name,
                               proplist **pred)
{
  proplist *item=root->next;
  proplist *prev=root;

  while (item!=NULL && (item->id!=id || stricmp(item->name,name)!=0)) {
    prev=item;
    item=item->next;
  } /* while */
  if (pred!=NULL)
    *pred=prev;
  return item;
}
#endif

#pragma argsused
static cell numargs(AMX *amx, cell *params)
{
  AMX_HEADER *hdr;
  uchar *data;
  cell bytes;

  hdr=(AMX_HEADER *)amx->base;
  data=amx->base+(int)hdr->dat;
  /* the number of bytes is on the stack, at "frm + 2*cell" */
  bytes= * (cell *)(data+(int)amx->frm+2*sizeof(cell));
  /* the number of arguments is the number of bytes divided
   * by the size of a cell */
  return bytes/sizeof(cell);
}

static cell getarg(AMX *amx, cell *params)
{
  AMX_HEADER *hdr;
  uchar *data;
  cell value;

  hdr=(AMX_HEADER *)amx->base;
  data=amx->base+(int)hdr->dat;
  /* get the base value */
  value= * (cell *)(data+(int)amx->frm+((int)params[1]+3)*sizeof(cell));
  /* adjust the address in "value" in case of an array access */
  value+=params[2]*sizeof(cell);
  /* get the value indirectly */
  value= * (cell *)(data+(int)value);
  return value;
}

static cell setarg(AMX *amx, cell *params)
{
  AMX_HEADER *hdr;
  uchar *data;
  cell value;

  hdr=(AMX_HEADER *)amx->base;
  data=amx->base+(int)hdr->dat;
  /* get the base value */
  value= * (cell *)(data+(int)amx->frm+((int)params[1]+3)*sizeof(cell));
  /* adjust the address in "value" in case of an array access */
  value+=params[2]*sizeof(cell);
  /* verify the address */
  if (value<0 || value>=amx->hea && value<amx->stk)
    return 0;
  /* set the value indirectly */
  * (cell *)(data+(int)value) = params[3];
  return 1;
}

#pragma argsused
static cell heapspace(AMX *amx,cell *params)
{
  return amx->stk - amx->hea;
}

static cell pack(AMX *amx,cell *params)
{
  cell c, *cptr;

  assert(sizeof(cell)==4 && sizeof(char)==1);
  c=params[2] & 0xff;
  c |= (params[3] & 0xff) << 8;
  c |= (params[4] & 0xff) << 16;
  c |= (params[5] & 0xff) << 24;

  amx_GetAddr(amx,params[1],&cptr);
  *cptr=c;
  return c;
}

static cell unpack(AMX *amx,cell *params)
{
  cell c, *cptr;
  int n = 0;

  assert(sizeof(cell)==4 && sizeof(char)==1);
  c=params[1];

  amx_GetAddr(amx,params[2],&cptr);
  *cptr=c & 0xff;
  n += *cptr!=0;

  amx_GetAddr(amx,params[3],&cptr);
  *cptr=(c>>8) & 0xff;
  n += *cptr!=0;

  amx_GetAddr(amx,params[4],&cptr);
  *cptr=(c>>16) & 0xff;
  n += *cptr!=0;

  amx_GetAddr(amx,params[5],&cptr);
  *cptr=(c>>24) & 0xff;
  n += *cptr!=0;

  return n;
}

static int amx_cstrlen(cell *cstr)
{
  int len;
  /* check the length of the string */
  for (len=0; cstr[len]!=0; len++)
    /* nothing */;
  return len;
}

int amx_StrIsPacked(cell *cstr)
{
  int len=amx_cstrlen(cstr);
  return (cstr[0] & CHARMASK)!=0 && len>0;
}

int amx_StrLen(cell *cstr)
{
  int len=amx_cstrlen(cstr);
  if ((cstr[0] & CHARMASK)!=0 && len>0) {
    /* check the number of packed characters in the last cell */
    cell last=cstr[len-1];
    int i=0;
    while (last!=0) {
      i++;
      last>>=CHAR_BIT;
    } /* while */
    len=(len-1)*CELLCHARS + i;
  } /* if */
  return len;
}

int amx_StrPack(cell *dest,cell *source)
{
  if (amx_StrIsPacked(source)) {
    /* source string is already packed */
    while ((*dest++ = *source++) != 0)
      /* nothing */;
  } else {
    /* pack string, from bottom up */
    cell c;
    int i,len;
    len=amx_StrLen(source);
    for (c=0,i=0; i<len; i++) {
      assert((*source & ~0xffL)==0);
      c=(c>>CHAR_BIT) | (*source++ << ((sizeof(cell)-1)*CHAR_BIT));
      if (i%sizeof(cell) == sizeof(cell)-1) {
	*dest++=c;
	c=0;
      } /* if */
    } /* for */
    if (i%sizeof(cell) != 0)	/* store remaining packed characters */
      *dest++=c >> (sizeof(cell)-i%sizeof(cell))*CHAR_BIT;
    *dest++=0;          /* store terminator */
  } /* if */
  return AMX_ERR_NONE;
}

int amx_StrUnpack(cell *dest,cell *source)
{
  if (amx_StrIsPacked(source)) {
    /* unpack string, from top down (so srtring can be unpacked in place) */
    cell c;
    int i,len;
    len=amx_StrLen(source);
    dest[len]=0;
    for (i=len-1; i>=0; i--) {
      c=(source[i/sizeof(cell)] >> (i%sizeof(cell))*CHAR_BIT) & UCHAR_MAX;
      dest[i]=c;
    } /* for */
  } else {
    /* source string is already unpacked */
    while ((*dest++ = *source++) != 0)
      /* nothing */;
  } /* if */
  return AMX_ERR_NONE;
}

static cell strpacked(AMX *amx,cell *params)
{
  cell *cptr;

  amx_GetAddr(amx,params[1],&cptr);
  return amx_StrIsPacked(cptr);
}

static cell core_strlen(AMX *amx,cell *params)
{
  cell *cptr;

  amx_GetAddr(amx,params[1],&cptr);
  return amx_StrLen(cptr);
}

static cell strpack(AMX *amx,cell *params)
{
  cell *cdest,*csrc;
  int len,needed,err;

  /* calculate number of cells needed for (packed) destination */
  amx_GetAddr(amx,params[2],&csrc);
  len=amx_StrLen(csrc);
  needed=(len+sizeof(cell)-1)/sizeof(cell) + 1; /* # of cells needed */
  err=amx_GetAddr(amx,params[1]+needed,&cdest);
  if (err!=AMX_ERR_NONE)
    return amx_RaiseError(amx,err);

  amx_GetAddr(amx,params[2],&cdest);
  err=amx_StrPack(cdest,csrc);
  if (err!=AMX_ERR_NONE)
    return amx_RaiseError(amx,err);

  return len;
}

static cell strunpack(AMX *amx,cell *params)
{
  cell *cdest,*csrc;
  int len,needed,err;

  /* calculate number of cells needed for (packed) destination */
  amx_GetAddr(amx,params[2],&csrc);
  len=amx_StrLen(csrc);
  needed=len+1;         /* # of cells needed */
  err=amx_GetAddr(amx,params[1]+needed,&cdest);
  if (err!=AMX_ERR_NONE)
    return amx_RaiseError(amx,err);

  amx_GetAddr(amx,params[2],&cdest);
  err=amx_StrUnpack(cdest,csrc);
  if (err!=AMX_ERR_NONE)
    return amx_RaiseError(amx,err);

  return len;
}

#if !defined NOPROPLIST
static char *MakePackedString(cell *cptr)
{
  int len=amx_StrLen(cptr);
  cell *dest=malloc(len+sizeof(cell));
  amx_StrPack(dest,cptr);
  return (char *)dest;
}

static cell getproperty(AMX *amx,cell *params)
{
  cell *cptr;
  char *name;
  proplist *item;

  amx_GetAddr(amx,params[2],&cptr);
  name=MakePackedString(cptr);
  item=list_finditem(&proproot,params[1],name,NULL);
  free(name);
  return (item!=NULL) ? item->value : 0;
}

static cell setproperty(AMX *amx,cell *params)
{
  cell prev=0;
  cell *cptr;
  char *name;
  proplist *item;

  amx_GetAddr(amx,params[2],&cptr);
  name=MakePackedString(cptr);
  item=list_finditem(&proproot,params[1],name,NULL);
  if (item==NULL)
    item=list_additem(&proproot);
  if (item==NULL) {
    amx_RaiseError(amx,AMX_ERR_MEMORY);
  } else {
    prev=item->value;
    list_setitem(item,params[1],name,params[3]);
  } /* if */
  free(name);
  return prev;
}

static cell delproperty(AMX *amx,cell *params)
{
  cell prev=0;
  cell *cptr;
  char *name;
  proplist *item,*pred;

  amx_GetAddr(amx,params[2],&cptr);
  name=MakePackedString(cptr);
  item=list_finditem(&proproot,params[1],name,&pred);
  if (item!=NULL) {
    prev=item->value;
    list_delete(pred,item);
  } /* if */
  free(name);
  return prev;
}

static cell existproperty(AMX *amx,cell *params)
{
  cell *cptr;
  char *name;
  proplist *item;

  amx_GetAddr(amx,params[2],&cptr);
  name=MakePackedString(cptr);
  item=list_finditem(&proproot,params[1],name,NULL);
  free(name);
  return (item!=NULL);
}
#endif

static cell core_init(void)
{
  is_init=TRUE;
  #if !defined NOPROPLIST
    proproot.next=NULL;
  #endif
  return 0;
}

#pragma argsused
static cell core_cleanup(AMX *amx,cell *params)
{
  assert(params!=NULL && params[0]==0);
  #if !defined NOPROPLIST
    if (is_init)
      while (proproot.next!=NULL)
        list_delete(&proproot,proproot.next);
  #endif
  is_init=FALSE;
  return 0;
}

int amx_Core(char *name, AMX_NATIVE *func)
{
static struct {
  char *name;
  AMX_NATIVE func;
} funclist[] = {
  { "numargs",       numargs },
  { "getarg",        getarg },
  { "setarg",        setarg },
  { "heapspace",     heapspace },
  { "pack",          pack },
  { "unpack",        unpack },
  { "strlen",        core_strlen },
  { "strpacked",     strpacked },
  { "strpack",       strpack },
  { "strunpack",     strunpack },
#if !defined NOPROPLIST
  { "getproperty",   getproperty },
  { "setproperty",   setproperty },
  { "deleteproperty",delproperty },
  { "existproperty", existproperty },
#endif
  { "",          core_cleanup },
  { NULL, NULL }        /* terminator */
};
  int i;

  assert(func!=NULL);
  *func=NULL;
  for (i=0; funclist[i].name!=NULL; i++) {
    if (strcmp(name,funclist[i].name)==0) {
      assert(*func==NULL);      /* otherwise, there is a double entry */
      if (!is_init)
        core_init();
      *func=funclist[i].func;
      #if defined NDEBUG
        return AMX_ERR_NONE;    /* no need to look further (but do so anyway
                                 * in debug mode) */
      #endif
    } /* if */
  } /* for */
  #if !defined NDEBUG
    if (*func!=NULL)
      return AMX_ERR_NONE;      /* function found, return OK */
  #endif
  return AMX_ERR_NOTFOUND;      /* function not found */
}

