/*  Small compiler
 *
 *  Function and variable definition and declaration, statement parser.
 */
#include <assert.h>
#include <ctype.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include "sc.h"

static void setopt(int argc,char **argv,char *iname,char *oname,char *bname,
                   char *ename,int *listing,int *verbose,int *debug);
static void setconfig(char *root);
static void about(void);
static void setconstants(void);
static void parse(void);
static void dumplits(void);
static void dumpzero(int count);
static void declglb(void);
static int declloc(void);
static void decl_const(int table);
static void decl_enum(int table);
static cell needsub(int *tag);
static void initials(int ident,int tag,cell *dim);
static cell init(int ident,int *tag);
static symbol *fetchfunc(char *name,int tag);
static void funcstub(void);
static void newfunc(int exported);
static int declargs(symbol *sym);
static void doarg(char *name,int ident,int offset,int tag,arginfo *arg);
static int testsymbols(symbol *root,int level,int testlabs);
static constval *add_constval(constval *table,char *name,cell val);
static void delete_consttable(constval *table);
static void statement(void);
static void compound(void);
static void doexpr(int comma,int *tag);
static void doassert(void);
static void doexit(void);
static void test(int label,int parens,int invert);
static void doif(void);
static void dowhile(void);
static void dodo(void);
static void dofor(void);
static void doswitch(void);
static void dogoto(void);
static void dolabel(void);
static symbol *fetchlab(char *name);
static void doreturn(void);
static void dobreak(void);
static void docont(void);

/*  main
 *
 *  Global references: listing,verbose,optimize (altered)
 *                     stksize,freading (altered)
 *                     inpfname         (altered)
 *                     outfname         (altered)
 */
int main(int argc, char **argv)
{
  int entry;

  litmax=_def_litmax;

  /* allocate all memory */
  inpfname=malloc(_linemax+1);
  lbltab=malloc(_numlabels*sizeof(cell));
  litq=malloc(litmax*sizeof(cell));
  if (lbltab==NULL || litq==NULL)
    error(103);         /* insufficient memory */

  wqptr=wq;             /* initialize while queue pointer */
  glbtab.next=NULL;     /* clear global variables/constants table */
  loctab.next=NULL;     /*   "   local      "    /    "       "   */
  mod_tab.next=NULL;    /* clear the module table */

  setopt(argc,argv,inpfname,outfname,binfname,errfname,
         &listing,&verbose,&debug);
  if (strlen(errfname)==0)
    puts("Small compiler\t\tCopyright (c) 1997-1998, ITB CompuPhase\n");
  setconfig(argv[0]);
  inpf=fopen(inpfname,"rt");
  if (inpf!=NULL)
    freading=1;
  else
    error(100,inpfname);
  outf=fopen(outfname,"wt");    /* first write to assembler file (may be temporary) */
  if (outf==NULL)
    error(101,outfname);
  setconstants();       /* set a few predefined constants and tagnames */
  setfile(inpfname,fnumber);
  preprocess();         /* fetch first line */
  parse();              /* process all input */
  entry=testsymbols(&glbtab,0,_yes);    /* test for unused or undefined
                                         * functions and variables */
  if (!entry)
    error(13);          /* no entry point (no exported functions) */
  /* inpf is already closed when readline() attempts to pop of a file */
  writetrailer();       /* write remaining stuff */
  fclose(outf);
  /* create the binary file */
  if (!listing && errnum==0){
    inpf=fopen(outfname,"rt");
    outf=fopen(binfname,"wb");
    assemble(outf,inpf);
    fclose(inpf);
    fclose(outf);
  } /* if */

  assert(loctab.next==NULL);    /* local symbols should already have been deleted */
  free(inpfname);
  free(lbltab);
  free(litq);
  delete_symbols(&glbtab,0,_yes);
  delete_consttable(&mod_tab);
  delete_consttable(&tagname_tab);
  if (!listing)
    unlink(outfname);

  if (errnum!=0){
    if (strlen(errfname)==0)
      printf("\n%d Error%s.\n",errnum,(errnum>1) ? "s" : "");
    unlink(binfname);
    return 2;
  } else if (warnnum!=0){
    if (strlen(errfname)==0)
      printf("\n%d Warning%s.\n",warnnum,(warnnum>1) ? "s" : "");
    return 1;
  } else {
    if (strlen(errfname)==0)
      puts("Done.");
    return 0;
  } /* if */
}

static void setopt(int argc,char **argv,char *iname,char *oname,char *bname,
                   char *ename,int *listing,int *verbose,int *debug)
{
  char str[_linemax],*ptr;
  int i;

  *iname='\0';
  *bname='\0';
  *ename='\0';
  for (i=1; i<argc; i++){
    if (argv[i][0]=='/' || argv[i][0]=='-') {
      ptr=&argv[i][1];
      switch (tolower(*ptr)) {
      case 'a':
        *listing=_yes;        /* skip last pass of making binary file */
        break;
      case 'd':               /* no assertions and bounds checking */
        *debug &= ~_chkbounds;
        break;
      case 'e':
        strcpy(ename,ptr+1);    /* set name of error file */
        break;
      case 'o':
        strcpy(bname,ptr+1);    /* set name of binary output file */
        break;
      case 's':               /* no symbolic information */
        *debug &= ~_symbolic;
        break;
      case 'v':
        *verbose=_yes;        /* output function names to the screen */
        break;
      default:                /* wrong option */
        about();
      } /* switch */
    } else if (strlen(iname)>0) {
      about();
    } else {
      strcpy(str,argv[i]);
      strlwr(str);
      strcpy(iname,str);
      if ((ptr=strchr(str,'.'))==NULL)
        strcat(iname,".sma");
      else
        *ptr=0;   /* set zero terminator at the position of the period */
      /* The output name is the input name with the extension .ASM. The
       * binary file has the extension .AMX. */
      strcpy(oname,str);
      strcat(oname,".asm");
      if (strlen(bname)==0) {
        strcpy(bname,str);
        strcat(bname,".amx");
      } /* if */
    } /* if */
  } /* for */
  if (strlen(iname)==0)
    about();
}

void setconfig(char *root)
{
  char *ptr;

  strcpy(includepath,root);     /* filename with options */
  if ((ptr=strchr(includepath,' '))!=NULL)
    *ptr='\0';
  if ((ptr=strchr(includepath,'\t'))!=NULL)
    *ptr='\0';
  if ((ptr=strchr(includepath,'/'))!=NULL)
    *ptr='\0';
  /* terminate just behind last \ or : */
  if ((ptr=strrchr(includepath,'\\'))!=NULL || (ptr=strchr(includepath,':'))!=NULL)
    *(ptr+1)='\0';
  strcat(includepath,"include\\");
}

static void about(void)
{
  if (strlen(errfname)==0) {
    puts("Usage:   sc <filename> [options]\n");
    puts("Options:");
    puts("         /a       output assembler code");
    puts("         /d       silence assertions; no array bounds checking");
    puts("         /e<name> set name of error file");
    puts("         /o<name> set name of binary output file");
    puts("         /s       strip symbolic information");
    puts("         /v       verbose, detailed messages");
  } /* if */
  exit(0);
}

static void setconstants(void)
{
  add_constval(&tagname_tab,"",0);              /* "untagged" */
  add_constval(&tagname_tab,"bool",1);

  add_constant("true",1,_global,1);     /* boolean flags */
  add_constant("false",0,_global,1);
  #if defined(BIT16)
    add_constant("maxval",INT_MAX,_global,0);
    add_constant("minval",INT_MIN,_global,0);
  #else
    add_constant("maxval",LONG_MAX,_global,0);
    add_constant("minval",LONG_MIN,_global,0);
  #endif
}

/*  parse       - process all input text
 *
 *  At this level, only static declarations and function definitions are legal.
 */
static void parse(void)
{
  int tok;
  cell val;
  char *str;

  while (freading){
    /* first try wether a declaration possibly is external or exported */
    tok=lex(&val,&str);  /* read in (new) token */
    switch (tok) {
    case 0:
      /* ignore zero's */
      break;
    case __new:
      declglb();
      break;
    case __const:
      decl_const(_global);
      break;
    case __enum:
      decl_enum(_global);
      break;
    case __public:
      newfunc(_yes);
      break;
    case __label:
    case __symbol:
      lexpush();
      newfunc(_no);
      break;
    case __native:
      funcstub();               /* create a dummy function */
      break;
    default:
      if (freading)
        error(10);              /* illegal function or declaration */
    } /* switch */
  } /* while */
}

/*  dumplits
 *
 *  Dump the literal pool (strings etc.)
 *
 *  Global references: litidx (referred to only)
 */
static void dumplits(void)
{
  int j,k;

  k=0;
  while (k<litidx){
    /* should be in the data segment */
    assert(curseg==2);
    defstorage();
    j=16;       /* 16 values per line */
    while (j && k<litidx){
      outval(litq[k], _no);
      stgwrite(" ");
      k++;
      j--;
      if (j==0 || k>=litidx)
        stgwrite("\n");         /* force a newline after 10 dumps */
      /* Note: stgwrite() buffers a line until it is complete. It recognizes
       * the end of line as a sequence of "\n\0", so something like "\n\t"
       * so  should not be passed to stgwrite().
       */
    } /* while */
  } /* while */
}

/*  dumpzero
 *
 *  Dump zero's for default initial values
 */
static void dumpzero(int count)
{
  int i;

  if (count<=0)
    return;
  assert(curseg==2);
  defstorage();
  i=0;
  while (count-- > 0) {
    outval(0, _no);
    i=(i+1) % 16;
    stgwrite((i==0 || count==0) ? "\n" : " ");
  } /* while */
}

int gettag(char *name)
{
  cell val;
  constval *ptr;
  int next;

  if (name==NULL) {
    /* no tagname was given, check for one */
    if (lex(&val,&name)!=__label) {
      lexpush();
      return 0;         /* untagged */
    } /* if */
  } /* if */

  next=0;
  ptr=tagname_tab.next;
  while (ptr!=NULL) {
    assert(next==(int)ptr->value);
    if (strcmp(name,ptr->name)==0)
      return next;      /* tagname is known, return its sequence number */
    ptr=ptr->next;
    next++;
  } /* while */

  /* tagname currently unknown, add it */
  add_constval(&tagname_tab,name,next);
  return next;
}

/*  declglb     - declare global symbols
 *
 *  Declare a static (global) variable. Global variables are stored in
 *  the DATA segment.
 *
 *  global references: glb_declared     (altered)
 */
static void declglb(void)
{
  int ident,tag,idxtag;
  char name[_namemax+1];
  cell val,size;
  char *str;
  symbol *sym;

  do {
    size=1;             /* single size (no array) */
    ident=_variable;
    tag=gettag(NULL);
    if (lex(&val,&str)!=__symbol)       /* read in (new) token */
      error(20,str);                    /* invalid symbol name */
    strcpy(name,str);                   /* save symbol name */
    if (findglb(name) || findconst(name))
      error(21,name);                   /* symbol already defined */
    if (matchtoken('[')){
      size=needsub(&idxtag);    /* get size; size==0 for "var[]" */
      ident=_array;     /* for global variables "var[]" is accepted... */
    } /* endif */       /* ...if the array is initialized. */
    defsymbol(name,ident,_global,sizeof(cell)*glb_declared);
    begdseg();          /* real (initialized) data in data segment */
    litidx=0;           /* global initial data is dumped, so restart at zero */
    initials(ident,tag,&size);  /* stores values in the literal queue */
    assert(size>=litidx);
    dumplits();         /* dump the literal queue */
    dumpzero((int)size-litidx);
    sym=addsym(name,sizeof(cell)*glb_declared,ident,_global,tag,_define);
    if (ident==_array) {
      sym->dim.array=size;
      sym->x.idxtag=idxtag;
    } /* if */
    glb_declared+=(int)size;    /* add total number of cells */
  } while (matchtoken(',')); /* enddo */   /* more? */
  needtoken(';');  /* if not comma, must be semicolumn */
}

/*  declloc     - declare local symbols
 *
 *  Declare local (automatic) variables. Since these variables are relative
 *  to the STACK, there is no switch to the DATA segment. These variables
 *  cannot be initialized either.
 *
 *  global references: declared   (altered)
 *                     funcstatus (referred to only)
 */
static int declloc(void)
{
  int ident,tag,idxtag;
  char name[_namemax+1];
  symbol *sym;
  cell val,size;
  char *str;
  value lval;
  int cur_lit;

  do {
    ident=_variable;
    size=1;
    tag=gettag(NULL);
    if (lex(&val,&str)!=__symbol)       /* read in (new) token */
      error(20,str);                    /* invalid symbol name */
    strcpy(name,str);                   /* save symbol name */
    /* Note: block locals may be named identical to locals at higher
     * compound blocks (as with standard C); so we must check (and add)
     * the "nesting level" of local variables to verify the
     * multi-definition of symbols
     */
    if ((sym=findloc(name))!=NULL && sym->compound==ncmp)
      error(21,name);                   /* symbol already defined */
    if (matchtoken('[')){
      ident=_array;
      size=needsub(&idxtag);    /* get size; size==0 for "var[]" */
      cur_lit=litidx;           /* save current index in the literal table */
      initials(ident,tag,&size);
    } /* if */
    /* reserve memory (on the stack) for the variable */
    declared+=(int)size;    /* variables are put on stack, adjust "declared" */
    sym=addsym(name,-declared*sizeof(cell),ident,_local,tag,_define);
    defsymbol(name,ident,_local,-declared*sizeof(cell));
    modstk(-(int)size*sizeof(cell));
    /* now that we have reserved memory for the variable, we can proceed
     * to initialize it */
    assert(sym!=NULL);          /* we declared it, it must be there */
    sym->compound=ncmp;         /* for multiple declaration check */
    if (ident==_array) {
      sym->dim.array=size;
      sym->x.idxtag=idxtag;
    } /* if */
    if (ident==_variable) {
      /* simple variable, also supports initialization */
      int ctag = tag;   /* set to "tag" by default */
      if (matchtoken('='))
        doexpr(_no,&ctag);
      else
        const1(0);      /* uninitialized variable, set to zero */
      /* now try to save the value (still in PRI) in the variable */
      lval.sym=sym;
      lval.ident=_variable;
      lval.constval=0;
      store(&lval);
      if (tag!=0 && tag!=ctag)
        error(213);   /* tagname mismatch */
    } else {
      /* an array */
      assert(cur_lit>=0 && cur_lit<=litidx && litidx<=litmax);
      /* if the array is not completely filled, set all values to zero first */
      assert(sym->dim.array > 0);
      if (litidx-cur_lit < sym->dim.array)
        fillarray(sym,sym->dim.array*sizeof(cell),0);
      if (cur_lit<litidx) {
        /* check whether the complete array is set to a single value; if
         * it is, more compact code can be generated */
        cell first=litq[cur_lit];
        int i;
        for (i=cur_lit; i<litidx && litq[i]==first; i++)
          /* nothing */;
        if (i==litidx) {
          /* all values are the same */
          fillarray(sym,(litidx-cur_lit)*sizeof(cell),first);
          litidx=cur_lit;   /* reset literal table */
        } else {
          /* copy the literals to the array */
          const1((cur_lit+glb_declared)*sizeof(cell));
          copyarray(sym,(litidx-cur_lit)*sizeof(cell));
        } /* if */
      } /* if */
    } /* if */
  } while (matchtoken(',')); /* enddo */   /* more? */
  needtoken(';');     /* if not comma, must be semicolumn */
  return ident;
}

/*  initials
 *
 *  Initialize global objects and local arrays.
 *    dim==number of objects (count)
 *    tag==required tagname id (not the returned tag)
 *
 *  Global references: litidx (altered)
 */
static void initials(int ident,int tag,cell *dim)
{
  cell prev1=0,prev2=0;
  int ctag;
  int ellips=_no;
  int curlit=litidx;

  if (matchtoken('=')) {
    if (matchtoken('{')) {
      do {
        if ((ellips=matchtoken(__ellips))!=0)
          break;
        prev2=prev1;
        prev1=init(ident,&ctag);
        if (tag!=0 && tag!=ctag)
          error(213);           /* tagname mismatch */
      } while (matchtoken(',')); /* do */
      needtoken('}');
    } else {
      init(ident,&ctag);
      if (tag!=0 && tag!=ctag)
        error(213);     /* tagname mismatch */
    } /* if */
  } else if (*dim==0){  /* declared as "myvar[];" which is senseless */
    error(9);           /* array has zero length -> invalid size */
  } /* if */
  /* fill up the literal queue with a series */
  if (ellips) {
    cell step=((litidx-curlit)==1) ? (cell)0 : prev1-prev2;
    if (*dim==0 || (litidx-curlit)==0)
      error(41);        /* invalid ellipsis, array size unknown */
    while ((litidx-curlit)<(int)*dim) {
      prev1+=step;
      stowlit(prev1);
    } /* while */
  } /* if */
  if (*dim==0)
    *dim=litidx-curlit;                 /* number of elements defined */
  else if (litidx-curlit>(int)*dim)     /* e.g. "myvar[3]={1,2,3,4};" */
    error(18);          /* initialisation data exceeds declared size */
}

/*  init
 *
 *  Evaluate one initializer.
 */
static cell init(int ident,int *tag)
{
  cell i = 0;

  if (matchtoken(__string)){
    /* lex() automatically stores strings in the literal table (and
     * increases "litidx") */
    if (ident==_variable)
      error(6);         /* must be assigned to an array */
    *tag=0;
  } else if (constexpr(&i,tag)){
    stowlit(i);         /* store expression result in literal table */
  } /* if */
  return i;
}

/*  needsub
 *
 *  Get required array size
 */
static cell needsub(int *tag)
{
  cell val;

  *tag=0;
  if (matchtoken(']'))  /* we've already seen "[" */
    return 0;           /* null size (like "char msg[]") */
  constexpr(&val,tag);  /* get value (must be constant expression) */
  if (val<0) {
    error(9);           /* negative array size is invalid; assumed zero */
    val=0;
  } /* if */
  needtoken(']');
  return val;           /* return array size */
}

/*  decl_const  - declare a single constant
 *
 */
static void decl_const(int vclass)
{
  char constname[_namemax+1];
  cell val;
  char *str;
  int tag;

  tag=gettag(NULL);
  if (lex(&val,&str)!=__symbol)         /* read in (new) token */
    error(20,str);                      /* invalid symbol name */
  strcpy(constname,str);                /* save symbol name */
  needtoken('=');
  constexpr(&val,NULL);                 /* get value */
  needtoken(';');
  /* add_constant() checks for duplicate definitions */
  add_constant(constname,val,vclass,tag);
}

/*  decl_enum   - declare enumerated constants
 *
 */
static void decl_enum(int vclass)
{
  char enumname[_namemax+1],constname[_namemax+1];
  cell val,value,size;
  char *str;
  int tag;

  if (lex(&val,&str)==__symbol) {       /* read in (new) token */
    strcpy(enumname,str);               /* save enum name (last constant) */
    tag=gettag(enumname);
  } else {
    lexpush();                          /* analyze again */
    enumname[0]='\0';
    tag=0;
  } /* if */
  needtoken('{');
  /* go through all constants */
  value=0;                              /* default starting value */
  do {
    if (matchtoken('}')) {              /* quick exit if '}' follows ',' */
      lexpush();
      break;
    } /* if */
    if (lex(&val,&str)!=__symbol)       /* read in (new) token */
      error(20,str);                    /* invalid symbol name */
    strcpy(constname,str);              /* save symbol name */
    size=1;                             /* default increment of 'val' */
    if (matchtoken('='))
      constexpr(&value,NULL);           /* get value */
    else if (matchtoken(':'))
      constexpr(&size,NULL);            /* get size */
    /* add_constant() checks whether a variable (global or local) or
     * a constant with the same name already exists */
    add_constant(constname,value,vclass,tag);
    value+=size;
  } while (matchtoken(','));
  needtoken('}');       /* terminates the constant list */
  matchtoken(';');      /* eat an optional ; */
  /* set the enum name to the last value plus one */
  if (strlen(enumname)>0)
    add_constant(enumname,value,vclass,tag);
}

/*
 *  Finds a function in the global symbol table or creates a new entry.
 *  It does some basic processing and error checking.
 */
static symbol *fetchfunc(char *name,int tag)
{
  symbol *sym;
  cell offset;

  offset=code_idx;
  if ((debug & _symbolic)!=0) {
    offset+=4+sizeof(cell)+strlen(name);
    /* ^^^ The address for the symbol is the code address. But the "symbol"
     *     instruction itself generates code. Therefore the offset is
     *     pre-adjusted to the value it will have after the symbol instruction.
     */
  } /* if */
  if ((sym=findglb(name))!=0) {         /* already in symbol table? */
    if (sym->ident!=_functn)
      error(21,name);                       /* yes, but not as function */
    else if ((sym->usage & _define)!=0)
      error(21,name);                       /* yes, and it's already defined */
    else if ((sym->usage & _external)!=0)
      error(21,name);                       /* yes, and it is an external */
    assert(sym->vclass==_global);
    if ((sym->usage & _define)==0) {
      /* as long as the function stays undefined, update the address */
      sym->addr=offset;
    } /* if */
  } else {
    /* don't set the "_define" flag; it may be a prototype */
    sym=addsym(name,offset,_functn,_global,tag,0);
    /* assume no arguments */
    sym->dim.arglist=malloc(1*sizeof(arginfo));
    sym->dim.arglist[0].ident=0;
  } /* if */
  return sym;
}

/* This routine adds symbolic information for each argument.
 */
static void define_args(void)
{
  symbol *sym;

  /* At this point, no local variables have been declared. All
   * local symbols are function arguments.
   */
  sym=loctab.next;
  while (sym!=NULL) {
    assert(sym->ident!=_label);
    assert(sym->vclass==_local);
    defsymbol(sym->name,sym->ident,_local,sym->addr);
    sym=sym->next;
  } /* while */
}

static void funcstub(void)
{
  int tok,mod,tag;
  char *str;
  cell val;
  char symbolname[_namemax+1],modname[_namemax+1];
  symbol *sym;
  constval *modptr;

  lastst=0;             /* no statement yet */
  litidx=0;             /* clear the literal pool */
  assert(loctab.next==NULL);    /* local symbol table should be empty */

  tok=lex(&val,&str);           /* read in (new) token */
  /* the module name may be given */
  if (tok==__string) {
    int i;
    /* the string may either be in packed or in unpacked format */
    if (litidx>1 && (litq[0] & 0xff00)==0) {
      /* unpacked format */
      for (i=0; i<litidx && i<_namemax; i++)
        modname[i]=(char)litq[i];
    } else {
      /* packed format */
      for (i=0; i<_namemax; i++) {
        modname[i]=(char)(litq[i/sizeof(cell)] >> 8*(i%sizeof(cell)));
        if (modname[i]=='\0')
          break;
      } /* for */
      modname[i]='\0';  /* make sure to terminate string */
    } /* if */
    tok=lex(&val,&str); /* read in (new) token */
  } else {
    strcpy(modname,"core");
  } /* if */
  /* look up the module */
  for (mod=0,modptr=mod_tab.next; modptr!=NULL && strcmp(modptr->name,modname)!=0; mod++,modptr=modptr->next)
    /* nothing */;
  if (modptr==NULL)
    add_constval(&mod_tab,modname,0);   /* module does not exist, create it */

  if (tok==__label) {
    tag=gettag(str);
    tok=lex(&val,&str); /* read in (new) token */
  } else {
    tag=0;
  } /* if */

  if (tok!=__symbol)
    error(10);                  /* illegal function or declaration */
  strcpy(symbolname,str);
  needtoken('(');               /* only functions may be external */

  sym=fetchfunc(symbolname,tag);/* get a pointer to the function entry */
  sym->usage|=_external | _define | _retvalue;
  sym->x.mod_id=mod;            /* save module index */

  declargs(sym);
  /* "declargs()" found the ")". An external declaration must be a prototype,
   * so the next token must be a semicolon */
  needtoken(';');
  litidx=0;                     /* clear the literal pool */
  delete_symbols(&loctab,0,_yes);/* clear local variables queue */
}

/*  newfunc    - begin a function
 *
 *  This routine is called from "parse" and tries to make a function
 *  out of the following text
 *
 *  Global references: funcstatus,lastst,litidx
 *                     rettype  (altered)
 *                     declared (altered)
 *                     glb_declared (altered)
 */
static void newfunc(int exported)
{
  symbol *sym;
  int argcnt,tok,tag;
  char symbolname[_namemax+1];
  char *str;
  cell val;

  lastst=0;             /* no statement yet */
  litidx=0;             /* clear the literal pool */
  assert(loctab.next==NULL);    /* local symbol table should be empty */

  tag=gettag(NULL);
  tok=lex(&val,&str);
  if (tok!=__symbol && freading)
    error(10);          /* illegal function or declaration */
  strcpy(symbolname,str);
  /* check whether this is a function or a variable declaration */
  if (!matchtoken('(')) {
    error(10);          /* illegal function or declaration */
    return;
  } /* if */
  sym=fetchfunc(symbolname,tag);    /* get a pointer to the function entry */
  if (exported)
    sym->usage|=_exported;
  /* so it is a function, proceed */
  argcnt=declargs(sym);
  if (strcmp(symbolname,"main")==0) {
    if (argcnt>0)
      error(5);         /* "main()" function may not have any arguments */
    sym->usage|=_refer; /* "main()" is the program's entry point: always used */
  } /* if */
  /* "declargs()" found the ")". If after this a ";" appears, it was a
   * prototype or a declaration of an external function */
  if (matchtoken(';')) {
    delete_symbols(&loctab,0,_yes);  /* prototype is done; forget everything */
    return;
  } /* if */
  /* so it is not a prototype, proceed */
  begcseg();
  sym->usage|=_define;  /* set the definition flag */
  if (exported) {
    sym->usage|=_refer; /* exported functions are always used */
    export_sym(symbolname);
  } /* if */
  defsymbol(symbolname,_functn,_global,
            code_idx+4+sizeof(cell)+strlen(symbolname));
         /* ^^^ The address for the symbol is the code address. But the
          * "symbol" instruction itself generates code. Therefore the
          * offset is pre-adjusted to the value it will have after the
          * symbol instruction.
          */
  startfunc();          /* creates stack frame */
  declared=0;           /* number of local cells */
  rettype=(sym->usage & _retvalue);      /* set "return type" variable */
  define_args();        /* add the symbolic info for the function arguments */
  statement();
  if ((rettype & _retvalue)!=0)
    sym->usage|=_retvalue;
  if ((lastst!=__return) && (lastst!=__goto)){
    ffret();
    if ((sym->usage & _retvalue)!=0)
      error(209);               /* function should return a value */
  } /* if */
  endfunc();
  if (litidx) {                 /* if there are literals defined */
    glb_declared+=litidx;
    begdseg();                  /* flip to DATA segment */
    dumplits();                 /* dump literal strings */
  } /* if */
  testsymbols(&loctab,0,_yes);  /* test for unused arguments and labels */
  delete_symbols(&loctab,0,_yes);/* clear local variables queue */
  assert(loctab.next==NULL);
}

/*  declargs()
 *
 *  This routine adds an entry in the local symbol table for each argument
 *  found in the argument list. It returns the number of arguments.
 */
static int declargs(symbol *sym)
{
  char *st;
  int argcnt,tok,tag;
  cell val;
  arginfo arg;
  char name[_namemax+1];
  int ident;

  /* the '(' parantheses has already been parsed */
  argcnt=0;                    /* zero aruments up to now */
  ident=_variable;
  tag=0;
  if (!matchtoken(')')){
    do {                       /* there are arguments; process them */
      /* any legal name increases argument count (and stack offset) */
      tok=lex(&val,&st);
      switch (tok) {
        case 0:
          /* nothing */
          break;
        case '&':
          if (ident!=_variable)
            error(1,"-identifier-","&");
          ident=_reference;
          break;
        case __label:
          tag=gettag(st);
          break;
        case __symbol:
          strcpy(name,st);    /* save symbol name */
          /* Stack layout:
           *   base + 0*sizeof(cell)  == previous "base"
           *   base + 1*sizeof(cell)  == function return address
           *   base + 2*sizeof(cell)  == number of arguments
           *   base + 3*sizeof(cell)  == first argument of the function
           * So the offset of each argument is "(argcnt+3) * sizeof(cell)".
           */
          doarg(name,ident,(argcnt+3)*sizeof(cell),tag,&arg);
          if ((sym->usage & _prototyped)==0) {
            /* redimension the argument list, add the entry _varargs */
            sym->dim.arglist=realloc(sym->dim.arglist,(argcnt+2)*sizeof(arginfo));
            sym->dim.arglist[argcnt]=arg;
            sym->dim.arglist[argcnt+1].ident=0; /* keep the list terminated */
          } else {
            /* check the argument with the earlier definition */
            if (memcmp(&sym->dim.arglist[argcnt],&arg,sizeof arg)!=0)
              error(25);    /* function definition does not match prototype */
          } /* if */
          argcnt++;
          ident=_variable;
          tag=0;
          break;
        case __ellips:
          if (ident!=_variable || tag!=0)
            error(10);  /* illegal function or declaration */
          if ((sym->usage & _prototyped)==0) {
            /* redimension the argument list, add the entry _varargs */
            sym->dim.arglist=realloc(sym->dim.arglist,(argcnt+2)*sizeof(arginfo));
            sym->dim.arglist[argcnt].ident=_varargs;
            sym->dim.arglist[argcnt+1].ident=0; /* keep the list terminated */
          } /* if */
          break;
        default:
          error(10);    /* illegal function or declaration */
      } /* endswitch */
    } while (tok=='&' || tok==__label
             || tok!=__ellips && matchtoken(',')); /* more? */
    /* if the next token is not ",", it should be ")" */
    needtoken(')');
  } /* endif */
  sym->usage|=_prototyped;
  return argcnt;
}

/*  doarg       - declare one argument type
 *
 *  this routine is called from "declargs()" and adds an entry in the local
 *  symbol table for one argument.
 */
static void doarg(char *name,int ident,int offset,int tag,arginfo *arg)
{
  symbol *argsym;
  cell size;
  int idxtag;

  arg->hasdefault=_no;  /* preset (most common case) */
  arg->defvalue=0;      /* clear */
  arg->defsize=0;
  if (matchtoken('[')){
    if (ident==_reference)
      error(10);        /* illegal declaration ("&name[]" is unsupported) */
    size=needsub(&idxtag);      /* may be zero here, it is a pointer anyway */
    ident=_refarray;            /* "reference to array" (is a pointer) */
    if (matchtoken('=')) {
      lexpush();                /* initials() needs it again */
      assert(litidx==0);        /* at the start of a function, this is reset */
      initials(ident,tag,&size);
      assert(size>=litidx);
      /* allocate memory to hold the initial values */
      arg->defvalue=(long)malloc(litidx*sizeof(cell));
      if (arg->defvalue!=0) {
        memcpy((cell *)arg->defvalue,litq,litidx*sizeof(cell));
        arg->hasdefault=_yes;   /* argument has default value */
        arg->defsize=litidx;
      } /* if */
      litidx=0;                 /* reset */
    } /* if */
  } else {
    if (matchtoken('=')) {
      assert(ident==_variable || ident==_reference);
      arg->hasdefault=_yes;     /* argument has a default value */
      constexpr(&arg->defvalue,NULL);
    } /* if */
  } /* if */
  arg->ident=(char)ident;
  arg->tag=tag;
  argsym=findloc(name);
  if (argsym){
    error(21,name);             /* symbol already defined */
  } else {
    /* add details of type and address */
    argsym=addsym(name,offset,ident,_local,tag,_define);
    if (ident==_refarray) {
      assert(size>=0);
      argsym->dim.array=size;
      argsym->x.idxtag=idxtag;
    } /* if */
    argsym->compound=0;
    if (ident==_reference)
      argsym->usage|=_read;     /* because references are passed back */
  } /* if */
}

/*  testsymbols - test for unused local variables
 *
 *  "Exported" functions are excluded from the check, since these
 *  may be exported to other object modules.
 *  Labels are excluded from the check if the argument 'testlabs'
 *  is 0. Thus, labels are not tested until the end of the function.
 *  Constants are also excluded.
 *
 *  When the nesting level drops below "level", the check stops.
 *
 *  The function returns whether there is an "entry" point for the file.
 *  This flag will only be 1 when browsing the global symbol table.
 */
static int testsymbols(symbol *root,int level,int testlabs)
{
  int entry=_no;

  symbol *sym=root->next;
  while (sym!=NULL && sym->compound>=level) {
    switch (sym->ident) {
    case _label:
      if (testlabs) {
        if ((sym->usage & _define)==0)
          error(19,sym->name);            /* not a label: ... */
        else if ((sym->usage & _refer)==0)
          error(203,sym->name);           /* symbol isn't used: ... */
      } /* if */
      break;
    case _functn:
      if ((sym->usage & (_define | _refer | _external))==_define)
        error(203,sym->name);   /* symbol isn't used ... (and not external) */
      else if ((sym->usage & (_define | _refer))==_refer)
        error(4,sym->name);     /* function not defined */
      if ((sym->usage & _exported)!=0 || strcmp(sym->name,"main")==0)
        entry=_yes;             /* there is an entry point */
      break;
    case _constexpr:
      /* ??? should check local constants, but not globals */
      break;
    default:
      /* a variable */
      if ((sym->usage & (_written | _read))==0)
        error(203,sym->name);   /* symbol isn't used ... */
      else if ((sym->usage & _read)==0)
        error(204,sym->name);   /* value assigned to symbol is never used */
    } /* if */
    sym=sym->next;
  } /* while */

  return entry;
}

static constval *add_constval(constval *table,char *name,cell val)
{
  constval *eq,*prev;

  /* find the end of the constant table */
  for (prev=table, eq=table->next; eq!=NULL; prev=eq, eq=eq->next)
    /* nothing */;
  if ((eq=malloc(sizeof(constval)))==NULL)
    error(103);       /* insufficient memory (fatal error) */
  memset(eq,0,sizeof(constval));
  strcpy(eq->name,name);
  eq->value=val;
  prev->next=eq;
  return eq;
}

#if 0
static constval *find_constval(constval *table,char *name)
{
  constval *ptr = table->next;

  while (ptr!=NULL) {
    if (strcmp(name,ptr->name)==0)
      return ptr;
    ptr=ptr->next;
  } /* while */
  return NULL;
}

static int delete_constval(constval *table,char *name)
{
  constval *prev = table;
  constval *cur = prev->next;

  while (cur!=NULL) {
    if (strcmp(name,cur->name)==0) {
      prev->next=cur->next;
      free(cur);
      return _yes;
    } /* if */
    prev=cur;
    cur=cur->next;
  } /* while */
  return _no;
}
#endif

static void delete_consttable(constval *table)
{
  constval *eq=table->next, *next;

  while (eq!=NULL) {
    next=eq->next;
    free(eq);
    eq=next;
  } /* while */
}

/*  add_constant
 *
 *  Adds a symbol to the #define symbol table.
 */
void add_constant(char *name,cell val,int vclass,int tag)
{
  symbol *sym;

  /* Test wether a global or local symbol with the same name exists. Since
   * constants are stored in the symbols table, this also finds previously
   * defind constants. */
  sym=findglb(name);
  if (!sym)
    sym=findloc(name);
  if (sym) {
    /* silently ignore redefinitions of constants with the same value */
    if (sym->ident==_constexpr) {
      if (sym->addr!=val)
        error(201,name);/* redefinition of constant (different value) */
    } else {
      error(21,name);   /* symbol already defined */
    } /* if */
    return;
  } /* if */

  /* constant doesn't exist yet, an entry must be created */
  addsym(name,val,_constexpr,vclass,tag,_define);
}

/*  statement           - The Statement Parser
 *
 *  This routine is called whenever the parser needs to know what statement
 *  it encounters (i.e. whenever program syntax requires a statement).
 *
 *  Global references: declared, ncmp
 */
static void statement(void)
{
  int tok;
  cell val;
  char *st;

  if (!freading)
    return;

  tok=lex(&val,&st);
  if (tok!='{')
    setline(fline,fnumber);
  switch (tok) {
  case 0:
    /* nothing */
    break;
  case __new:
    declloc();
    lastst=__new;
    break;
  case '{':
    if (!matchtoken('}'))       /* {} is the empty statement */
      compound();
    /* "last statement" does not change */
    break;
  case ';':
    error(36);                  /* empty statement */
    break;
  case __if:
    doif();
    lastst=__if;
    break;
  case __while:
    dowhile();
    lastst=__while;
    break;
  case __do:
    dodo();
    lastst=__do;
    break;
  case __for:
    dofor();
    lastst=__for;
    break;
  case __switch:
    doswitch();
    lastst=__switch;
    break;
  case __case:
    error(14);     /* not in switch */
    break;
  case __default:
    error(14);     /* not in switch */
    break;
  case __goto:
    dogoto();
    lastst=__goto;
    break;
  case __label:
    dolabel();
    lastst=__label;
    break;
  case __return:
    doreturn();
    lastst=__return;
    break;
  case __break:
    dobreak();
    lastst=__break;
    break;
  case __continue:
    docont();
    lastst=__continue;
    break;
  case __exit:
    doexit();
    break;
  case __assert:
    doassert();
    break;
  case __const:
    decl_const(_local);
    break;
  case __enum:
    decl_enum(_local);
    break;
  default:          /* non-empty expression */
    lexpush();      /* analyze token later */
    doexpr(_yes,NULL);
    needtoken(';');
    lastst=__expr;
  } /* switch */
  return;
}

static void compound(void)
{
  cell save_decl;

  save_decl=declared;
  ncmp+=1;              /* increase compound statement level */
  while (matchtoken('}')==0){  /* repeat until compound statement is closed */
    if (!freading){
      needtoken('}');   /* gives error: "expected token }" */
      break;
    } else {
      statement();      /* do a statement */
    } /* if */
  } /* while */
  if ((lastst!=__return) && (lastst!=__goto))
    modstk((int)(declared-save_decl)*sizeof(cell));  /* delete local variable space */
  testsymbols(&loctab,ncmp,_no);        /* look for unused block locals */
  declared=save_decl;
  delete_symbols(&loctab,ncmp,_no);     /* erase local symbols, but retain
                                         * block local labels (within the
                                         * function) */
  ncmp-=1;              /* decrease compound statement level */
}

/*  doexpr
 *
 *  Global references: stgidx   (referred to only)
 */
static void doexpr(int comma,int *tag)
{
  int constant,index;
  cell val;

  stgset(_yes);    /* start stage-buffering */
  do {
    index=stgidx;
    expression(&constant,&val,tag);
    endexpr();
    stgout(index);
  } while (comma && matchtoken(',')); /* more? */
  stgset(_no);     /* stop stage-buffering */
}

/*  constexpr
 *
 *  Global references: stgidx   (referred to only)
 */
int constexpr(cell *val,int *tag)
{
  int constant,index;
  cell cidx;

  stgset(_yes);         /* start stage-buffering */
  stgget(&index,&cidx); /* mark position in code generator */
  expression(&constant,val,tag);
  stgdel(index,cidx);   /* scratch generated code */
  stgset(_no);          /* stop stage-buffering */
  if (constant==0)
    error(8);           /* must be constant expression */
  return constant;
}

/*  test
 *
 *  In the case a "simple assignment" operator ("=") is used within a test,
 *  the warning "possibly unintended assignment" is displayed. This routine
 *  sets the global variable "intest" to true, it is restored upon termination.
 *  In the case the assignment was intended, use parantheses around the
 *  expression to avoid the warning; primary() sets "intest" to 0.
 *
 *  Global references: stgidx   (referred to only)
 *                     intest   (altered, but restored upon termination)
 */
static void test(int label,int parens,int invert)
{
  int index,tok;
  cell cidx;
  value lval;

  stgset(_yes);         /* start staging */
  pushstk((stkitem)intest);
  intest=1;
  if (parens)
    needtoken('(');
  do {
    stgget(&index,&cidx);       /* mark position (of last expression) in
                                 * code generator */
    if (hier14(&lval))
      rvalue(&lval);
    tok=matchtoken(',');
    if (tok)            /* if there is another expression... */
      stgout(index);    /* ...write first one */
  } while (tok); /* enddo */
  if (parens)
    needtoken(')');
  if (lval.ident==_constexpr) { /* constant expression */
    stgdel(index,cidx);
    if (lval.constval) {        /* code always executed */
      error(206);               /* redundant test: always non-zero */
      return;
    } /* if */
    error(205);                 /* redundant code: never executed */
    jumplabel(label);
    return;
  } /* if */
  if (invert)
    jmp_ne0(label);             /* jump to label if true (different from 0) */
  else
    jmp_eq0(label);             /* jump to label if false (equal to 0) */
  intest=(int)(long)popstk();   /* double typecast to avoid warning with Microsoft C */
  stgout(index);
  stgset(_no);          /* stop staging */
}

static void doif(void)
{
  int flab1,flab2;

  flab1=getlabel();     /* get label number for false branch */
  test(flab1,_yes,_no); /* get expression and branch to flab1 if false */
  statement();          /* if true, do a statement */
  if (matchtoken(__else)==0){  /* if...else ? */
    setlabel(flab1);    /* no, simple if..., print false label */
  } else {
    flab2=getlabel();
    if ((lastst!=__return) && (lastst!=__goto))
      jumplabel(flab2);
    setlabel(flab1);    /* print false label */
    statement();        /* do "else" clause */
    setlabel(flab2);    /* print true label */
  } /* endif */
}

static void dowhile(void)
{
  int wq[4];              /* allocate local queue */

  addwhile(wq);           /* add entry to queue for "break" */
  setlabel(wq[_wqloop]);  /* loop label */
  test(wq[_wqexit],_yes,_no);   /* branch to wq[_wqexit] if false */
  statement();            /* if so, do a statement */
  jumplabel(wq[_wqloop]); /* and loop to "while" start */
  setlabel(wq[_wqexit]);  /* exit label */
  delwhile();             /* delete queue entry */
}

/*
 *  Note that "continue" will in this case not jump to the top of the loop, but
 *  to the end: just before the TRUE-or-FALSE testing code.
 */
static void dodo(void)
{
  int wq[4],top;

  addwhile(wq);           /* see "dowhile" for more info */
  top=getlabel();         /* make a label first */
  setlabel(top);          /* loop label */
  statement();
  needtoken(__while);
  setlabel(wq[_wqloop]);  /* "continue" always jumps to WQLOOP. */
  test(wq[_wqexit],_yes,_no);
  jumplabel(top);
  setlabel(wq[_wqexit]);
  delwhile();
  needtoken(';');
}

/*  dofor
 *
 *  Despite what some books say,     for (expr1; expr2; expr3)
 *                                     statement;
 *
 *  is NOT equivalent to:            expr1;
 *                                   while (expr2){
 *                                     statement;
 *                                     expr3;
 *                                   }
 *
 *  If in the compound statement of the "while" expression the statement
 *  "continue" appears, "expr3" is skipped; if this statement appears in
 *  the "for" statement, "expr3" is executed.
 */
static void dofor(void)
{
  int wq[4],flab1,flab2;
  cell save_decl;
  int save_ncmp;

  save_decl=declared;
  save_ncmp=ncmp;

  addwhile(wq);
  flab1=getlabel();
  flab2=getlabel();
  needtoken('(');
  if (matchtoken(';')==0) {
    /* new variable declarations are allowed here */
    if (matchtoken(__new)) {
      /* The variable in expr1 of the for loop is at a
       * 'compound statement' level of it own.
       */
      ncmp++;
      declloc();/* declare local variable */
    } else {
      doexpr(_yes,NULL);        /* expression 1 */
      needtoken(';');
    } /* if */
  } /* if */
  setlabel(flab1);
  if (matchtoken(';')==0) {
    test(wq[_wqexit],_no,_no);  /* expression 2 (jump to wq[_wqexit] if false) */
    needtoken(';');
  } /* if */
  jumplabel(flab2);             /* skip expression 3 for now */
  setlabel(wq[_wqloop]);        /* "continue" goes to this label: expr3 */
  if (matchtoken(')')==0) {
    doexpr(_yes,NULL);          /* expression 3 */
    needtoken(')');
  } /* if */
  jumplabel(flab1);
  setlabel(flab2);
  statement();
  jumplabel(wq[_wqloop]);
  setlabel(wq[_wqexit]);
  delwhile();

  /* Clean up the space and the symbol table for the local
   * variable in "expr1".
   */
  modstk((int)(declared-save_decl)*sizeof(cell));
  declared=save_decl;
  delete_symbols(&loctab,ncmp,_yes);
  ncmp=save_ncmp;       /* reset 'compound statement' nesting level */
}

/* The switch statement is incompatible with its C sibling: the cases
 * are not drop through and the syntax is much stricter. A list of
 * case tests is still allowed (that is, the tests are drop through as
 * long as no statement executes).
 */
static void doswitch(void)
{
  int lbl_exit,lbl_next;
  int tok,swdefault,first;
  cell val;
  char *str;
  constval caselist = { "", 0, NULL};   /* case list starts empty */
  constval *cse;

  needtoken('(');
  doexpr(_yes,NULL);            /* evaluate switch expression */
  needtoken(')');
  /* move the expression result to ALT (PRI is modified in the comparison) */
  stgset(_yes);    /* start stage-buffering */
  push1();
  pop2();
  stgout(0);
  stgset(_no);

  needtoken('{');
  lbl_exit=getlabel();          /* get label number for jumping out of switch */
  swdefault=_no;
  first=_yes;
  lbl_next=0;                   /* to avoid a compiler warning */
  do {
    if (!first)
      setlabel(lbl_next);
    lbl_next=getlabel();
    tok=lex(&val,&str);         /* read in (new) token */
    switch (tok) {
    case __case:
      if (swdefault!=_no)
        error(15);      /* "default" case must be last in switch statement */
      constexpr(&val,NULL);
      needtoken(':');
      /* check for duplicate case */
      for (cse=caselist.next; cse!=NULL && cse->value!=val; cse=cse->next)
        /* nothing */;
      if (cse!=NULL)
        error(40,val);          /* duplicate "case" label */
      else
        add_constval(&caselist,"",val);
      cmpcase(val,lbl_next);    /* generate code to compare the primary
                                 * register to the case value and jump to
                                 * lbl_next if no match */
      if (matchtoken(__case)) {
        endcase();      /* generate code to jump around the next case test */
        lexpush();      /* analyse the token again */
      } else {
        statement();
        jumplabel(lbl_exit);
      } /* if */
      first=_no;
      break;
    case __default:
      if (swdefault!=_no)
        error(16);      /* multiple defaults in switch */
      needtoken(':');
      first=_no;
      swdefault=_yes;
      statement();
      /* there is no need to jump to lbl_exit, because this *must*
       * be the last case in a switch */
      break;
    case '}':
      /* nothing */
      break;
    default:
      lexpush();        /* let needtoken() do the error handling */
      needtoken('}');
    } /* switch */
  } while (tok!='}');
  setlabel(lbl_next);
  setlabel(lbl_exit);
  delete_consttable(&caselist); /* clear list of case labels */
  lastst=__switch;
}

static void doassert(void)
{
  int flab1,index;
  cell cidx;
  value lval;

  if ((debug & _chkbounds)!=0) {
    flab1=getlabel();           /* get label number for "OK" branch */
    test(flab1,_no,_yes);       /* get expression and branch to flab1 if true */
    ffabort(_assertion);
    setlabel(flab1);
  } else {
    stgset(_yes);               /* start staging */
    stgget(&index,&cidx);       /* mark position in code generator */
    do {
      if (hier14(&lval))
        rvalue(&lval);
      stgdel(index,cidx);       /* just scrap the code */
    } while (matchtoken(',')); /* do */
    stgset(_no);                /* stop staging */
  } /* if */
  needtoken(';');
}

static void dogoto(void)
{
  char *st;
  cell val;
  symbol *sym;

  if (lex(&val,&st)==__symbol){
    sym=fetchlab(st);
    /* The stack state should not change between a goto and its label.
     * That is, no variables may appear or disappear between the branch
     * and its destination.
     */
    if (sym->x.declared!=(int)declared)
      error(3); /* local variables disallowed between goto and its label */
    jumplabel((int)sym->addr);
    sym->usage|=_refer; /* set "_refer" bit */
  } else {
    error(20,st);       /* illegal symbol name */
  } /* endif */
  needtoken(';');
}

static void dolabel(void)
{
  char *st;
  cell val;
  symbol *sym;

  tokeninfo(&val,&st);  /* retrieve label name again */
  sym=fetchlab(st);
  if (sym->x.declared!=(int)declared)   /* see dogoto() */
    error(3);   /* local variables disallowed between goto and its label */
  setlabel((int)sym->addr);
  sym->usage|=_define;  /* label is now defined */
}

/*  fetchlab
 *
 *  Finds a label from the (local) symbol table or adds one to it.
 *  Labels are local in scope.
 *
 *  Note: The "_usage" bit is set to zero. The routines that call "fetchlab()"
 *        must set this bit accordingly.
 */
static symbol *fetchlab(char *name)
{
  symbol *sym;

  sym=findloc(name);            /* labels are local in scope */
  if (sym){
    if (sym->ident!=_label)
      error(19,sym->name);       /* not a label: ... */
  } else {
    sym=addsym(name,getlabel(),_label,_local,0,0);
    sym->x.declared=(int)declared;
    sym->compound=ncmp;
  } /* if */
  return sym;
}

/*  doreturn
 *
 *  Global references: rettype  (altered)
 */
static void doreturn(void)
{
  int tag;
  if (matchtoken(';')==0){
    if ((rettype & _retnone)!=0)
      error(208);                       /* mix "return;" and "return value;" */
    doexpr(_yes,&tag);
    needtoken(';');
    rettype|=_retvalue;                 /* function returns a value */
    /* ??? check tagname with function tagname */
  } else {
    /* this return statement contains no expression */
    const1(0);
    if ((rettype & _retvalue)!=0)
      error(209);                       /* function should return a value */
    rettype|=_retnone;                  /* function does not return anything */
  } /* if */
  modstk((int)declared*sizeof(cell));   /* end of function, remove *all*
                                         * local variables */
  ffret();
}

static void dobreak(void)
{
  int *ptr;

  ptr=readwhile(wqptr);  /* readwhile() gives an error if not in loop */
  needtoken(';');
  if (ptr==NULL)
    return;
  modstk(((int)declared-ptr[_wqsp])*sizeof(cell));
  jumplabel(ptr[_wqexit]);
}

static void docont(void)
{
  int *ptr;

  ptr=readwhile(wqptr);  /* readwhile() gives an error if not in loop */
  needtoken(';');
  if (ptr==NULL)
    return;
  modstk(((int)declared-ptr[_wqsp])*sizeof(cell));
  jumplabel(ptr[_wqloop]);
}

static void doexit(void)
{
  if (matchtoken(';')==0){
    doexpr(_yes,NULL);
    needtoken(';');
  } /* if */
  ffabort(_exit);
}

